<?php
// admin/rooms.php
declare(strict_types=1);
require_once __DIR__ . '/inc/auth.php';
require_login();

/* ---------------- CSRF ---------------- */
if (!isset($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
function csrf_token(): string { return $_SESSION['csrf']; }
function csrf_ok(?string $t): bool { return $t && hash_equals($_SESSION['csrf'], $t); }

/* -------------- Helpers --------------- */
if (!function_exists('e')) {
  function e(string $s): string { return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }
}
function fmt_dt(?string $s): string {
  if (!$s) return '';
  $t = strtotime($s); if ($t<=0) return e($s);
  return date('Y-m-d H:i', $t);
}

/* -------------- Actions ---------------- */
$pdo = db();
$msg = null; $err = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $action = $_POST['action'] ?? '';
  $id     = (int)($_POST['id'] ?? 0);
  if (!csrf_ok($_POST['csrf'] ?? '')) {
    $err = 'رمز الأمان غير صالح.';
  } elseif ($id <= 0) {
    $err = 'مُعرّف غير صالح.';
  } else {
    try {
      if ($action === 'toggle_status') {
        $pdo->prepare("UPDATE `rooms` SET `status` = CASE WHEN `status`='open' THEN 'closed' ELSE 'open' END WHERE `id`=? LIMIT 1")->execute([$id]);
        $msg = 'تم تغيير حالة الغرفة.';
      } elseif ($action === 'clear_image') {
        $pdo->prepare("UPDATE `rooms` SET `image_url`=NULL WHERE `id`=? LIMIT 1")->execute([$id]);
        $msg = 'تم تفريغ صورة الغرفة.';
      } elseif ($action === 'clear_bg') {
        $pdo->prepare("UPDATE `rooms` SET `background_url`=NULL WHERE `id`=? LIMIT 1")->execute([$id]);
        $msg = 'تم تفريغ خلفية الغرفة.';
      } elseif ($action === 'delete_room') {
        $pdo->prepare("DELETE FROM `rooms` WHERE `id`=? LIMIT 1")->execute([$id]);
        $msg = 'تم حذف الغرفة نهائيًا.';
      }
    } catch (Throwable $e) {
      $err = 'حدث خطأ أثناء تنفيذ العملية.';
    }
  }
}

/* -------------- Search & Page ---------- */
$q       = trim($_GET['q'] ?? '');
$statusF = trim($_GET['status'] ?? ''); // '', open, closed
$page    = max(1, (int)($_GET['page'] ?? 1));
$perPage = 50;
$offset  = ($page - 1) * $perPage;

$where   = "1=1"; $params = [];
if ($q !== '') {
  $where .= " AND (r.name LIKE ? OR r.uid LIKE ? OR r.owner_uid LIKE ? OR r.agora_channel LIKE ?)";
  $kw = "%$q%"; array_push($params,$kw,$kw,$kw,$kw);
}
if ($statusF === 'open' || $statusF === 'closed') {
  $where .= " AND r.status = ?";
  $params[] = $statusF;
}

/* إجمالي النتائج */
$st = $pdo->prepare("SELECT COUNT(*) FROM `rooms` r WHERE $where");
$st->execute($params);
$total = (int)$st->fetchColumn();
$pages = max(1, (int)ceil($total / $perPage));

/* جلب الصفحة */
$sql = "SELECT r.id, r.uid, r.owner_uid, r.name, r.description, r.image_url, r.agora_channel,
               r.status, r.max_seats, r.created_at, r.background_url,
               u.username AS owner_name, u.email AS owner_email
        FROM `rooms` r
        LEFT JOIN `users` u ON u.uid = r.owner_uid
        WHERE $where
        ORDER BY r.created_at ASC, r.id ASC
        LIMIT $perPage OFFSET $offset";
$st = $pdo->prepare($sql); $st->execute($params);
$rows = $st->fetchAll();

/* -------------- Render ------------------ */
$PAGE_TITLE = "الغرف";
require __DIR__ . '/inc/header.php';
require __DIR__ . '/inc/sidebar.php';
?>

<!-- ====== CSS داخلي للصفحة ====== -->
<style>
.rooms-page .table{
  width:100%;
  border-collapse:separate;border-spacing:0;
  background:var(--panel); border:1px solid var(--border);
  border-radius:12px; overflow:hidden;
  table-layout:fixed;
}
.rooms-page .table td,.rooms-page .table th{
  padding:10px 12px; border-bottom:1px solid var(--border);
  vertical-align:middle; text-align:start;
}
.rooms-page .table thead th{
  background:#0b1220;color:var(--text);font-weight:700;
  position:sticky; top:0; z-index:3;
  box-shadow:inset 0 -1px 0 var(--border);
}
.rooms-page .table tbody tr:nth-child(even) td{background:rgba(255,255,255,.015)}
.rooms-page .table tbody tr:hover td{background:rgba(59,130,246,.08)}
.rooms-page .table-wrap{max-height:70vh; overflow:auto; position:relative}

.rooms-page .td-center{text-align:center}
.rooms-page .ellipsis{max-width:240px; display:inline-block; overflow:hidden; white-space:nowrap; text-overflow:ellipsis}
.rooms-page .ellipsis.small{max-width:160px}
.rooms-page .room-img{width:44px;height:44px;border-radius:12px;object-fit:cover;border:1px solid var(--border)}
.rooms-page .badge{display:inline-block;padding:2px 8px;border-radius:999px;border:1px solid var(--border);font-size:12px}
.rooms-page .badge.status-open{border-color:#22c55e}
.rooms-page .badge.status-closed{border-color:#ef4444}
.rooms-page .actions{display:flex;gap:8px;flex-wrap:nowrap;align-items:center}
.rooms-page .actions form{display:inline-block;margin:0}
.rooms-page .btn.sm{padding:6px 10px; font-size:14px}

.rooms-page .col-id{width:60px}.rooms-page .col-img{width:70px}.rooms-page .col-name{width:240px}
.rooms-page .col-owner{width:220px}.rooms-page .col-uid{width:200px}.rooms-page .col-agora{width:200px}
.rooms-page .col-status{width:110px}.rooms-page .col-seats{width:80px}.rooms-page .col-created{width:170px}
.rooms-page .col-actions{width:380px}
</style>

<div class="rooms-page">
  <div class="card" style="margin-bottom:16px;">
    <form method="get" class="form" style="display:flex; gap:12px; align-items:flex-end; flex-wrap:wrap;">
      <div class="input" style="min-width:260px; flex:1;">
        <label>ابحث بالاسم/UID/مالك الغرفة/قناة Agora</label>
        <input type="text" name="q" value="<?= e($q) ?>" placeholder="مثال: غرفة الدردشة أو 11X...">
      </div>

      <div class="input">
        <label>الحالة</label>
        <select name="status" style="padding:10px 12px; border-radius:10px; border:1px solid var(--border); background:#0b1220; color:var(--text);">
          <option value=""     <?= $statusF===''?'selected':'' ?>>الكل</option>
          <option value="open" <?= $statusF==='open'?'selected':'' ?>>مفتوحة</option>
          <option value="closed" <?= $statusF==='closed'?'selected':'' ?>>مغلقة</option>
        </select>
      </div>

      <button class="btn sm" type="submit">بحث</button>
      <?php if ($q!=='' || $statusF!==''): ?>
        <a class="btn sm secondary" href="rooms.php">مسح الفلاتر</a>
      <?php endif; ?>
      <div style="margin-inline-start:auto; opacity:.9;">إجمالي: <b><?= $total ?></b> نتيجة</div>
    </form>
    <?php if ($msg): ?><div class="alert" style="margin-top:12px; border-color:#22c55e;"><?= e($msg) ?></div><?php endif; ?>
    <?php if ($err): ?><div class="alert" style="margin-top:12px;"><?= e($err) ?></div><?php endif; ?>
  </div>

  <div class="card">
    <div class="table-wrap">
      <table class="table">
        <colgroup>
          <col class="col-id"><col class="col-img"><col class="col-name"><col class="col-owner">
          <col class="col-uid"><col class="col-agora"><col class="col-status">
          <col class="col-seats"><col class="col-created"><col class="col-actions">
        </colgroup>
        <thead>
          <tr>
            <th class="td-center">#</th>
            <th class="td-center">الصورة</th>
            <th>الاسم</th>
            <th>المالك</th>
            <th>UID</th>
            <th>قناة Agora</th>
            <th>الحالة</th>
            <th>المقاعد</th>
            <th>أُنشيء</th>
            <th>إجراءات</th>
          </tr>
        </thead>
        <tbody>
          <?php foreach ($rows as $r): ?>
            <tr>
              <td class="td-center"><?= (int)$r['id'] ?></td>
              <td class="td-center">
                <?php if (!empty($r['image_url'])): ?>
                  <img src="<?= e($r['image_url']) ?>" class="room-img" alt="">
                <?php else: ?>
                  <span class="badge">بدون</span>
                <?php endif; ?>
              </td>
              <td title="<?= e($r['name'] ?? '') ?>"><span class="ellipsis"><?= e($r['name'] ?? '') ?></span></td>
              <td>
                <?php if (!empty($r['owner_name'])): ?>
                  <span title="<?= e($r['owner_uid'] ?? '') ?>"><?= e($r['owner_name']) ?></span>
                <?php else: ?>
                  <span class="ellipsis small" title="<?= e($r['owner_uid'] ?? '') ?>"><?= e($r['owner_uid'] ?? '') ?></span>
                <?php endif; ?>
              </td>
              <td title="<?= e($r['uid'] ?? '') ?>"><span class="ellipsis small" style="font-family:monospace;"><?= e($r['uid'] ?? '') ?></span></td>
              <td title="<?= e($r['agora_channel'] ?? '') ?>"><span class="ellipsis small" style="font-family:monospace;"><?= e($r['agora_channel'] ?? '') ?></span></td>
              <td>
                <?php $stxt = ($r['status']==='open') ? 'مفتوحة' : 'مغلقة'; ?>
                <span class="badge <?= $r['status']==='open'?'status-open':'status-closed' ?>"><?= e($stxt) ?></span>
              </td>
              <td class="td-center"><?= (int)$r['max_seats'] ?></td>
              <td title="<?= e($r['created_at'] ?? '') ?>"><?= fmt_dt($r['created_at'] ?? '') ?></td>
              <td class="actions">
                <a class="btn sm secondary" href="rooms_edit.php?id=<?= (int)$r['id'] ?>">تعديل</a>

                <form method="post" onsubmit="return confirm('تغيير حالة الغرفة؟');">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                  <input type="hidden" name="action" value="toggle_status">
                  <button class="btn sm" type="submit"><?= $r['status']==='open'?'إغلاق':'فتح' ?></button>
                </form>

                <form method="post" onsubmit="return confirm('تفريغ صورة الغرفة؟');">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                  <input type="hidden" name="action" value="clear_image">
                  <button class="btn sm" type="submit">تفريغ الصورة</button>
                </form>

                <form method="post" onsubmit="return confirm('تفريغ الخلفية؟');">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                  <input type="hidden" name="action" value="clear_bg">
                  <button class="btn sm" type="submit">تفريغ الخلفية</button>
                </form>

                <form method="post" onsubmit="return confirm('حذف نهائي؟ لا يمكن التراجع.');">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                  <input type="hidden" name="action" value="delete_room">
                  <button class="btn sm" type="submit" style="background:var(--danger);">حذف</button>
                </form>
              </td>
            </tr>
          <?php endforeach; ?>
          <?php if (!$rows): ?>
            <tr><td colspan="10" style="text-align:center; opacity:.8;">لا توجد بيانات</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>

    <?php if ($pages > 1): ?>
      <div style="display:flex; gap:8px; margin-top:12px; flex-wrap:wrap;">
        <?php for($p=1;$p<=$pages;$p++):
          $url = 'rooms.php?page='.$p
               .($q!=='' ? '&q='.urlencode($q) : '')
               .($statusF!=='' ? '&status='.urlencode($statusF) : '');
          $is = $p===$page; ?>
          <a href="<?= e($url) ?>" class="btn sm <?= $is ? '' : 'secondary' ?>"<?= $is?' style="pointer-events:none; opacity:.8;"':''; ?>><?= $p ?></a>
        <?php endfor; ?>
      </div>
    <?php endif; ?>
  </div>
</div>

<?php require __DIR__ . '/inc/footer.php'; ?>
