<?php
// admin/items_edit.php — يسحب الأصناف من store_categories + رفع صورة (frame) أو فيديو (entry)

declare(strict_types=1);

$CFG_PATH = __DIR__ . '/config.php';
$INC_DIR  = __DIR__ . '/inc';
require_once $CFG_PATH;
require_once $INC_DIR . '/auth.php';
require_login();

/* ===== إعدادات الرفع ===== */
// الصور (الإطارات)
const MAX_IMAGE_BYTES   = 4 * 1024 * 1024; // 4MB
$FRAME_UPLOAD_BASE_URL = 'https://battle-party.com/voicechat/api/files/frame/';
$FRAME_UPLOAD_DIR      = rtrim($_SERVER['DOCUMENT_ROOT'], '/') . '/voicechat/api/files/frame/';

// الفيديو (الدخوليات)
const MAX_VIDEO_BYTES   = 120 * 1024 * 1024; // 120MB
$ENTRY_UPLOAD_BASE_URL  = 'https://battle-party.com/voicechat/api/files/entry/';
$ENTRY_UPLOAD_DIR       = rtrim($_SERVER['DOCUMENT_ROOT'], '/') . '/voicechat/api/files/entry/';

/* CSRF */
if (!isset($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
function csrf_token(): string { return $_SESSION['csrf']; }
function csrf_ok(?string $t): bool { return $t && hash_equals($_SESSION['csrf'], $t); }

/* Helpers */
if (!function_exists('e')) {
  function e(string $s): string { return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }
}
function to_en_digits(string $s): string {
  $ar=['٠','١','٢','٣','٤','٥','٦','٧','٨','٩']; $fa=['۰','۱','۲','۳','۴','۵','۶','۷','۸','۹']; $en=['0','1','2','3','4','5','6','7','8','9'];
  return str_replace($fa,$en,str_replace($ar,$en,$s));
}
function to_int($v): int { $v=to_en_digits(trim((string)$v)); if($v==='') return 0; $v=preg_replace('/[^0-9\-]/','',$v); return (int)$v; }
function to_float($v): float { $v=to_en_digits(trim((string)$v)); if($v==='') return 0.0; $v=preg_replace('/[^0-9\.\-]/','',$v); return (float)$v; }
/* بناء slug ASCII فقط مع بديل تلقائي */
function make_slug(string $s, string $fallback=''): string {
  $s = to_en_digits(trim($s));
  $s = strtolower($s);
  $s = preg_replace('/\s+/', '-', $s);
  $s = preg_replace('/[^a-z0-9_-]/', '', $s);
  $s = trim($s, '-_');
  if ($s==='') {
    $fallback = strtolower($fallback);
    $fallback = preg_replace('/\s+/', '-', $fallback);
    $fallback = preg_replace('/[^a-z0-9_-]/', '', $fallback);
    $fallback = trim($fallback, '-_');
    $s = $fallback !== '' ? $fallback : ('item-'.time());
  }
  return $s;
}
/* فحص وجود عمود */
function col_exists(PDO $pdo, string $table, string $col): bool {
  $st=$pdo->prepare("SHOW COLUMNS FROM `$table` LIKE ?"); $st->execute([$col]);
  return (bool)$st->fetchColumn();
}

/* رفع الصورة وإرجاع [url|null, error|null] */
function upload_image_file(array $file, string $slug, string $UPLOAD_DIR, string $UPLOAD_BASE_URL): array {
  if (!isset($file['tmp_name']) || $file['error'] === UPLOAD_ERR_NO_FILE) return [null, null]; // لم يرفع شيء
  if ($file['error'] !== UPLOAD_ERR_OK) return [null, 'فشل رفع الملف (رمز: '.$file['error'].').'];

  if ($file['size'] > MAX_IMAGE_BYTES) return [null, 'حجم الصورة يتجاوز المسموح ('.(MAX_IMAGE_BYTES/1024/1024).'MB).'];

  // تحقق نوع الملف
  $finfo = new finfo(FILEINFO_MIME_TYPE);
  $mime  = $finfo->file($file['tmp_name']) ?: '';
  $allowed = [
    'image/jpeg' => 'jpg',
    'image/png'  => 'png',
    'image/webp' => 'webp',
    'image/gif'  => 'gif',
  ];
  if (!isset($allowed[$mime])) return [null, 'نوع الصورة غير مسموح. الأنواع المسموحة: JPG/PNG/WEBP/GIF.'];
  $ext = $allowed[$mime];

  // تأمين اسم الملف
  $slug = $slug !== '' ? $slug : 'item';
  $slug = preg_replace('/[^a-z0-9_-]/', '', strtolower($slug));
  $filename = $slug . '-' . date('Ymd-His') . '-' . bin2hex(random_bytes(3)) . '.' . $ext;

  // تأكد من وجود المسار
  if (!is_dir($UPLOAD_DIR)) {
    if (!@mkdir($UPLOAD_DIR, 0755, true)) {
      return [null, 'تعذر إنشاء مجلد الرفع على السيرفر.'];
    }
  }

  // انقل الملف
  $destPath = rtrim($UPLOAD_DIR,'/').'/'.$filename;
  if (!@move_uploaded_file($file['tmp_name'], $destPath)) {
    return [null, 'تعذر حفظ الصورة على السيرفر.'];
  }

  // ارجع الرابط العام
  $url = rtrim($UPLOAD_BASE_URL,'/').'/'.$filename;
  return [$url, null];
}

/* رفع الفيديو وإرجاع [url|null, error|null] */
function upload_video_file(array $file, string $slug, string $UPLOAD_DIR, string $UPLOAD_BASE_URL): array {
  if (!isset($file['tmp_name']) || $file['error'] === UPLOAD_ERR_NO_FILE) return [null, null]; // لم يرفع شيء
  if ($file['error'] !== UPLOAD_ERR_OK) return [null, 'فشل رفع الملف (رمز: '.$file['error'].').'];

  if ($file['size'] > MAX_VIDEO_BYTES) return [null, 'حجم الفيديو يتجاوز المسموح ('.(int)(MAX_VIDEO_BYTES/1024/1024).'MB).'];

  $finfo = new finfo(FILEINFO_MIME_TYPE);
  $mime  = $finfo->file($file['tmp_name']) ?: '';
  $allowed = [
    'video/mp4'        => 'mp4',
    'video/webm'       => 'webm',
    'video/ogg'        => 'ogv',
    'video/quicktime'  => 'mov',
    'video/x-matroska' => 'mkv',
    'video/mpeg'       => 'mpg',
  ];
  if (!isset($allowed[$mime])) return [null, 'نوع الفيديو غير مسموح. المسموح: MP4/WEBM/MOV/MKV/OGG/MPEG.'];
  $ext = $allowed[$mime];

  $slug = $slug !== '' ? $slug : 'entry';
  $slug = preg_replace('/[^a-z0-9_-]/', '', strtolower($slug));
  $filename = $slug . '-' . date('Ymd-His') . '-' . bin2hex(random_bytes(3)) . '.' . $ext;

  if (!is_dir($UPLOAD_DIR)) {
    if (!@mkdir($UPLOAD_DIR, 0755, true)) {
      return [null, 'تعذر إنشاء مجلد رفع الفيديو.'];
    }
  }

  $destPath = rtrim($UPLOAD_DIR,'/').'/'.$filename;
  if (!@move_uploaded_file($file['tmp_name'], $destPath)) {
    return [null, 'تعذر حفظ الفيديو على السيرفر.'];
  }

  $url = rtrim($UPLOAD_BASE_URL,'/').'/'.$filename;
  return [$url, null];
}

$pdo = db();
$msg = null; $err = null; $dbg = isset($_GET['dbg']);

/* ===== تحميل أصناف المتجر من store_categories ===== */
$cats = []; $catBySlug = []; $catById = [];
try {
  $st = $pdo->query("SELECT id, slug, name, is_active, sort_order
                     FROM store_categories
                     ORDER BY sort_order ASC, id DESC");
  $cats = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
  foreach ($cats as $c) { $catBySlug[(string)$c['slug']]=$c; $catById[(int)$c['id']]=$c; }
} catch (Throwable $e) {
  $err = 'تعذّر جلب الأصناف.' . ($dbg ? ' ('.$e->getMessage().')' : '');
}

/* هل يوجد عمود category_id في store_items؟ */
$hasCategoryId = col_exists($pdo, 'store_items', 'category_id');

$id = (int)($_GET['id'] ?? 0);
$editing = $id > 0;

/* حقول العنصر */
$fields = [
  'category'    => '',
  'category_id' => '0',
  'slug'        => '',
  'name'        => '',
  'image_url'   => '',
  'price'       => '0',
  'days'        => '0',
  'rating'      => '0',
  'is_active'   => '1',
  'sort_order'  => '0',
  'meta_json'   => '',
];

/* تحميل بيانات العنصر */
$existingVideoUrl = '';
if ($editing) {
  $st = $pdo->prepare("SELECT * FROM store_items WHERE id=? LIMIT 1");
  $st->execute([$id]);
  $row = $st->fetch(PDO::FETCH_ASSOC);
  if ($row) {
    foreach ($fields as $k => $_) if (array_key_exists($k, $row)) $fields[$k] = (string)$row[$k];
    if ($fields['category']==='' && $hasCategoryId && (int)$fields['category_id']>0 && isset($catById[(int)$fields['category_id']])) {
      $fields['category'] = (string)$catById[(int)$fields['category_id']]['slug'];
    }
    if (!empty($fields['meta_json'])) {
      $m = json_decode($fields['meta_json'], true);
      if (is_array($m)) {
        $existingVideoUrl = (string)($m['video_url'] ?? $m['video'] ?? '');
      }
    }
  } else { $editing=false; $id=0; }
}

/* ===== حفظ ===== */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  if (!csrf_ok($_POST['csrf'] ?? '')) {
    $err = 'رمز الأمان غير صالح.';
  } else {
    foreach ($fields as $k=>$_) $fields[$k] = trim((string)($_POST[$k] ?? $fields[$k]));
    $fields['is_active'] = isset($_POST['is_active']) ? '1' : '0';

    // اختيار الصنف من القائمة
    $selectedSlug = trim((string)($_POST['category_slug'] ?? $fields['category']));
    $fields['category'] = $selectedSlug;
    $selectedId = isset($catBySlug[$selectedSlug]) ? (int)$catBySlug[$selectedSlug]['id'] : 0;
    if ($hasCategoryId) $fields['category_id'] = (string)$selectedId;

    // تنظيف القيم العددية
    $fields['price']      = (string)to_float($fields['price']);
    $fields['days']       = (string)to_int($fields['days']);
    $fields['rating']     = (string)to_int($fields['rating']);
    $fields['sort_order'] = (string)to_int($fields['sort_order']);

    // slug آمن
    $fields['slug'] = make_slug($fields['slug'], $fields['name']);

    // تحقق الصنف
    if (!$err && $cats && !isset($catBySlug[$fields['category']])) $err = 'الرجاء اختيار صنف صحيح من القائمة.';

    // نقرأ/نُحضّر الميتا
    $meta = [];
    if ($fields['meta_json']!=='') {
      $tmp = json_decode($fields['meta_json'], true);
      if (json_last_error()!==JSON_ERROR_NONE) {
        $err = 'حقل meta_json ليس JSON صالحًا: '.json_last_error_msg();
      } elseif (is_array($tmp)) {
        $meta = $tmp;
      }
    }

    // حذف الصورة؟
    $removeImage = isset($_POST['remove_image']) && $_POST['remove_image']=='1';

    // رفع الصورة (تُرفع دومًا إلى مجلد frame كصورة عرض/مصغّرة)
    if (!$err && isset($_FILES['image_file'])) {
      [$urlImg, $upErrImg] = upload_image_file($_FILES['image_file'], $fields['slug'], $FRAME_UPLOAD_DIR, $FRAME_UPLOAD_BASE_URL);
      if ($upErrImg) $err = $upErrImg;
      elseif ($urlImg) $fields['image_url'] = $urlImg;
    }
    if ($removeImage) $fields['image_url'] = '';

    // إن كان الصنف entry نسمح برفع فيديو إلى مجلد entry ونحدّث الميتا تلقائيًا
    $removeVideo = isset($_POST['remove_video']) && $_POST['remove_video']=='1';
    if (!$err && $fields['category'] === 'entry' && isset($_FILES['video_file'])) {
      [$urlVid, $upErrVid] = upload_video_file($_FILES['video_file'], $fields['slug'], $ENTRY_UPLOAD_DIR, $ENTRY_UPLOAD_BASE_URL);
      if ($upErrVid) $err = $upErrVid;
      elseif ($urlVid) {
        $meta['media_type'] = 'video';
        $meta['video_url']  = $urlVid;
      }
    }
    if ($removeVideo) {
      unset($meta['video_url'], $meta['video'], $meta['media_type']);
    }

    // إن كانت لدينا صورة عرض مع فيديو، نسجلها كـ thumb_url
    if (!empty($fields['image_url'])) {
      $meta['thumb_url'] = $fields['image_url'];
    }

    // حفظ الميتا النهائي
    $fields['meta_json'] = !empty($meta) ? json_encode($meta, JSON_UNESCAPED_UNICODE) : '';

    if (!$err) {
      try {
        if ($editing) {
          $cols = "category=?, slug=?, name=?, image_url=?, price=?, days=?, rating=?, is_active=?, sort_order=?, meta_json=?";
          $args = [
            $fields['category'], $fields['slug'], $fields['name'], $fields['image_url'],
            (float)$fields['price'], (int)$fields['days'], (int)$fields['rating'], (int)$fields['is_active'],
            (int)$fields['sort_order'], $fields['meta_json']
          ];
          if ($hasCategoryId) { $cols = "category_id=?, ".$cols; array_unshift($args, (int)$fields['category_id']); }
          $args[] = $id;
          $sql = "UPDATE store_items SET $cols WHERE id=? LIMIT 1";
          $pdo->prepare($sql)->execute($args);
          $msg = 'تم تحديث العنصر.';
        } else {
          $cols = "category,slug,name,image_url,price,days,rating,is_active,sort_order,meta_json,created_at";
          $qs   = "?,?,?,?,?,?,?,?,?,?,NOW()";
          $args = [
            $fields['category'], $fields['slug'], $fields['name'], $fields['image_url'],
            (float)$fields['price'], (int)$fields['days'], (int)$fields['rating'], (int)$fields['is_active'],
            (int)$fields['sort_order'], $fields['meta_json']
          ];
          if ($hasCategoryId) { $cols = "category_id,".$cols; $qs = "?,".$qs; array_unshift($args, (int)$fields['category_id']); }
          $sql = "INSERT INTO store_items ($cols) VALUES ($qs)";
          $pdo->prepare($sql)->execute($args);
          $newId = (int)$pdo->lastInsertId();
          header('Location: items_edit.php?id='.$newId.'&created=1'); exit;
        }
      } catch (Throwable $e) {
        $em = $e->getMessage(); $msgText='تعذّر الحفظ.';
        if (stripos($em,'Duplicate entry')!==false && stripos($em,"for key 'slug'")!==false) $msgText='تعذّر الحفظ: قيمة SLUG مستخدمة مسبقًا.';
        $err = $msgText . ($dbg ? ' ('.$em.')' : '');
      }
    }
  }
}

/* عنوان الصفحة والرأس */
$PAGE_TITLE = $editing ? "تعديل عنصر #$id" : "إضافة عنصر";
require $INC_DIR . '/header.php';
require $INC_DIR . '/sidebar.php';
?>
<style>
.form .row{display:flex;gap:12px;flex-wrap:wrap}
.form .row .input{flex:1;min-width:240px}
.preview{display:flex;align-items:center;gap:12px}
.preview img,.preview video{width:120px;height:80px;object-fit:cover;border-radius:12px;border:1px solid var(--border)}
.select-note{font-size:12px;opacity:.8;margin-top:4px}
.opt-off{opacity:.7}
</style>

<div class="card">
  <?php if (isset($_GET['created'])): ?><div class="alert" style="border-color:#22c55e;">تمت إضافة العنصر. يمكنك متابعة التعديل هنا.</div><?php endif; ?>
  <?php if ($msg): ?><div class="alert" style="border-color:#22c55e;"><?= e($msg) ?></div><?php endif; ?>
  <?php if ($err): ?><div class="alert"><?= e($err) ?></div><?php endif; ?>

  <?php if (!$cats): ?>
    <div class="alert">
      لا توجد أصناف في المتجر بعد. يرجى إضافة صنف أولًا من صفحة
      <a href="store_categories_edit.php" style="text-decoration:underline">إضافة صنف</a>.
    </div>
  <?php endif; ?>

  <form class="form" method="post" enctype="multipart/form-data">
    <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">

    <div class="row">
      <div class="input">
        <label>الاسم</label>
        <input type="text" name="name" required value="<?= e($fields['name']) ?>">
      </div>

      <div class="input">
        <label>SLUG (فريد)</label>
        <input type="text" name="slug" required value="<?= e($fields['slug']) ?>" placeholder="english-letters-numbers">
      </div>

      <div class="input">
        <label>الصنف</label>
        <select name="category_slug" required>
          <option value="" <?= $fields['category']===''?'selected':''; ?> disabled>اختر الصنف</option>
          <?php foreach ($cats as $c): ?>
            <?php
              $slug = (string)$c['slug'];
              $sel  = ($fields['category'] === $slug) ? ' selected' : '';
              $off  = ((int)$c['is_active'] === 1) ? '' : ' class="opt-off"';
            ?>
            <option value="<?= e($slug) ?>"<?= $sel ?><?= $off ?>>
              <?= e($c['name']) ?> (<?= e($slug) ?>)<?= ((int)$c['is_active']===1 ? '' : ' — معطّل') ?>
            </option>
          <?php endforeach; ?>
        </select>
        <div class="select-note">* تُخزَّن قيمة الصنف كـ <b>slug</b><?= $hasCategoryId ? ' وكذلك <b>category_id</b> إن وُجد' : '' ?>.</div>
      </div>

      <label class="input" style="align-self:flex-end;display:flex;gap:8px">
        <input type="checkbox" name="is_active" value="1"<?= $fields['is_active']==='1'?' checked':''; ?>> مفعل
      </label>
    </div>

    <div class="row">
      <div class="input">
        <label>الصورة (اختيارية – تستخدم كصورة عرض/مصغّرة)</label>
        <input type="file" name="image_file" accept="image/*">
        <div class="select-note">يتم رفع الصورة إلى: <code>/voicechat/api/files/frame/</code></div>
        <div class="preview" style="margin-top:6px;">
          <?php if ($fields['image_url']!==''): ?>
            <img src="<?= e($fields['image_url']) ?>" alt="">
            <label style="display:flex;align-items:center;gap:6px">
              <input type="checkbox" name="remove_image" value="1"> إزالة الصورة الحالية
            </label>
          <?php endif; ?>
        </div>
      </div>

      <div class="input">
        <label>السعر</label>
        <input type="number" name="price" step="0.01" min="0" value="<?= e($fields['price']) ?>">
      </div>
      <div class="input">
        <label>عدد الأيام (0 = دائم)</label>
        <input type="number" name="days" step="1" min="0" value="<?= e($fields['days']) ?>">
      </div>
      <div class="input">
        <label>التقييم/الظهور (اختياري)</label>
        <input type="number" name="rating" step="1" min="0" value="<?= e($fields['rating']) ?>">
      </div>
      <div class="input">
        <label>الترتيب</label>
        <input type="number" name="sort_order" step="1" value="<?= e($fields['sort_order']) ?>">
      </div>
    </div>

    <!-- حقل رفع الفيديو يظهر دائمًا، لكنه يُستخدم منطقيًا مع صنف entry -->
    <div class="row">
      <div class="input">
        <label>الفيديو (لصنف الدخوليات entry)</label>
        <input type="file" name="video_file" accept="video/*">
        <div class="select-note">يتم رفع الفيديو إلى: <code>/voicechat/api/files/entry/</code></div>
        <div class="preview" style="margin-top:6px;">
          <?php if (!empty($existingVideoUrl)): ?>
            <video src="<?= e($existingVideoUrl) ?>" controls></video>
            <label style="display:flex;align-items:center;gap:6px">
              <input type="checkbox" name="remove_video" value="1"> إزالة الفيديو الحالي
            </label>
          <?php endif; ?>
        </div>
      </div>
    </div>

    <div class="input">
      <label>Meta JSON (اختياري)</label>
      <textarea name="meta_json" rows="6" placeholder='{"color":"#FFD700","size":"large"}'><?= e($fields['meta_json']) ?></textarea>
      <div class="select-note">* عند رفع فيديو سيتم ضبط <code>media_type</code> و <code>video_url</code> تلقائيًا هنا، ويمكن أيضًا حفظ <code>thumb_url</code> من الصورة أعلاه.</div>
    </div>

    <?php if ($hasCategoryId): ?>
      <input type="hidden" name="category_id" value="<?= e($fields['category_id']) ?>">
    <?php endif; ?>

    <div style="display:flex;gap:8px;margin-top:12px">
      <button class="btn" type="submit"><?= $editing ? 'حفظ التعديلات' : 'إضافة' ?></button>
      <a class="btn secondary" href="items.php">رجوع إلى القائمة</a>
    </div>
  </form>
</div>

<?php require $INC_DIR . '/footer.php'; ?>
