<?php
// wesalchat/admin/store/items.php
declare(strict_types=1);

/* ===== bootstrap & auth ===== */
$CFG_PATH = __DIR__ . '/config.php';
$INC_DIR  = __DIR__ . '/inc';
require_once $CFG_PATH;
require_once $INC_DIR . '/auth.php';
require_login();

/* ---------------- CSRF ---------------- */
if (!isset($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
function csrf_token(): string { return $_SESSION['csrf']; }
function csrf_ok(?string $t): bool { return $t && hash_equals($_SESSION['csrf'], $t); }

/* -------------- Helpers --------------- */
if (!function_exists('e')) {
  function e(string $s): string { return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }
}
function fmt_dt(?string $s): string {
  if (!$s) return '';
  $t = strtotime($s); if ($t<=0) return e($s);
  return date('Y-m-d H:i', $t);
}
function fmt_money($v): string { return number_format((float)$v, 2); }
function categories(): array {
  // حدّث هذه القائمة حسب المتاح عندك
  return [
    'frame'   => 'إطار',
    'badge'   => 'شارة',
    'effect'  => 'تأثير',
    'vip'     => 'VIP',
    'bg'      => 'خلفية',
    'cover'   => 'غلاف',
    'boost'   => 'تعزيز',
    'username'=> 'اسم ملوّن',
    'other'   => 'أخرى',
  ];
}

/* ----------------- DB ------------------ */
$pdo = db();
$msg = null; $err = null;

/* --------- POST actions (toggle/delete) -------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  if (!csrf_ok($_POST['csrf'] ?? '')) {
    $err = 'رمز الأمان غير صالح.';
  } else {
    $id = (int)($_POST['id'] ?? 0);
    $act = $_POST['action'] ?? '';
    try {
      if ($act === 'toggle_active' && $id>0) {
        $pdo->prepare("UPDATE store_items SET is_active = IF(is_active=1,0,1) WHERE id=? LIMIT 1")->execute([$id]);
        $msg = 'تم تبديل حالة التفعيل.';
      } elseif ($act === 'delete_item' && $id>0) {
        // يفضّل عدم الحذف إن كان هناك أوامر مرتبطة؛ هنا حذف بسيط
        $pdo->prepare("DELETE FROM store_items WHERE id=? LIMIT 1")->execute([$id]);
        $msg = 'تم حذف العنصر.';
      }
    } catch (Throwable $e) {
      $err = 'تعذّر تنفيذ العملية.' . (isset($_GET['dbg']) ? ' ('.$e->getMessage().')' : '');
    }
  }
}

/* ---------------- Filters ---------------- */
$q        = trim($_GET['q'] ?? '');
$cat      = trim($_GET['category'] ?? '');
$active   = $_GET['active'] ?? ''; // '', '1', '0'
$page     = max(1, (int)($_GET['page'] ?? 1));
$perPage  = 50;
$offset   = ($page-1)*$perPage;

$where = "1=1"; $params = [];
if ($q !== '') {
  $where .= " AND (name LIKE ? OR slug LIKE ?)";
  $kw = "%$q%"; array_push($params, $kw, $kw);
}
if ($cat !== '') { $where .= " AND category = ?"; $params[] = $cat; }
if ($active === '1' || $active === '0') { $where .= " AND is_active = ?"; $params[] = $active; }

/* إجمالي */
$total = 0;
try {
  $st = $pdo->prepare("SELECT COUNT(*) FROM store_items WHERE $where");
  $st->execute($params); $total = (int)$st->fetchColumn();
} catch (Throwable $e) { $err = 'تعذّر حساب الإجمالي.'; }

$pages = max(1, (int)ceil($total/$perPage));

/* البيانات */
$rows = [];
try {
  $sql = "SELECT id,category,slug,name,image_url,price,days,rating,is_active,sort_order,created_at
          FROM store_items
          WHERE $where
          ORDER BY sort_order ASC, id DESC
          LIMIT $perPage OFFSET $offset";
  $st = $pdo->prepare($sql); $st->execute($params);
  $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
} catch (Throwable $e) {
  $err = 'تعذّر جلب البيانات.' . (isset($_GET['dbg']) ? ' ('.$e->getMessage().')' : '');
}

/* -------------- Render ------------------ */
$PAGE_TITLE = "عناصر المتجر";
require $INC_DIR . '/header.php';
require $INC_DIR . '/sidebar.php';
?>
<style>
.items-page .table{width:100%;border-collapse:separate;border-spacing:0;background:var(--panel);border:1px solid var(--border);border-radius:12px;overflow:hidden;table-layout:fixed}
.items-page .table td,.items-page .table th{padding:10px 12px;border-bottom:1px solid var(--border);vertical-align:middle;text-align:start}
.items-page .table thead th{background:#0b1220;color:var(--text);font-weight:700;position:sticky;top:0;z-index:3;box-shadow:inset 0 -1px 0 var(--border)}
.items-page .table tbody tr:nth-child(even) td{background:rgba(255,255,255,.015)}
.items-page .table tbody tr:hover td{background:rgba(59,130,246,.08)}
.items-page .table-wrap{max-height:70vh;overflow:auto;position:relative}
.items-page .td-center{text-align:center}
.items-page .ellipsis{max-width:280px;display:inline-block;overflow:hidden;white-space:nowrap;text-overflow:ellipsis}
.items-page .badge{display:inline-block;padding:2px 8px;border-radius:999px;border:1px solid var(--border);font-size:12px}
.items-page .img{width:40px;height:40px;border-radius:8px;object-fit:cover;border:1px solid var(--border)}
.items-page .col-id{width:60px}.items-page .col-img{width:64px}.items-page .col-name{width:260px}.items-page .col-cat{width:120px}.items-page .col-price{width:120px}.items-page .col-days{width:90px}.items-page .col-active{width:90px}.items-page .col-sort{width:90px}.items-page .col-actions{width:220px}
</style>

<div class="items-page">
  <div class="card" style="margin-bottom:16px;">
    <form method="get" class="form" style="display:flex; gap:12px; align-items:flex-end; flex-wrap:wrap;">
      <div class="input" style="min-width:240px; flex:1;">
        <label>بحث بالاسم / المعرّف النصي</label>
        <input type="text" name="q" value="<?= e($q) ?>" placeholder="مثال: VIP / frame_gold">
      </div>
      <div class="input">
        <label>التصنيف</label>
        <select name="category">
          <option value="">الكل</option>
          <?php foreach (categories() as $k=>$v): ?>
            <option value="<?= e($k) ?>"<?= $cat===$k?' selected':''; ?>><?= e($v) ?> (<?= e($k) ?>)</option>
          <?php endforeach; ?>
        </select>
      </div>
      <div class="input">
        <label>الحالة</label>
        <select name="active">
          <option value="">الكل</option>
          <option value="1"<?= $active==='1'?' selected':''; ?>>مفعل</option>
          <option value="0"<?= $active==='0'?' selected':''; ?>>معطّل</option>
        </select>
      </div>
      <button class="btn sm" type="submit">بحث</button>
      <?php if ($q!=='' || $cat!=='' || $active!==''): ?>
        <a class="btn sm secondary" href="items.php">مسح الفلاتر</a>
      <?php endif; ?>
      <a class="btn sm" style="margin-inline-start:auto;" href="items_edit.php">+ إضافة عنصر</a>
      <div style="opacity:.9;">إجمالي: <b><?= $total ?></b></div>
    </form>
    <?php if ($msg): ?><div class="alert" style="margin-top:12px; border-color:#22c55e;"><?= e($msg) ?></div><?php endif; ?>
    <?php if ($err): ?><div class="alert" style="margin-top:12px;"><?= e($err) ?></div><?php endif; ?>
  </div>

  <div class="card">
    <div class="table-wrap">
      <table class="table">
        <colgroup>
          <col class="col-id"><col class="col-img"><col class="col-name"><col class="col-cat">
          <col class="col-price"><col class="col-days"><col class="col-active"><col class="col-sort"><col class="col-actions">
        </colgroup>
        <thead>
          <tr>
            <th class="td-center">#</th>
            <th>صورة</th>
            <th>الاسم / SLUG</th>
            <th>التصنيف</th>
            <th>السعر</th>
            <th>الأيام</th>
            <th>الحالة</th>
            <th>الترتيب</th>
            <th>إجراءات</th>
          </tr>
        </thead>
        <tbody>
          <?php foreach ($rows as $r): ?>
            <tr>
              <td class="td-center"><?= (int)$r['id'] ?></td>
              <td><?php if (!empty($r['image_url'])): ?><img class="img" src="<?= e($r['image_url']) ?>" alt=""><?php endif; ?></td>
              <td>
                <div class="ellipsis" title="<?= e($r['name']) ?>"><?= e($r['name']) ?></div>
                <div class="badge">slug: <?= e((string)$r['slug']) ?></div>
              </td>
              <td><span class="badge"><?= e($r['category']) ?></span></td>
              <td><?= fmt_money($r['price']) ?></td>
              <td class="td-center"><?= (int)$r['days'] ?></td>
              <td class="td-center"><?= ((int)$r['is_active']===1?'✅ مفعل':'⛔ معطّل') ?></td>
              <td class="td-center"><?= (int)$r['sort_order'] ?></td>
              <td style="display:flex; gap:8px; flex-wrap:nowrap;">
                <a class="btn sm secondary" href="items_edit.php?id=<?= (int)$r['id'] ?>">تعديل</a>
                <form method="post">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                  <input type="hidden" name="action" value="toggle_active">
                  <button class="btn sm" type="submit"><?= ((int)$r['is_active']===1?'تعطيل':'تفعيل') ?></button>
                </form>
                <form method="post" onsubmit="return confirm('حذف نهائي للعنصر؟');">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                  <input type="hidden" name="action" value="delete_item">
                  <button class="btn sm" type="submit" style="background:var(--danger);">حذف</button>
                </form>
              </td>
            </tr>
          <?php endforeach; ?>
          <?php if (!$rows): ?>
            <tr><td colspan="9" style="text-align:center; opacity:.8;">لا توجد بيانات</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>

    <?php if ($pages>1): ?>
      <div style="display:flex; gap:8px; margin-top:12px; flex-wrap:wrap;">
        <?php for($p=1;$p<=$pages;$p++):
          $url='items.php?page='.$p
            .($q!==''? '&q='.urlencode($q):'')
            .($cat!==''? '&category='.urlencode($cat):'')
            .($active!==''? '&active='.urlencode($active):'');
          $is=$p===$page; ?>
          <a href="<?= e($url) ?>" class="btn sm <?= $is ? '' : 'secondary' ?>"<?= $is?' style="pointer-events:none; opacity:.8;"':''; ?>><?= $p ?></a>
        <?php endfor; ?>
      </div>
    <?php endif; ?>
  </div>
</div>

<?php require $INC_DIR . '/footer.php'; ?>
