<?php
// wesaladmin/inc/debug.php
declare(strict_types=1);

/**
 * مخصص لبيئة الأدمن فقط — يعرض سبب الخطأ بدل HTTP 500 الفارغة،
 * ويسجل رسالة الخطأ في ملف لوج داخل wesaladmin/storage/logs/
 *
 * لتشغيل العرض المباشر: ضع ?debug=1 في الرابط أو فعّل ADMIN_DEBUG=true بالكونفيج.
 */

if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }

// حالة الديبج: من الكونفيج أو من بارامتر GET
$__ADMIN_DEBUG = defined('ADMIN_DEBUG') ? (bool)ADMIN_DEBUG : false;
if (isset($_GET['debug']) && $_GET['debug'] == '1') $__ADMIN_DEBUG = true;

// إعدادات الإبلاغ
if ($__ADMIN_DEBUG) {
  ini_set('display_errors', '1');
  error_reporting(E_ALL);
} else {
  ini_set('display_errors', '0');
  error_reporting(E_ALL); // نسجل في اللوج حتى لو لا نعرض
}

// مجلد اللوج
$__LOG_DIR = dirname(__DIR__) . '/storage/logs';
if (!is_dir($__LOG_DIR)) @mkdir($__LOG_DIR, 0775, true);
$__LOG_FILE = $__LOG_DIR . '/admin-php-' . date('Ymd') . '.log';

function __admin_log_line(string $msg): void {
  global $__LOG_FILE;
  @file_put_contents($__LOG_FILE, '['.date('H:i:s').'] '.$msg.PHP_EOL, FILE_APPEND);
}

// حوّل كل الأخطاء إلى Exceptions (عدا @-silenced)
set_error_handler(function ($severity, $message, $file, $line) {
  if (!(error_reporting() & $severity)) { return false; }
  throw new ErrorException($message, 0, $severity, $file, $line);
});

// التقاط الفاتال أخيرًا
register_shutdown_function(function() {
  $e = error_get_last();
  if ($e && in_array($e['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR, E_USER_ERROR], true)) {
    $__safe = htmlspecialchars($e['message'], ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
    $__file = htmlspecialchars($e['file'], ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
    $__line = (int)$e['line'];

    __admin_log_line("FATAL {$__safe} in {$__file}:{$__line}");

    if (defined('ADMIN_DEBUG') && ADMIN_DEBUG || (isset($_GET['debug']) && $_GET['debug'] == '1')) {
      http_response_code(500);
      echo '<div style="font-family:system-ui,-apple-system,Segoe UI,Roboto,Arial,sans-serif;max-width:960px;margin:40px auto;padding:16px;border:1px solid #e11;border-radius:10px;background:#fff5f5">';
      echo '<h2 style="margin:0 0 8px;color:#b00">PHP Fatal Error</h2>';
      echo '<div><strong>Message:</strong> '.$__safe.'</div>';
      echo '<div><strong>File:</strong> '.$__file.'</div>';
      echo '<div><strong>Line:</strong> '.$__line.'</div>';
      echo '<div style="margin-top:10px;font-size:12px;color:#555">This screen is visible because debug mode is on (ADMIN_DEBUG or ?debug=1)</div>';
      echo '</div>';
    } else {
      // شاشة مختصرة للمستخدم بدون تفاصيل
      http_response_code(500);
      echo 'Internal Server Error.';
    }
  }
});

// التقاط الاستثناءات غير المُعالجة
set_exception_handler(function(Throwable $ex) {
  $__msg  = htmlspecialchars($ex->getMessage(), ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
  $__file = htmlspecialchars($ex->getFile(), ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
  $__line = (int)$ex->getLine();

  __admin_log_line("EXC {$__msg} in {$__file}:{$__line}");

  if (defined('ADMIN_DEBUG') && ADMIN_DEBUG || (isset($_GET['debug']) && $_GET['debug'] == '1')) {
    http_response_code(500);
    echo '<div style="font-family:system-ui,-apple-system,Segoe UI,Roboto,Arial,sans-serif;max-width:960px;margin:40px auto;padding:16px;border:1px solid #e11;border-radius:10px;background:#fff5f5">';
    echo '<h2 style="margin:0 0 8px;color:#b00">Uncaught Exception</h2>';
    echo '<div><strong>Message:</strong> '.$__msg.'</div>';
    echo '<div><strong>File:</strong> '.$__file.'</div>';
    echo '<div><strong>Line:</strong> '.$__line.'</div>';
    echo '<div style="margin-top:10px;font-size:12px;color:#555">This screen is visible because debug mode is on (ADMIN_DEBUG or ?debug=1)</div>';
    echo '</div>';
  } else {
    http_response_code(500);
    echo 'Internal Server Error.';
  }
});
