<?php
// voicechat/wesaladmin/fcm_send.php
declare(strict_types=1);

require_once __DIR__ . '/inc/auth.php';
require_login();

/* ---------------- CSRF ---------------- */
if (!isset($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
function csrf_token(): string { return $_SESSION['csrf']; }
function csrf_ok(?string $t): bool { return $t && hash_equals($_SESSION['csrf'], $t); }

/* -------------- Helpers --------------- */
if (!function_exists('e')) {
  function e(string $s): string { return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }
}
function badge(string $text, string $class=''): string {
  $cls = 'badge'.($class ? ' '.$class : '');
  return '<span class="'.$cls.'">'.e($text).'</span>';
}
function ok_alert(string $t): string {
  return '<div class="alert" style="margin-top:12px;border-color:#22c55e">'.e($t).'</div>';
}
function err_alert(string $t): string {
  return '<div class="alert" style="margin-top:12px">'.e($t).'</div>';
}
// بديل array_is_list لأن بعض بيئات PHP قد لا تحتويها
if (!function_exists('array_is_list')) {
  function array_is_list(array $arr): bool {
    $i = 0;
    foreach ($arr as $k => $_) { if ($k !== $i++) return false; }
    return true;
  }
}

/* -------------- Config ---------------- */
$pdo = db();

// ضَع Project ID الصحيح لمشروع Firebase لديك (أو خزّنه في config.php/ENV واقرأه هنا)
$PROJECT_ID   = getenv('FIREBASE_PROJECT_ID') ?: 'wesal-chat-26277';
$SERVICE_JSON = realpath(__DIR__ . '/../api/keys/firebase-admin.json'); // كما طلبت
$FCM_LIB      = __DIR__ . '/../api/lib/fcm_lib.php';

require_once __DIR__ . '/../api/vendor/autoload.php'; // لو تستعمل composer
if (is_file($FCM_LIB)) {
  require_once $FCM_LIB;
} else {
  // منع الأعطال لو الملف غير موجود
  function fcm_get_access_token(string $p, array $s=["https://www.googleapis.com/auth/firebase.messaging"]): string { throw new RuntimeException('fcm_lib.php غير موجود'); }
  function fcm_send_message_v1(string $pid, array $m, string $tok): array { return []; }
}

/* -------------- Actions ---------------- */
$msg = null; $err = null;
if (($_SERVER['REQUEST_METHOD'] ?? '') === 'POST' && ($_POST['action'] ?? '') === 'send') {
  if (!csrf_ok($_POST['csrf'] ?? '')) {
    $err = 'رمز الأمان غير صالح. أعد المحاولة.';
  } elseif (!$SERVICE_JSON || !is_file($SERVICE_JSON)) {
    $err = 'لم يتم العثور على ملف firebase-admin.json في: ' . ($SERVICE_JSON ?: '../api/keys/firebase-admin.json');
  } elseif ($PROJECT_ID === 'YOUR_FIREBASE_PROJECT_ID') {
    $err = 'رجاءً عيّن $PROJECT_ID إلى Project ID الصحيح.';
  } else {
    $mode  = $_POST['mode'] ?? 'all'; // all | user | topic
    $title = trim((string)($_POST['title'] ?? ''));
    $body  = trim((string)($_POST['body']  ?? ''));
    $uid   = trim((string)($_POST['user_uid'] ?? ''));
    $topic = trim((string)($_POST['topic'] ?? ''));
    $data_json = trim((string)($_POST['data_json'] ?? ''));

    // ✅ ضمان أن data تكون Map بمفاتيح وقيم نصية فقط
    $data = [];
    if ($data_json !== '') {
      $tmp = json_decode($data_json, true);
      if ($tmp === null && json_last_error() !== JSON_ERROR_NONE) {
        $err = 'صيغة JSON غير صحيحة في الحقل "بيانات إضافية".';
      } else {
        if (!is_array($tmp) || array_is_list($tmp)) {
          $err = 'حقل "بيانات إضافية" يجب أن يكون كائن JSON مثل {"key":"value"} وليس مصفوفة [].';
        } else {
          foreach ($tmp as $k => $v) {
            // ممنوع كائنات/مصفوفات متداخلة داخل data — إن وجدت نحوّلها لسلسة
            if (is_array($v) || is_object($v)) {
              $v = json_encode($v, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
            }
            $data[(string)$k] = (string)$v;
          }
        }
      }
    }

    if (!$err && ($title === '' || $body === '')) {
      $err = 'العنوان والنص مطلوبان.';
    }

    if (!$err) {
      try {
        $access = fcm_get_access_token($SERVICE_JSON);

        if ($mode === 'all') {
          $message = [
            'topic' => 'all',
            'notification' => ['title' => $title, 'body' => $body],
            'android' => ['notification' => ['channel_id' => 'wesal_default']],
            'apns' => ['payload' => ['aps' => ['sound' => 'default']]],
          ];
          if (!empty($data)) $message['data'] = $data; // لا نرسل data إن كانت فارغة
          $res = fcm_send_message_v1($PROJECT_ID, $message, $access);
          $msg = 'تم الإرسال إلى جميع المشتركين في Topic all. الرد: ' . json_encode($res, JSON_UNESCAPED_UNICODE);

        } elseif ($mode === 'user') {
          if ($uid === '') throw new RuntimeException('أدخل user_uid.');
          $st = $pdo->prepare("SELECT token FROM fcm_tokens WHERE user_uid=?");
          $st->execute([$uid]);
          $tokens = $st->fetchAll(PDO::FETCH_COLUMN);
          if (!$tokens) throw new RuntimeException('لا توجد توكِنات لهذا المستخدم.');

          $sent=0; $errc=0; $last=null;
          foreach ($tokens as $t) {
            $message = [
              'token' => $t,
              'notification' => ['title' => $title, 'body' => $body],
              'android' => ['notification' => ['channel_id' => 'wesal_default']],
              'apns' => ['payload' => ['aps' => ['sound' => 'default']]],
            ];
            if (!empty($data)) $message['data'] = $data;
            $r = fcm_send_message_v1($PROJECT_ID, $message, $access);
            $last = $r;
            if (isset($r['name']) || isset($r['message_id'])) $sent++; else $errc++;
          }
          $msg = "تم الإرسال إلى {$sent} جهاز/أجهزة — أخطاء: {$errc}. آخر رد: ".json_encode($last, JSON_UNESCAPED_UNICODE);

        } elseif ($mode === 'topic') {
          if ($topic === '') throw new RuntimeException('أدخل اسم Topic.');
          $message = [
            'topic' => $topic,
            'notification' => ['title' => $title, 'body' => $body],
            'android' => ['notification' => ['channel_id' => 'wesal_default']],
            'apns' => ['payload' => ['aps' => ['sound' => 'default']]],
          ];
          if (!empty($data)) $message['data'] = $data;
          $res = fcm_send_message_v1($PROJECT_ID, $message, $access);
          $msg = 'تم الإرسال إلى Topic '.e($topic).'. الرد: ' . json_encode($res, JSON_UNESCAPED_UNICODE);
        } else {
          $err = 'وضع إرسال غير معروف.';
        }
      } catch (Throwable $e) {
        $err = 'خطأ: '.$e->getMessage();
      }
    }
  }
}

/* -------------- Tokens summary --------- */
$uq      = trim((string)($_GET['uq'] ?? '')); // بحث user_uid/token
$page    = max(1, (int)($_GET['page'] ?? 1));
$perPage = 25;
$offset  = ($page - 1) * $perPage;

$summary = ['android'=>0,'ios'=>0,'web'=>0,'total'=>0];
try {
  $summary['total'] = (int)($pdo->query("SELECT COUNT(*) FROM fcm_tokens")->fetchColumn() ?: 0);
  $rowsPlat = $pdo->query("SELECT platform, COUNT(*) c FROM fcm_tokens GROUP BY platform")->fetchAll(PDO::FETCH_KEY_PAIR) ?: [];
  foreach ($rowsPlat as $k=>$v) $summary[$k] = (int)$v;
} catch (Throwable $e) { /* ignore */ }

$where = "1=1"; $params=[];
if ($uq !== '') {
  $where .= " AND (user_uid LIKE ? OR token LIKE ?)";
  $kw = "%$uq%"; $params[]=$kw; $params[]=$kw;
}
try {
  $st = $pdo->prepare("SELECT COUNT(*) FROM fcm_tokens WHERE $where");
  $st->execute($params);
  $total = (int)$st->fetchColumn();
  $pages = max(1, (int)ceil($total / $perPage));

  $st = $pdo->prepare("SELECT id, user_uid, token, platform, app_ver, device_id, last_seen, created_at
                       FROM fcm_tokens
                       WHERE $where
                       ORDER BY last_seen DESC
                       LIMIT $perPage OFFSET $offset");
  $st->execute($params);
  $tokensRows = $st->fetchAll(PDO::FETCH_ASSOC);
} catch (Throwable $e) {
  $tokensRows = [];
  $total = 0; $pages = 1;
}

/* -------------- Render ------------------ */
$PAGE_TITLE = "إرسال إشعارات FCM";
require __DIR__ . '/inc/header.php';
require __DIR__ . '/inc/sidebar.php';
?>
<style>
.fcm-page .table{
  width:100%; border-collapse:separate; border-spacing:0; background:var(--panel);
  border:1px solid var(--border); border-radius:12px; overflow:hidden; table-layout:fixed;
}
.fcm-page .table td, .fcm-page .table th{ padding:10px 12px; border-bottom:1px solid var(--border); vertical-align:middle; text-align:start }
.fcm-page .table thead th{ background:#0b1220; color:var(--text); font-weight:700; position:sticky; top:0; z-index:3; box-shadow:inset 0 -1px 0 var(--border) }
.fcm-page .table tbody tr:nth-child(even) td{ background:rgba(255,255,255,.015) }
.fcm-page .table tbody tr:hover td{ background:rgba(59,130,246,.08) }
.fcm-page .table-wrap{ max-height:60vh; overflow:auto; position:relative }
.fcm-page .col-id{width:70px}
.fcm-page .col-uid{width:320px}
.fcm-page .col-plat{width:110px}
.fcm-page .col-app{width:120px}
.fcm-page .col-dev{width:200px}
.fcm-page .col-last{width:170px}
.fcm-page .col-actions{width:220px}
.fcm-page .mono{font-family:monospace}
.fcm-page .badge{display:inline-block;padding:2px 8px;border-radius:999px;border:1px solid var(--border);font-size:12px;opacity:.95}
.fcm-page .btn.sm{padding:6px 10px;font-size:14px}
</style>

<div class="fcm-page">

  <div class="card" style="margin-bottom:16px;">
    <h2 style="margin-top:0">إرسال إشعار جديد</h2>

    <?php if ($msg) echo ok_alert($msg); ?>
    <?php if ($err) echo err_alert($err); ?>

    <form method="post" class="form" style="display:grid; grid-template-columns:1fr 1fr; gap:14px;">
      <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
      <input type="hidden" name="action" value="send">

      <div class="input">
        <label>وضع الإرسال</label>
        <select name="mode" id="modeSel">
          <option value="all">لكل المستخدمين (Topic all)</option>
          <option value="user">مستخدم محدد</option>
          <option value="topic">Topic مخصص</option>
        </select>
      </div>

      <div class="input">
        <label>user_uid (لوضع: مستخدم محدد)</label>
        <input type="text" name="user_uid" placeholder="مثال: uMA0Rdz29LfmZi0ivKWo1on670v1">
      </div>

      <div class="input">
        <label>Topic (لوضع: Topic)</label>
        <input type="text" name="topic" placeholder="مثال: android_ar">
      </div>

      <div class="input">
        <label>العنوان (title)</label>
        <input type="text" name="title" required placeholder="عنوان الإشعار">
      </div>

      <div class="input" style="grid-column:1 / -1;">
        <label>النص (body)</label>
        <textarea name="body" rows="3" required placeholder="محتوى الإشعار"></textarea>
      </div>

      <div class="input" style="grid-column:1 / -1;">
        <label>بيانات إضافية (JSON في data) — اختيارية</label>
        <textarea name="data_json" rows="3" placeholder='{"type":"open_room","room_uid":"r_123"}'></textarea>
      </div>

      <div style="grid-column:1 / -1; display:flex; gap:8px; align-items:center; flex-wrap:wrap;">
        <button class="btn sm" type="submit">إرسال</button>
        <div class="badge">Project ID: <b class="mono"><?= e($PROJECT_ID) ?></b></div>
        <div class="badge">Service JSON: <b class="mono"><?= e((string)$SERVICE_JSON) ?></b></div>
      </div>
    </form>
  </div>

  <div class="card" style="margin-bottom:16px;">
    <h3 style="margin-top:0">ملخص التوكِنات</h3>
    <div style="display:flex; gap:10px; flex-wrap:wrap; opacity:.95;">
      <div><?= badge('Android','ok') ?> <b><?= (int)$summary['android'] ?></b></div>
      <div><?= badge('iOS','ok') ?> <b><?= (int)$summary['ios'] ?></b></div>
      <div><?= badge('Web','ok') ?> <b><?= (int)$summary['web'] ?></b></div>
      <div>الإجمالي: <b><?= (int)$summary['total'] ?></b></div>
    </div>
  </div>

  <div class="card">
    <form method="get" class="form" style="display:flex; gap:12px; align-items:flex-end; flex-wrap: wrap; margin-bottom:12px;">
      <div class="input" style="min-width:260px; flex:1;">
        <label>بحث (user_uid / token)</label>
        <input type="text" name="uq" value="<?= e($uq) ?>" placeholder="..">
      </div>
      <button class="btn sm" type="submit">بحث</button>
      <?php if ($uq!==''): ?>
        <a class="btn sm secondary" href="fcm_send.php">تفريغ البحث</a>
      <?php endif; ?>
    </form>

    <div class="table-wrap">
      <table class="table">
        <colgroup>
          <col class="col-id"><col class="col-uid"><col><col class="col-plat"><col class="col-app"><col class="col-dev"><col class="col-last"><col class="col-actions">
        </colgroup>
        <thead>
          <tr>
            <th>#</th>
            <th>user_uid</th>
            <th>token</th>
            <th>المنصة</th>
            <th>إصدار التطبيق</th>
            <th>الجهاز</th>
            <th>آخر ظهور</th>
            <th>إجراءات</th>
          </tr>
        </thead>
        <tbody>
          <?php if ($tokensRows): foreach ($tokensRows as $r): ?>
            <tr>
              <td class="mono"><?= (int)$r['id'] ?></td>
              <td class="mono" title="<?= e($r['user_uid']) ?>"><?= e($r['user_uid']) ?></td>
              <td class="mono" title="<?= e($r['token']) ?>" style="overflow:hidden;white-space:nowrap;text-overflow:ellipsis;"><?= e($r['token']) ?></td>
              <td><?= e($r['platform']) ?></td>
              <td class="mono"><?= e((string)($r['app_ver'] ?? '')) ?></td>
              <td class="mono"><?= e((string)($r['device_id'] ?? '')) ?></td>
              <td class="mono"><?= e((string)($r['last_seen'] ?? '')) ?></td>
              <td>
                <form method="post" class="actions" style="display:flex;gap:6px;align-items:center">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="action" value="send">
                  <input type="hidden" name="mode" value="user">
                  <input type="hidden" name="user_uid" value="<?= e($r['user_uid']) ?>">
                  <input type="hidden" name="title" value="إشعار من وصال">
                  <input type="hidden" name="body" value="مرحبًا!">
                  <button class="btn sm" type="submit" title="إرسال إشعار سريع لهذا المستخدم">إرسال سريع</button>
                </form>
              </td>
            </tr>
          <?php endforeach; else: ?>
            <tr><td colspan="8" style="text-align:center;opacity:.8;">لا توجد نتائج</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>

    <?php if (($pages ?? 1) > 1): ?>
      <div style="display:flex; gap:8px; margin-top:12px; flex-wrap:wrap;">
        <?php
          $qs = [];
          if ($uq!=='') $qs[] = 'uq='.urlencode($uq);
          $base = 'fcm_send.php'.($qs?('?'.implode('&',$qs)).'&':'?');
        ?>
        <?php for($p=1; $p<=$pages; $p++):
          $url = $base.'page='.$p;
          $is  = $p===$page; ?>
          <a href="<?= e($url) ?>" class="btn sm <?= $is ? '' : 'secondary' ?>"<?= $is?' style="pointer-events:none; opacity:.8;"':''; ?>><?= $p ?></a>
        <?php endfor; ?>
      </div>
    <?php endif; ?>
  </div>

</div>

<?php require __DIR__ . '/inc/footer.php'; ?>
