<?php
// admin/agent_rate.php
declare(strict_types=1);
require_once __DIR__ . '/inc/auth.php';
require_login();

if (!isset($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
function csrf_token(): string { return $_SESSION['csrf']; }
function csrf_ok(?string $t): bool { return $t && hash_equals($_SESSION['csrf'], $t); }
if (!function_exists('e')) { function e(string $s): string { return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); } }

$pdo = db();
$msg = null; $err = null;

/* helpers */
function load_one(PDO $pdo, string $k): string {
  $st = $pdo->prepare("SELECT `value` FROM app_settings WHERE `key`=? LIMIT 1");
  $st->execute([$k]);
  $v = $st->fetchColumn();
  return $v !== false ? (string)$v : '';
}
function save_one(PDO $pdo, string $k, string $v): void {
  $st = $pdo->prepare("REPLACE INTO app_settings (`key`,`value`) VALUES (?,?)");
  $st->execute([$k,$v]);
}
function effective_rate(?string $coins_per_usd, ?string $usd_per_coin): ?float {
  $cpu = trim((string)$coins_per_usd);
  $upc = trim((string)$usd_per_coin);
  if ($cpu !== '' && is_numeric($cpu) && (float)$cpu > 0) return (float)$cpu;
  if ($upc !== '' && is_numeric($upc) && (float)$upc > 0) return 1.0 / (float)$upc;
  return null;
}
function col_exists(PDO $pdo, string $table, string $col): bool {
  try { $st=$pdo->prepare("SHOW COLUMNS FROM `$table` LIKE ?"); $st->execute([$col]); return (bool)$st->fetchColumn(); }
  catch(Throwable $e){ return false; }
}

/* تحميل القيم */
$agent_coins_per_usd = load_one($pdo, 'agent_coins_per_usd');
$agent_usd_per_coin  = load_one($pdo, 'agent_usd_per_coin');

/* حفظ */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  if (!csrf_ok($_POST['csrf'] ?? '')) {
    $err = 'رمز الأمان غير صالح.';
  } else {
    $agent_coins_per_usd = trim((string)($_POST['agent_coins_per_usd'] ?? ''));
    $agent_usd_per_coin  = trim((string)($_POST['agent_usd_per_coin'] ?? ''));

    // التحقق
    if ($agent_coins_per_usd !== '' && (!is_numeric($agent_coins_per_usd) || (float)$agent_coins_per_usd <= 0)) {
      $err = 'قيمة (Coins لكل 1$) يجب أن تكون رقمًا أكبر من 0.';
    } elseif ($agent_usd_per_coin !== '' && (!is_numeric($agent_usd_per_coin) || (float)$agent_usd_per_coin <= 0)) {
      $err = 'قيمة (دولار لكل 1 كوين) يجب أن تكون رقمًا أكبر من 0.';
    } else {
      try {
        save_one($pdo, 'agent_coins_per_usd', $agent_coins_per_usd);
        save_one($pdo, 'agent_usd_per_coin',  $agent_usd_per_coin);
        $msg = 'تم حفظ سعر التحويل بنجاح.';
      } catch (Throwable $e) {
        $err = 'تعذّر الحفظ.';
      }
    }
  }
}

/* حساب السعر الفعّال */
$rate = effective_rate($agent_coins_per_usd, $agent_usd_per_coin);

/* فحص بنية جدول agents بخصوص الدولار */
$has_cents = col_exists($pdo, 'agents', 'usd_balance_cents');
$has_dec   = col_exists($pdo, 'agents', 'usd_balance');
$storage_hint = $has_cents ? 'بالسنت (INT cents)' : ($has_dec ? 'عشري (DECIMAL)' : 'غير مُعدّ — لا يوجد عمود دولار');

/* واجهة */
$PAGE_TITLE = 'إعداد سعر تحويل الوكلاء ($ ↔︎ كوين)';
require __DIR__ . '/inc/header.php';
require __DIR__ . '/inc/sidebar.php';
?>
<style>
  .settings-page .hint {opacity:.8;font-size:12px}
  .settings-page .mono {font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", monospace}
  .settings-page .grid-2 {display:grid;grid-template-columns:1fr 1fr;gap:12px}
  @media (max-width: 920px){ .settings-page .grid-2 {grid-template-columns:1fr;} }
  .settings-page .chips{display:flex;gap:8px;flex-wrap:wrap;margin:8px 0 4px}
  .settings-page .chip{display:inline-flex;align-items:center;gap:8px;border:1px solid var(--border);background:var(--panel);padding:8px 12px;border-radius:999px}
</style>

<div class="settings-page">
  <div class="card" style="max-width:980px;">
    <h1 style="margin-top:0"><?= e($PAGE_TITLE) ?></h1>
    <?php if($msg): ?><div class="alert" style="border-color:#22c55e;"><?= e($msg) ?></div><?php endif; ?>
    <?php if($err): ?><div class="alert"><?= e($err) ?></div><?php endif; ?>

    <div class="chips">
      <div class="chip">🗄️ طريقة تخزين الدولار في جدول <span class="mono">agents</span>: <b><?= e($storage_hint) ?></b></div>
      <div class="chip">📈 السعر الفعّال الحالي: 
        <b class="mono"><?= $rate !== null ? rtrim(rtrim(number_format($rate, 6, '.', ''), '0'), '.') : 'غير مضبوط' ?></b> كوين / $1
      </div>
    </div>

    <form method="post" class="form">
      <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">

      <h3>سعر التحويل</h3>
      <div class="grid-2">
        <div class="input">
          <label>عدد الكوين مقابل 1 دولار <span class="mono">(agent_coins_per_usd)</span> — <b>له أولوية</b></label>
          <input type="number" name="agent_coins_per_usd" value="<?= e($agent_coins_per_usd) ?>" min="0" step="0.000001" placeholder="مثال: 100">
          <div class="hint">إن كانت هذه الخانة مملوءة تُستخدم مباشرة (1$ = هذا العدد من الكوين).</div>
        </div>

        <div class="input">
          <label>كم دولار لكل 1 كوين <span class="mono">(agent_usd_per_coin)</span> — بديل</label>
          <input type="number" name="agent_usd_per_coin" value="<?= e($agent_usd_per_coin) ?>" min="0" step="0.000001" placeholder="مثال: 0.01">
          <div class="hint">تُستخدم فقط إذا كانت الخانة الأولى فارغة. السعر الفعّال = 1 ÷ هذه القيمة.</div>
        </div>
      </div>

      <div class="chips">
        <div class="chip">
          💡 تجربة سريعة: $ 
          <input id="try_usd" type="number" step="0.01" value="10" style="width:90px;border:none;outline:none;background:transparent;color:inherit"> 
          ⟶ كوين: <b class="mono" id="try_coins">—</b>
        </div>
      </div>

      <div style="display:flex;gap:8px;margin-top:14px;">
        <button class="btn" type="submit">حفظ السعر</button>
        <a class="btn secondary" href="app_settings.php">رجوع إلى إعدادات التطبيق</a>
      </div>

      <hr style="margin:18px 0;opacity:.25">

      <details>
        <summary style="cursor:pointer;">ملاحظات</summary>
        <div class="hint" style="margin-top:8px;line-height:1.7">
          - السعر يُخزّن في جدول <span class="mono">app_settings</span> بالمفتاحين المذكورين أعلاه.<br>
          - صفحات الوكلاء (الداشبورد/التحويل) تقرأ السعر الفعّال مباشرة من هنا.<br>
          - إذا لم يظهر عمود دولار في جدول <span class="mono">agents</span> ستظهر لك ملاحظة لتجهيزه (cents أو decimal).
        </div>
      </details>
    </form>
  </div>
</div>

<script>
(function(){
  const rate = <?= $rate !== null ? json_encode($rate) : 'null' ?>;
  const usd  = document.getElementById('try_usd');
  const out  = document.getElementById('try_coins');
  function calc(){
    if (!rate){ out.textContent = '—'; return; }
    const u = parseFloat(usd.value || '0');
    const c = Math.floor((u>0?u:0) * rate);
    out.textContent = c.toString();
  }
  if (usd && out){ usd.addEventListener('input', calc); calc(); }
})();
</script>

<?php require __DIR__ . '/inc/footer.php'; ?>
