<?php
declare(strict_types=1);
require __DIR__ . '/config.php';

/* ===== هيدرز عامة ومنع الكاش ===== */
header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');

/* ===== أدوات رد JSON ===== */
function jerr(string $err, array $extra = [], int $code = 200): void {
  http_response_code($code);
  echo json_encode(['ok' => false, 'error' => $err] + $extra, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  exit;
}
function jok(array $data, int $code = 200): void {
  http_response_code($code);
  echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  exit;
}

/**
 * استخراج التوكن من عدة أماكن (بعض السيرفرات لا تمُرّر HTTP_AUTHORIZATION تلقائيًا)
 * - يدعم: HTTP_AUTHORIZATION / Authorization / REDIRECT_HTTP_AUTHORIZATION
 * - يقبل شكل Bearer xxx أو حتى jwt ثلاثي مقسّم بنقاط (للاختبار)
 * - كـ fallback للاختبار فقط: ?token= أو POST[token]
 */
function get_bearer_token(): string {
  $candidates = [
    $_SERVER['HTTP_AUTHORIZATION']            ?? '',
    $_SERVER['Authorization']                 ?? '',
    $_SERVER['REDIRECT_HTTP_AUTHORIZATION']   ?? '',
  ];
  foreach ($candidates as $h) {
    if (!is_string($h) || $h === '') continue;
    $h = trim($h);
    if (stripos($h, 'Bearer ') === 0) return trim(substr($h, 7));
    // أحيانًا يصل التوكن بلا كلمة Bearer
    if (preg_match('~^[A-Za-z0-9\-_]+\.[A-Za-z0-9\-_]+\.[A-Za-z0-9\-_]+$~', $h)) return $h;
  }
  $q = $_GET['token']  ?? $_POST['token'] ?? '';
  if (is_string($q) && $q !== '') return trim($q);
  return '';
}

/**
 * محاولة الحصول على UID للمستخدم المصادق:
 * - إن وُجد require_auth() سنستخدمه
 * - وإن التقطنا التوكن يدويًا ولم يكن HTTP_AUTHORIZATION مضبوطًا، نُمرّره له
 */
function get_auth_uid_or_empty(): string {
  $token = get_bearer_token();
  if ($token !== '' && empty($_SERVER['HTTP_AUTHORIZATION'])) {
    // نضبطها ليقرأها require_auth()
    $_SERVER['HTTP_AUTHORIZATION'] = 'Bearer ' . $token;
  }
  if (function_exists('require_auth')) {
    try {
      $auth = require_auth(); // متوقّع يعيد ['uid' => '...']
      if (is_array($auth) && !empty($auth['uid'])) {
        return (string)$auth['uid'];
      }
    } catch (\Throwable $e) {
      // تجاهل: سنعتبره غير مصادق
    }
  }
  return '';
}

/* ===== Input ===== */
$param = trim((string)($_GET['room_uid'] ?? $_GET['id'] ?? $_POST['room_uid'] ?? $_POST['id'] ?? ''));
if ($param === '') jerr('missing_room_uid');

try {
  $pdo = db();

  // ابحث بالرقمي id أو بالنصي uid أو بالقناة agora_channel
  $isNumeric = ctype_digit($param);
  $sql = "SELECT id, uid, owner_uid, name FROM rooms WHERE ";
  $params = [];
  if ($isNumeric) {
    $sql .= "id = :rid OR ";
    $params[':rid'] = (int)$param;
  }
  $sql .= "uid = :ruid OR agora_channel = :ruid LIMIT 1";
  $params[':ruid'] = $param;

  $st = $pdo->prepare($sql);
  $st->execute($params);
  $room = $st->fetch(PDO::FETCH_ASSOC);
  if (!$room) jerr('room_not_found');

  $room_uid      = (string)($room['uid'] ?? '');
  $owner_raw     = (string)($room['owner_uid'] ?? ''); // قد تكون users.uid أو users.id (رقمي)
  $owner_uid     = '';   // users.uid النهائي للاستخدام في الواجهة
  $owner_id      = '';   // users.id (رقمي) إن توفر
  $owner_userid  = '';   // كود العرض للمستخدم (userid)

  // حوّل owner_raw إلى owner_uid/owner_id/owner_userid من جدول users
  if ($owner_raw !== '') {
    try {
      if (ctype_digit($owner_raw)) {
        // مخزّن كرقم users.id
        $owner_id = $owner_raw;
        $q = $pdo->prepare("SELECT uid, userid FROM users WHERE id = ? LIMIT 1");
        $q->execute([$owner_id]);
        if ($u = $q->fetch(PDO::FETCH_ASSOC)) {
          if (!empty($u['uid']))     $owner_uid    = (string)$u['uid'];
          if (!empty($u['userid']))  $owner_userid = (string)$u['userid'];
        }
      } else {
        // مخزّن كنصي users.uid
        $owner_uid = $owner_raw;
        $q = $pdo->prepare("SELECT id, userid FROM users WHERE uid = ? LIMIT 1");
        $q->execute([$owner_uid]);
        if ($u = $q->fetch(PDO::FETCH_ASSOC)) {
          if (isset($u['id']))       $owner_id     = (string)$u['id'];
          if (!empty($u['userid']))  $owner_userid = (string)$u['userid'];
        }
      }
    } catch (\Throwable $e) {
      // تجاهل: سنُعيد ما أمكن
    }
  }

  // حساب is_owner: يتطلّب توكن صالح
  $is_owner = false;
  $auth_uid = get_auth_uid_or_empty();
  if ($auth_uid !== '') {
    // مقارنة مباشرة بالـ uid إن توفر
    if ($owner_uid !== '' && $auth_uid === $owner_uid) {
      $is_owner = true;
    } elseif ($owner_id !== '') {
      // نحصل على users.id الخاص بالمستخدم الحالي من uid للمقارنة الرقمية
      try {
        $m = $pdo->prepare("SELECT id FROM users WHERE uid = ? LIMIT 1");
        $m->execute([$auth_uid]);
        if ($me = $m->fetch(PDO::FETCH_ASSOC)) {
          if ((string)$me['id'] === $owner_id) $is_owner = true;
        }
      } catch (\Throwable $e) {
        // تجاهل
      }
    }
  }

  // مخرجات موحّدة وثابتة للاستخدام في Flutter
  jok([
    'ok'           => true,
    'room_uid'     => $room_uid,
    'owner_uid'    => $owner_uid,     // هذا هو UID الحقيقي للمقارنة في التطبيق
    'owner_userid' => $owner_userid,  // كود عرض فقط (لا يُستخدم للمقارنة)
    'is_owner'     => $is_owner,
  ]);

} catch (\Throwable $e) {
  jerr('server_error', ['msg' => $e->getMessage()], 500);
}
