<?php
// voicechat/api/room_password.php
declare(strict_types=1);
require __DIR__ . '/config.php';

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Authorization, Content-Type');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }

function ok($d = []) { echo json_encode(['ok' => true] + $d, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES); exit; }
function bad($m, $code = 400) { http_response_code($code); echo json_encode(['ok' => false, 'error' => $m], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES); exit; }

try {
  $pdo = db();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

  $method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
  // نقرأ المدخلات
  $raw = file_get_contents('php://input') ?: '';
  $body = json_decode($raw, true);
  if (!is_array($body)) $body = $_POST;

  // params المشتركة
  $room_uid = trim((string)($_GET['room_uid'] ?? $body['room_uid'] ?? ''));
  $action   = strtolower((string)($_GET['action'] ?? $body['action'] ?? 'status'));

  if ($room_uid === '') bad('room_uid required');

  // helper: إحضار الغرفة
  $st = $pdo->prepare("SELECT id, uid, owner_uid, lock_enabled, lock_pin_hash FROM rooms WHERE uid = ? LIMIT 1");
  $st->execute([$room_uid]);
  $room = $st->fetch(PDO::FETCH_ASSOC);
  if (!$room) bad('Room not found', 404);

  // -------- status: عام، لا يحتاج توكن --------
  if ($action === 'status') {
    $locked = (bool)($room['lock_enabled'] ?? 0);
    ok(['locked' => $locked]);
  }

  // باقي الأفعال تحتاج توكن
  $auth = function_exists('require_auth') ? require_auth() : null;
  if (!$auth) bad('missing_token', 401);
  $user_uid = (string)($auth['uid'] ?? '');

  // تحقق الملكية لمرة واحدة عند set
  $ensureOwner = function () use ($room, $user_uid) {
    if ((string)$room['owner_uid'] !== $user_uid) {
      bad('forbidden_not_owner', 403);
    }
  };

  if ($action === 'set') {
    $enabled = (string)($body['enabled'] ?? '') === '1';
    $pin     = isset($body['pin']) ? trim((string)$body['pin']) : null;

    $ensureOwner();

    if ($enabled) {
      if ($pin === null || !preg_match('/^\d{4}$/', $pin)) bad('pin_must_be_4_digits');
      $hash = password_hash($pin, PASSWORD_DEFAULT);
      $st = $pdo->prepare("UPDATE rooms SET lock_enabled = 1, lock_pin_hash = ? WHERE uid = ?");
      $st->execute([$hash, $room_uid]);
      ok(['enabled' => true]);
    } else {
      // السماح بـ NULL
      $st = $pdo->prepare("UPDATE rooms SET lock_enabled = 0, lock_pin_hash = NULL WHERE uid = ?");
      $st->execute([$room_uid]);
      ok(['enabled' => false]);
    }
  }

  if ($action === 'check') {
    $pin = trim((string)($body['pin'] ?? ''));
    if (!preg_match('/^\d{4}$/', $pin)) bad('pin_must_be_4_digits');

    if (!(bool)$room['lock_enabled']) {
      // الغرفة غير مقفولة أصلاً
      ok(['access_token' => 'OPEN']);
    }

    $hash = (string)($room['lock_pin_hash'] ?? '');
    if ($hash === '' || !password_verify($pin, $hash)) {
      bad('wrong_pin', 403);
    }

    // توكن بسيط خاص بالغرفة (يمكن استبداله بـ JWT لديك)
    $secret = 'room-secret-change-me';
    $token  = hash('sha256', $room_uid . '|' . $user_uid . '|' . $secret . '|' . time());
    ok(['access_token' => $token]);
  }

  bad('unknown_action');
} catch (Throwable $e) {
  // لا نكشف التفاصيل للمستخدم، فقط نكتبها في اللوج
  error_log('[room_password] ' . $e->getMessage());
  bad('internal_error', 500);
}
