<?php
// room_mic_state.php
declare(strict_types=1);

require __DIR__ . '/config.php';

function ok($data, int $code = 200) {
  http_response_code($code);
  header('Content-Type: application/json; charset=utf-8');
  echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  exit;
}
function bad(string $msg, int $code = 400) { ok(['ok' => false, 'error' => $msg], $code); }

/** مصادقة اختيارية للقراءة، لكن سنشترطها للكتابة */
$auth = function_exists('require_auth') ? require_auth() : ['uid' => 'guest'];
$user_uid = (string)($auth['uid'] ?? 'guest');

$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
$pdo = db();

if ($method === 'GET') {
  $room_uid = trim((string)($_GET['room_uid'] ?? ''));
  if ($room_uid === '') bad('missing_room_uid', 422);

  // NOTE:
  // - username / avatar_url: من room_mics إن وُجدت وإلا من users.
  // - نجلب frame_pag_url من users ونُرجعه باسمين لراحة العميل:
  //   frame_url و frame_pag_url (نفس القيمة).
  $sql = "
    SELECT
      m.seat_index,
      m.user_uid,
      COALESCE(NULLIF(m.username, ''), u.username, '')    AS username,
      COALESCE(NULLIF(m.avatar_url,''), u.avatar_url, '') AS avatar_url,
      COALESCE(u.frame_pag_url, '')                       AS frame_pag_url
    FROM room_mics m
    LEFT JOIN users u ON u.uid = m.user_uid
    WHERE m.room_uid = ?
    ORDER BY m.seat_index ASC
  ";
  $st = $pdo->prepare($sql);
  $st->execute([$room_uid]);
  $rows = $st->fetchAll(PDO::FETCH_ASSOC);

  $seats = [];
  foreach ($rows as $r) {
    $uid      = (string)($r['user_uid'] ?? '');
    $frame    = (string)($r['frame_pag_url'] ?? '');
    $seats[] = [
      'seat_index'     => (int)$r['seat_index'],
      'user_uid'       => $uid,
      'uid'            => $uid, // alias لتوافق العميل
      'username'       => (string)$r['username'],
      'avatar_url'     => (string)$r['avatar_url'],
      'frame_url'      => $frame,          // ← نفس القيمة
      'frame_pag_url'  => $frame,          // ← الاسم المتوقع في الواجهة
      'is_me'          => ($uid !== '' && $uid === $user_uid),
    ];
  }

  ok(['ok' => true, 'room_uid' => $room_uid, 'seats' => $seats]);
}

/* ========================= كتابة (أخذ/تحرير مقعد) ========================= */
if ($method === 'POST') {
  // نمنع الكتابة بدون مصادقة فعلية
  if ($user_uid === 'guest') {
    bad('unauthorized', 401);
  }

  // دعم JSON و x-www-form-urlencoded
  $raw  = file_get_contents('php://input');
  $body = json_decode($raw, true);
  if (!is_array($body)) $body = $_POST ?? [];

  $action     = strtolower(trim((string)($body['action'] ?? '')));
  $room_uid   = trim((string)($body['room_uid'] ?? ''));
  $seat_index = array_key_exists('seat_index', $body) ? (int)$body['seat_index'] : -1;
  $username   = trim((string)($body['username'] ?? ''));
  $avatar_url = trim((string)($body['avatar_url'] ?? ''));

  if ($room_uid === '') bad('invalid_params', 422);

  if ($action === 'take') {
    if ($seat_index < 0) bad('invalid_seat', 422);

    try {
      $pdo->beginTransaction();

      // اقفل صف المقعد لمنع سباق التحديثات
      $st = $pdo->prepare("SELECT user_uid FROM room_mics WHERE room_uid=? AND seat_index=? FOR UPDATE");
      $st->execute([$room_uid, $seat_index]);
      $row = $st->fetch(PDO::FETCH_ASSOC);

      // إن كان محجوزًا لشخص آخر — ارفض
      if ($row && (string)$row['user_uid'] !== $user_uid) {
        $pdo->rollBack();
        bad('seat_taken', 409);
      }

      // امنع تعدد المقاعد لنفس المستخدم داخل نفس الغرفة
      $del = $pdo->prepare("DELETE FROM room_mics WHERE room_uid=? AND user_uid=?");
      $del->execute([$room_uid, $user_uid]);

      // احجز/حدِّث المقعد
      $ins = $pdo->prepare(
        "REPLACE INTO room_mics (room_uid, seat_index, user_uid, username, avatar_url, updated_at)
         VALUES (?,?,?,?,?,NOW())"
      );
      $ins->execute([
        $room_uid,
        $seat_index,
        $user_uid,
        ($username !== '' ? $username : $user_uid),
        ($avatar_url !== '' ? $avatar_url : null),
      ]);

      $pdo->commit();
      ok(['ok' => true]);
    } catch (Throwable $e) {
      if ($pdo->inTransaction()) $pdo->rollBack();
      bad('db_error', 500);
    }
  }
  elseif ($action === 'release') {
    try {
      if ($seat_index >= 0) {
        // حرِّر مقعدًا محددًا إن كان يخص هذا المستخدم
        $st = $pdo->prepare("DELETE FROM room_mics WHERE room_uid=? AND seat_index=? AND user_uid=?");
        $st->execute([$room_uid, $seat_index, $user_uid]);
      } else {
        // أو حرِّر أي مقعد يخص هذا المستخدم داخل الغرفة
        $st = $pdo->prepare("DELETE FROM room_mics WHERE room_uid=? AND user_uid=?");
        $st->execute([$room_uid, $user_uid]);
      }
      ok(['ok' => true]);
    } catch (Throwable $e) {
      bad('db_error', 500);
    }
  }
  else {
    bad('unknown_action', 422);
  }
}

bad('method_not_allowed', 405);
