<?php
// backend/api/room_join_leave.php
declare(strict_types=1);

require __DIR__ . '/config.php';

/** مدة صلاحية الـ heartbeat بالثواني قبل اعتبار العضو "متجمداً" وإزالته */
define('HEARTBEAT_TTL', 25);

/** يقرأ JSON من body ويُمزجه مع $_POST دون أن يطغى أحدهما */
function input_array(): array {
  $j = [];
  $raw = file_get_contents('php://input');
  if ($raw !== false && $raw !== '') {
    $dec = json_decode($raw, true);
    if (is_array($dec)) $j = $dec;
  }
  foreach ($_POST as $k=>$v) {
    if (!array_key_exists($k, $j)) $j[$k] = $v;
  }
  return $j;
}

/** يتأكد أن الغرفة موجودة ومفتوحة (عدّل شرط الحالة حسب سكيمتك) */
function assert_room_exists(PDO $pdo, string $room_uid): void {
  $st = $pdo->prepare("SELECT 1 FROM rooms WHERE uid=? AND status='open' LIMIT 1");
  $st->execute([$room_uid]);
  if (!$st->fetch()) {
    json_out(['ok'=>false, 'error'=>'room_not_found'], 404);
  }
}

/** إحضار سجل الحظر إن وُجد */
function fetch_ban(PDO $pdo, string $room_uid, string $user_uid): ?array {
  try {
    $st = $pdo->prepare("
      SELECT is_permanent, banned_until, reason
      FROM room_bans
      WHERE room_uid=? AND target_uid=?
      LIMIT 1
    ");
    $st->execute([$room_uid, $user_uid]);
    $row = $st->fetch(PDO::FETCH_ASSOC);
    return $row ?: null;
  } catch (\Throwable $e) {
    return null;
  }
}

/** إزالة الحظر المؤقت المنتهي (تنظيف) */
function prune_expired_ban(PDO $pdo, string $room_uid, string $user_uid): void {
  try {
    $st = $pdo->prepare("
      DELETE FROM room_bans
      WHERE room_uid=? AND target_uid=? AND is_permanent=0
        AND (banned_until IS NULL OR banned_until <= NOW())
    ");
    $st->execute([$room_uid, $user_uid]);
  } catch (\Throwable $e) {
    // تجاهل
  }
}

/** إن كان المستخدم محظورًا حاليًا يعيد مصفوفة فيها النوع والموعد، وإلا يعيد null */
function active_ban(PDO $pdo, string $room_uid, string $user_uid): ?array {
  $ban = fetch_ban($pdo, $room_uid, $user_uid);
  if (!$ban) return null;

  $isPermanent = (int)($ban['is_permanent'] ?? 0) === 1;
  $untilStr    = $ban['banned_until'] ?? null;
  $untilTs     = $untilStr ? strtotime((string)$untilStr) : null;

  if ($isPermanent) {
    return ['type'=>'permanent', 'until'=>null];
  }
  if ($untilTs !== null && $untilTs > time()) {
    // مؤقت وما زال ساريًا
    return ['type'=>'temp', 'until'=>date('Y-m-d H:i:s', $untilTs)];
  }

  // منتهي: نظّفه وأعد null
  prune_expired_ban($pdo, $room_uid, $user_uid);
  return null;
}

/**
 * تحرير أي مقعد محجوز للمستخدم داخل الغرفة.
 * (متوافق مع room_mics الحالية)
 */
function release_mic_for_user(PDO $pdo, string $room_uid, string $user_uid): void {
  try {
    $st = $pdo->prepare("DELETE FROM room_mics WHERE room_uid=? AND user_uid=?");
    $st->execute([$room_uid, $user_uid]);
  } catch (\Throwable $e) {
    // تجاهل وسنحاول fallbacks بالأسفل
  }

  // fallbacks اختيارية لو كانت لديك جداول/أعمدة مختلفة
  $candidates = [
    "UPDATE room_mics
       SET user_uid=NULL, username=NULL, avatar_url=NULL
     WHERE room_uid=? AND user_uid=?",
    "UPDATE room_mic_state
       SET user_uid=NULL, username=NULL, avatar_url=NULL
     WHERE room_uid=? AND user_uid=?",
    "UPDATE room_seats
       SET user_uid=NULL, username=NULL, avatar_url=NULL, taken=0
     WHERE room_uid=? AND user_uid=?",
  ];

  foreach ($candidates as $sql) {
    try {
      $st = $pdo->prepare($sql);
      $st->execute([$room_uid, $user_uid]);
    } catch (\Throwable $e) {
      // قد لا توجد الجداول/الأعمدة — تجاهل
    }
  }
}

/** إدراج/تحديث العضو داخل الغرفة */
function upsert_member(PDO $pdo, string $room_uid, string $user_uid): void {
  try {
    $st = $pdo->prepare(
      "INSERT INTO room_members (room_uid, user_uid, joined_at, last_seen)
       VALUES (?,?, NOW(), NOW())
       ON DUPLICATE KEY UPDATE last_seen=VALUES(last_seen)"
    );
    $st->execute([$room_uid, $user_uid]);
  } catch (\Throwable $e) {
    try {
      $st = $pdo->prepare(
        "INSERT INTO room_members (room_uid, user_uid)
         VALUES (?,?)
         ON DUPLICATE KEY UPDATE user_uid=VALUES(user_uid)"
      );
      $st->execute([$room_uid, $user_uid]);
    } catch (\Throwable $e2) {
      json_out(['ok'=>false, 'error'=>'members_upsert_failed', 'msg'=>$e2->getMessage()], 500);
    }
  }
}

/** تحديث last_seen (إن وُجد العمود) */
function touch_member(PDO $pdo, string $room_uid, string $user_uid): void {
  try {
    $st = $pdo->prepare("UPDATE room_members SET last_seen=NOW() WHERE room_uid=? AND user_uid=?");
    $st->execute([$room_uid, $user_uid]);
  } catch (\Throwable $e) {
    // ربما العمود غير موجود — تجاهل
  }
}

/** هل المستخدم منضم حاليًا؟ */
function is_joined(PDO $pdo, string $room_uid, string $user_uid): bool {
  try {
    $st = $pdo->prepare("SELECT 1 FROM room_members WHERE room_uid=? AND user_uid=? LIMIT 1");
    $st->execute([$room_uid, $user_uid]);
    return (bool)$st->fetchColumn();
  } catch (\Throwable $e) {
    return false;
  }
}

/** عدد الأعضاء المنضمين */
function members_count(PDO $pdo, string $room_uid): int {
  try {
    $st = $pdo->prepare("SELECT COUNT(*) FROM room_members WHERE room_uid=?");
    $st->execute([$room_uid]);
    return (int)$st->fetchColumn();
  } catch (\Throwable $e) {
    return 0;
  }
}

/** قائمة الأعضاء (مع محاولة جلب الاسم/الأفاتار إن كان لديك جدول users) */
function members_list(PDO $pdo, string $room_uid): array {
  // نحاول join على جدول users إن وُجد
  try {
    $sql = "
      SELECT m.user_uid AS uid,
             COALESCE(u.username, u.name, u.nickname, '') AS username,
             COALESCE(u.avatar_url, u.avatar, u.image_url, '') AS avatar_url
      FROM room_members m
      LEFT JOIN users u ON u.uid = m.user_uid
      WHERE m.room_uid=?
      ORDER BY m.joined_at ASC, m.user_uid ASC
    ";
    $st = $pdo->prepare($sql);
    $st->execute([$room_uid]);
    $rows = $st->fetchAll(PDO::FETCH_ASSOC);

    if (!$rows || !is_array($rows)) {
      throw new \RuntimeException('no_rows');
    }

    $out = [];
    foreach ($rows as $r) {
      $out[] = [
        'uid'        => (string)($r['uid'] ?? ''),
        'username'   => (string)($r['username'] ?? ''),
        'avatar_url' => (string)($r['avatar_url'] ?? ''),
      ];
    }
    return $out;
  } catch (\Throwable $e) {
    // بدون users
    try {
      $st = $pdo->prepare("
        SELECT user_uid AS uid
        FROM room_members
        WHERE room_uid=?
        ORDER BY joined_at ASC, user_uid ASC
      ");
      $st->execute([$room_uid]);
      $rows = $st->fetchAll(PDO::FETCH_ASSOC);

      $out = [];
      foreach ($rows as $r) {
        $uid = (string)($r['uid'] ?? '');
        $out[] = [
          'uid'        => $uid,
          'username'   => $uid,
          'avatar_url' => '',
        ];
      }
      return $out;
    } catch (\Throwable $e2) {
      return [];
    }
  }
}

/** 🔥 تنظيف تلقائي: إزالة الأعضاء الذين لم يرسلوا heartbeat وتحرير مقاعدهم */
function prune_stale_members(PDO $pdo, string $room_uid): void {
  try {
    $pdo->beginTransaction();

    // من هم المتجمدون؟
    $st = $pdo->prepare("
      SELECT user_uid
      FROM room_members
      WHERE room_uid=? 
        AND (last_seen IS NULL OR last_seen < (NOW() - INTERVAL ? SECOND))
    ");
    $st->execute([$room_uid, HEARTBEAT_TTL]);
    $stale = $st->fetchAll(PDO::FETCH_COLUMN);
    if (!$stale) $stale = [];

    // حرّر المايكات لكل متجمّد
    foreach ($stale as $uid) {
      release_mic_for_user($pdo, $room_uid, (string)$uid);
    }

    // احذفهم من room_members
    $st = $pdo->prepare("
      DELETE FROM room_members
      WHERE room_uid=? 
        AND (last_seen IS NULL OR last_seen < (NOW() - INTERVAL ? SECOND))
    ");
    $st->execute([$room_uid, HEARTBEAT_TTL]);

    // تنظيف أي مايك يخص مستخدمًا خرج بالفعل (حماية إضافية)
    $st = $pdo->prepare("
      DELETE rm
      FROM room_mics rm
      LEFT JOIN room_members m
        ON m.room_uid = rm.room_uid AND m.user_uid = rm.user_uid
      WHERE rm.room_uid = ? AND m.user_uid IS NULL
    ");
    $st->execute([$room_uid]);

    $pdo->commit();
  } catch (\Throwable $e) {
    if ($pdo->inTransaction()) $pdo->rollBack();
  }
}

// -------------------------------------------------------------
$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';

try {
  $auth = require_auth(); // يُرجع 401 عند فشل التوكن
  $user_uid = (string)($auth['uid'] ?? '');
  if ($user_uid === '') json_out(['ok'=>false,'error'=>'bad_user'], 401);

  $pdo = db();

  if ($method === 'GET') {
    $room_uid = trim((string)($_GET['room_uid'] ?? ''));
    $action   = strtolower((string)($_GET['action'] ?? ''));

    if ($room_uid === '') json_out(['ok'=>false,'error'=>'missing_room_uid'], 422);

    // تنظيف سريع قبل أي رد
    prune_stale_members($pdo, $room_uid);

    if ($action === 'list') {
      // قائمة المنضمين
      $items = members_list($pdo, $room_uid);
      json_out(['ok'=>true, 'items'=>$items]);
    } else {
      // الحالة (joined + count) + إشارة الحظر (اختياري معلوماتية)
      $joined = is_joined($pdo, $room_uid, $user_uid);
      $count  = members_count($pdo, $room_uid);
      $ban    = active_ban($pdo, $room_uid, $user_uid);
      if ($ban) {
        json_out(['ok'=>true, 'joined'=>$joined, 'count'=>$count, 'ban'=>$ban]);
      } else {
        json_out(['ok'=>true, 'joined'=>$joined, 'count'=>$count]);
      }
    }
  }

  // POST: join/leave/ping
  $in = input_array();
  $room_uid = trim((string)($in['room_uid'] ?? ''));
  $action   = strtolower((string)($in['action'] ?? 'join'));

  if ($room_uid === '') json_out(['ok'=>false,'error'=>'missing_room_uid'], 422);

  // تنظيف سريع في كل نداء POST
  prune_stale_members($pdo, $room_uid);

  if ($action === 'join') {
    assert_room_exists($pdo, $room_uid);

    // ✅ منع الانضمام إن كان محظورًا (دائمًا أو مؤقتًا)
    $ban = active_ban($pdo, $room_uid, $user_uid);
    if ($ban) {
      if ($ban['type'] === 'permanent') {
        json_out(['ok'=>false, 'error'=>'banned_permanent'], 403);
      } else {
        json_out(['ok'=>false, 'error'=>'banned_temp', 'until'=>$ban['until']], 403);
      }
    }

    upsert_member($pdo, $room_uid, $user_uid);

    // تنظيف إضافي بعد الانضمام (غير ضروري لكن لا يضر)
    prune_stale_members($pdo, $room_uid);

    json_out(['ok'=>true, 'action'=>'join']);
  }
  elseif ($action === 'leave') {
    $pdo->beginTransaction();
    try {
      // أولًا: حرّر المايك (إن وُجد)
      release_mic_for_user($pdo, $room_uid, $user_uid);

      // ثانيًا: احذف العضوية
      $st = $pdo->prepare("DELETE FROM room_members WHERE room_uid=? AND user_uid=?");
      $st->execute([$room_uid, $user_uid]);

      $pdo->commit();

      // تنظيف جماعي بعد الخروج
      prune_stale_members($pdo, $room_uid);

      json_out(['ok'=>true, 'action'=>'leave']);
    } catch (\Throwable $e) {
      if ($pdo->inTransaction()) $pdo->rollBack();
      json_out(['ok'=>false, 'error'=>'leave_failed', 'msg'=>$e->getMessage()], 500);
    }
  }
  elseif ($action === 'ping') {
    // ✅ لو تم حظر المستخدم بعد دخوله، اخرجه قسرًا
    $ban = active_ban($pdo, $room_uid, $user_uid);
    if ($ban) {
      // تنظيف: تحرير المايك والخروج من العضوية
      try {
        $pdo->beginTransaction();
        release_mic_for_user($pdo, $room_uid, $user_uid);
        $st = $pdo->prepare("DELETE FROM room_members WHERE room_uid=? AND user_uid=?");
        $st->execute([$room_uid, $user_uid]);
        $pdo->commit();
      } catch (\Throwable $e) {
        if ($pdo->inTransaction()) $pdo->rollBack();
      }

      if ($ban['type'] === 'permanent') {
        json_out(['ok'=>false, 'error'=>'banned_permanent'], 403);
      } else {
        json_out(['ok'=>false, 'error'=>'banned_temp', 'until'=>$ban['until']], 403);
      }
    }

    // تحديث last_seen بشكل دوري
    touch_member($pdo, $room_uid, $user_uid);

    // تنظيف جماعي (قد يزيل أشباح باقي الأعضاء)
    prune_stale_members($pdo, $room_uid);

    json_out(['ok'=>true, 'action'=>'ping']);
  }
  else {
    json_out(['ok'=>false, 'error'=>'invalid_action'], 422);
  }

} catch (\Throwable $e) {
  json_out(['ok'=>false, 'error'=>'server_error', 'msg'=>$e->getMessage()], 500);
}
