<?php
// /public_html/voicechat/agents/web/convert.php
declare(strict_types=1);
require_once __DIR__ . '/_web_common.php';

$auth   = require_agent_or_redirect();
$token  = $auth['token'];
$agent  = $auth['agent'];

$pdo = db();
$msg = null; $err = null; $rate = null; $after = null;

/* ---- اجلب بيانات الوكيل مباشرة من DB لضمان وجود أعمدة الدولار ---- */
try {
  $st = $pdo->prepare("
    SELECT id, username, main_balance,
           IFNULL(usd_balance_cents, NULL) AS usd_balance_cents,
           IFNULL(usd_balance,       NULL) AS usd_balance
    FROM agents
    WHERE id = ?
    LIMIT 1
  ");
  $st->execute([(int)$agent['id']]);
  if ($row = $st->fetch(PDO::FETCH_ASSOC)) {
    $agent['main_balance'] = (int)$row['main_balance'];
    if ($row['usd_balance_cents'] !== null) {
      $agent['usd_display'] = number_format(((int)$row['usd_balance_cents'])/100, 2, '.', '');
    } elseif ($row['usd_balance'] !== null) {
      $agent['usd_display'] = number_format((float)$row['usd_balance'], 2, '.', '');
    } else {
      $agent['usd_display'] = '0.00';
    }
  }
} catch (Throwable $e) {
  // تجاهل القراءة — الواجهة ستعرض 0.00 إن فشلت
}

/* ---- قراءة سعر التحويل من app_settings مباشرة ----
   الأولوية: agent_coins_per_usd (عدد الكوين لكل 1$)
   بديل:     agent_usd_per_coin  (كم دولار لكل 1 كوين) => rate = 1 / القيمة
   احتياطي:  مفاتيح قديمة (usd_to_coins_rate / rate_coins_per_usd / coins_per_usd)
-------------------------------------------------------------------------- */
try {
  $st = $pdo->query("
    SELECT `key`,`value` FROM app_settings
    WHERE `key` IN ('agent_coins_per_usd','agent_usd_per_coin','usd_to_coins_rate','rate_coins_per_usd','coins_per_usd')
  ");
  $pairs = $st->fetchAll(PDO::FETCH_KEY_PAIR);

  if (!empty($pairs['agent_coins_per_usd']) && (float)$pairs['agent_coins_per_usd'] > 0) {
    $rate = (float)$pairs['agent_coins_per_usd'];
  } elseif (!empty($pairs['agent_usd_per_coin']) && (float)$pairs['agent_usd_per_coin'] > 0) {
    $rate = 1.0 / (float)$pairs['agent_usd_per_coin'];
  } elseif (!empty($pairs['usd_to_coins_rate']) && (float)$pairs['usd_to_coins_rate'] > 0) {
    $rate = (float)$pairs['usd_to_coins_rate'];
  } elseif (!empty($pairs['rate_coins_per_usd']) && (float)$pairs['rate_coins_per_usd'] > 0) {
    $rate = (float)$pairs['rate_coins_per_usd'];
  } elseif (!empty($pairs['coins_per_usd']) && (float)$pairs['coins_per_usd'] > 0) {
    $rate = (float)$pairs['coins_per_usd'];
  }
} catch (Throwable $e) { /* ignore */ }

if ($rate === null) $err = 'سعر التحويل غير مضبوط من الأدمن.';

/* ---- تنفيذ التحويل مباشرة في DB ---- */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $rate !== null) {
  if (!csrf_ok($_POST['csrf'] ?? '')) {
    $err = 'رمز الأمان غير صالح.';
  } else {
    $usd = (float)($_POST['usd'] ?? 0);
    $usd = round($usd, 2);
    $coins = (int)floor($usd * $rate);

    if ($usd <= 0 || $coins <= 0) {
      $err = 'المبلغ غير صالح.';
    } else {
      try {
        $pdo->beginTransaction();

        // هل لدينا usd_balance_cents أم usd_balance؟
        $has_usd_cents = false; $has_usd_dec = false;
        try { $s=$pdo->query("SHOW COLUMNS FROM agents LIKE 'usd_balance_cents'"); $has_usd_cents = (bool)$s->fetch(); } catch(Throwable $e){}
        try { $s=$pdo->query("SHOW COLUMNS FROM agents LIKE 'usd_balance'");       $has_usd_dec   = (bool)$s->fetch(); } catch(Throwable $e){}

        // قفل صف الوكيل
        $sel_cols = "id, main_balance";
        if ($has_usd_cents) $sel_cols .= ", IFNULL(usd_balance_cents,0) AS usd_balance_cents";
        if ($has_usd_dec)   $sel_cols .= ", IFNULL(usd_balance,0) AS usd_balance";
        $st = $pdo->prepare("SELECT $sel_cols FROM agents WHERE id=? FOR UPDATE");
        $st->execute([(int)$agent['id']]);
        $row = $st->fetch(PDO::FETCH_ASSOC);
        if (!$row) throw new RuntimeException('agent_not_found');

        // خصم الدولار وإضافة الكوين
        if ($has_usd_cents) {
          $need_cents = (int)round($usd * 100);
          $cur_cents  = (int)$row['usd_balance_cents'];
          if ($cur_cents < $need_cents) throw new RuntimeException('insufficient_usd');
          $new_cents = $cur_cents - $need_cents;

          $u = $pdo->prepare("UPDATE agents SET usd_balance_cents=?, main_balance=main_balance+? WHERE id=? LIMIT 1");
          $u->execute([$new_cents, $coins, (int)$agent['id']]);
          $usd_after = $new_cents / 100.0;
        } elseif ($has_usd_dec) {
          $cur = (float)$row['usd_balance'];
          if ($cur + 1e-9 < $usd) throw new RuntimeException('insufficient_usd');
          $new = $cur - $usd;

          $u = $pdo->prepare("UPDATE agents SET usd_balance=?, main_balance=main_balance+? WHERE id=? LIMIT 1");
          $u->execute([$new, $coins, (int)$agent['id']]);
          $usd_after = $new;
        } else {
          throw new RuntimeException('usd_column_missing');
        }

        // تسجيل في دفتر الوكيل (إن وُجد currency سنسجل عمليتين)
        $has_currency = false;
        try { $s=$pdo->query("SHOW COLUMNS FROM agent_ledger LIKE 'currency'"); $has_currency = (bool)$s->fetch(); } catch(Throwable $e){}

        if ($has_currency) {
          $ins = $pdo->prepare("
            INSERT INTO agent_ledger (agent_id, direction, reason, amount, balance_after, currency)
            VALUES (?,?,?,?,?,?)
          ");
          // قيد خروج دولار
          $ins->execute([(int)$agent['id'], 'out', 'convert_usd_to_coins', $usd, $usd_after, 'usd']);
          // قيد دخول كوين
          $st2 = $pdo->prepare("SELECT main_balance FROM agents WHERE id=?");
          $st2->execute([(int)$agent['id']]);
          $coins_after = (int)$st2->fetchColumn();
          $ins->execute([(int)$agent['id'], 'in', 'convert_usd_to_coins', $coins, $coins_after, 'coins']);
        } else {
          // نسخة بدون currency — نسجل قيد الكوين فقط
          $st2 = $pdo->prepare("SELECT main_balance FROM agents WHERE id=?");
          $st2->execute([(int)$agent['id']]);
          $coins_after = (int)$st2->fetchColumn();

          $ins = $pdo->prepare("
            INSERT INTO agent_ledger (agent_id, direction, reason, amount, balance_after)
            VALUES (?,?,?,?,?)
          ");
          $ins->execute([(int)$agent['id'], 'in', 'convert_usd_to_coins', $coins, $coins_after]);
        }

        $pdo->commit();

        $msg   = 'تم تحويل $'.number_format($usd,2).' إلى '.(int)$coins.' كوين.';
        $after = ['usd'=>$usd_after, 'coins'=>$coins_after];

        // تحديث الأرقام الظاهرة بعد العملية
        $agent['main_balance'] = $coins_after;
        $agent['usd_display']  = number_format((float)$usd_after, 2, '.', '');
      } catch (Throwable $e) {
        if ($pdo->inTransaction()) { try { $pdo->rollBack(); } catch (Throwable $e2) {} }
        $code = $e->getMessage();
        if     ($code === 'insufficient_usd')      $err = 'رصيد الدولار غير كافٍ.';
        elseif ($code === 'usd_column_missing')     $err = 'لا يوجد عمود رصيد الدولار في جدول الوكلاء.';
        elseif ($code === 'agent_not_found')        $err = 'الوكيل غير موجود.';
        else                                         $err = 'تعذّر التنفيذ.';
      }
    }
  }
}
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width,initial-scale=1,maximum-scale=1">
  <title>تحويل $ ➜ كوين</title>
  <style>
    :root{--bg:#0b1220;--panel:#0f172a;--text:#e5e7eb;--sub:#94a3b8;--border:#243047;--primary:#3b82f6;--danger:#ef4444;--ok:#22c55e}
    *{box-sizing:border-box}
    body{margin:0;background:var(--bg);color:var(--text);font-family:system-ui,-apple-system,Segoe UI,Roboto}
    .wrap{max-width:900px;margin:0 auto;padding:16px}
    .card{background:var(--panel);border:1px solid var(--border);border-radius:14px;padding:16px}
    h1{margin:0 0 12px}
    .grid{display:grid;gap:16px;grid-template-columns:1fr 1fr}
    @media (max-width:860px){.grid{grid-template-columns:1fr}}
    .input label{display:block;font-size:13px;color:var(--sub);margin-bottom:6px}
    .input input{width:100%;padding:12px;border-radius:12px;border:1px solid var(--border);background:#0c1426;color:var(--text);font-size:16px}
    .btn{display:inline-block;padding:10px 14px;border-radius:10px;border:1px solid var(--border);background:var(--primary);color:#fff;text-decoration:none;cursor:pointer}
    .alert{border:1px solid var(--danger);padding:10px 12px;border-radius:12px;background:rgba(239,68,68,.08);margin:10px 0}
    .ok{border-color:var(--ok)!important;background:rgba(34,197,94,.08)!important}
    .chips{display:flex;gap:8px;flex-wrap:wrap;margin-bottom:12px}
    .chip{display:inline-flex;align-items:center;gap:8px;border:1px solid #2b3753;background:linear-gradient(180deg,#0f1a33,#0d1529);padding:8px 12px;border-radius:999px;font-weight:700;white-space:nowrap}
    .mono{font-family:ui-monospace,Menlo,Consolas,monospace}
    .muted{color:var(--sub)}
    .light-nav{display:flex;gap:8px;flex-wrap:wrap;margin-bottom:12px}
    .light-nav .btn{background:#1a2336;color:var(--text)}
  </style>
</head>
<body>

<div class="wrap">
  <div class="light-nav">
    <a class="btn" href="dashboard.php">الرئيسية</a>
    <a class="btn" href="charges.php">سجل الشحن</a>
  </div>

  <?php if ($err): ?><div class="alert"><?= e($err) ?></div><?php endif; ?>
  <?php if ($msg): ?><div class="alert ok"><?= e($msg) ?></div><?php endif; ?>

  <div class="chips">
    <div class="chip">💵 <span class="muted">USD:</span> <b class="mono"><?= e($agent['usd_display'] ?? '0.00') ?></b></div>
    <div class="chip">🪙 <span class="muted">Coins:</span> <b class="mono"><?= (int)($agent['main_balance'] ?? 0) ?></b></div>
    <?php if ($rate !== null): ?>
      <div class="chip">📈 <span class="muted">السعر:</span> <b class="mono"><?= rtrim(rtrim(number_format($rate, 6, '.', ''), '0'), '.') ?></b> كوين / $1</div>
    <?php endif; ?>
  </div>

  <div class="grid">
    <div class="card">
      <h1>تحويل دولار إلى كوين</h1>
      <form method="post" class="form" autocomplete="off">
        <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
        <div class="input">
          <label>المبلغ بالدولار ($)</label>
          <input type="number" name="usd" id="usd" min="0.01" step="0.01" required>
        </div>
        <div class="input">
          <label>سيتم إضافة (كوين)</label>
          <input type="text" id="coinsPreview" readonly>
        </div>
        <button class="btn" type="submit" <?= $rate===null?'disabled':''; ?>>تنفيذ التحويل</button>
        <?php if ($rate===null): ?><div class="muted" style="margin-top:8px">لا يمكن التحويل الآن لأن السعر غير مضبوط.</div><?php endif; ?>
      </form>
    </div>

    <div class="card">
      <h1>بعد العملية</h1>
      <?php if ($after): ?>
        <div class="muted">الأرصدة بعد آخر عملية ناجحة:</div>
        <div style="margin-top:8px">
          <div>💵 USD: <b class="mono"><?= number_format((float)$after['usd'],2) ?></b></div>
          <div>🪙 Coins: <b class="mono"><?= (int)$after['coins'] ?></b></div>
        </div>
      <?php else: ?>
        <div class="muted">لم يتم تنفيذ عملية في هذه الجلسة بعد.</div>
      <?php endif; ?>
    </div>
  </div>
</div>

<script>
(function(){
  const rate = <?= $rate!==null ? json_encode($rate) : 'null' ?>;
  const usd  = document.getElementById('usd');
  const out  = document.getElementById('coinsPreview');
  if (rate && usd && out) {
    function calc(){
      const u = parseFloat(usd.value || '0');
      const coins = Math.floor((u>0?u:0) * rate);
      out.value = coins.toString();
    }
    usd.addEventListener('input', calc);
  }
})();
</script>
</body>
</html>
