<?php
// /public_html/voicechat/agents/web/_web_common.php
declare(strict_types=1);

// جلسة للبوابة
if (session_status() !== PHP_SESSION_ACTIVE) session_start();

/* -------- تحميل config.php (db, respond, ...) -------- */
$loaded = false;
$try = [
  dirname(__DIR__) . '/config.php',
  dirname(__DIR__) . '/api/config.php',
  __DIR__ . '/../config.php',
  __DIR__ . '/../api/config.php',
];
foreach ($try as $p) {
  if (is_file($p)) { require_once $p; $loaded = true; break; }
}
if (!$loaded || !function_exists('db')) {
  http_response_code(500);
  echo "config.php not found";
  exit;
}

/* -------- Helpers عامّة -------- */
if (!function_exists('e')) {
  function e(string $s): string { return htmlspecialchars($s, ENT_QUOTES|ENT_SUBSTITUTE, 'UTF-8'); }
}
if (!isset($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
function csrf_token(): string { return $_SESSION['csrf']; }
function csrf_ok(?string $t): bool { return $t && hash_equals($_SESSION['csrf'], $t); }

function is_https(): bool {
  return (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off')
      || (isset($_SERVER['SERVER_PORT']) && $_SERVER['SERVER_PORT'] == 443);
}
function agent_cookie_name(): string { return 'AgentToken'; }
function agent_get_token(): ?string {
  $t = $_COOKIE[agent_cookie_name()] ?? null;
  return (is_string($t) && preg_match('~^[A-Fa-f0-9]{64}$~', $t)) ? $t : null;
}
function agent_set_token(string $token, int $ttlSeconds = 12*3600): void {
  setcookie(agent_cookie_name(), $token, [
    'expires'  => time() + $ttlSeconds,
    'path'     => '/',
    'secure'   => is_https(),
    'httponly' => true,
    'samesite' => 'Lax',
  ]);
}
function agent_clear_token(): void {
  setcookie(agent_cookie_name(), '', [
    'expires'  => time() - 3600,
    'path'     => '/',
    'secure'   => is_https(),
    'httponly' => true,
    'samesite' => 'Lax',
  ]);
}

/* -------- أدوات قاعدة البيانات -------- */
function col_exists(PDO $pdo, string $table, string $col): bool {
  try {
    $st = $pdo->prepare("SHOW COLUMNS FROM `$table` LIKE ?");
    $st->execute([$col]);
    return (bool)$st->fetchColumn();
  } catch (Throwable $e) { return false; }
}

/**
 * يجلب بيانات الوكيل انطلاقًا من التوكن مباشرة من DB
 * يرجع مصفوفة ['id','username','display_name','status','main_balance','usd_balance_cents','usd_balance','last_login_at']
 */
function agent_fetch_by_token(string $token): ?array {
  $pdo = db();

  $has_usd_cents = col_exists($pdo, 'agents', 'usd_balance_cents');
  $has_usd_dec   = col_exists($pdo, 'agents', 'usd_balance');

  $sel = "
    SELECT
      a.id, a.username, a.display_name, a.status, a.main_balance,
      " . ($has_usd_cents ? "a.usd_balance_cents" : "0 AS usd_balance_cents") . ",
      " . ($has_usd_dec   ? "a.usd_balance"      : "0 AS usd_balance") . ",
      a.last_login_at
    FROM agent_tokens t
    JOIN agents a ON a.id = t.agent_id
    WHERE t.token = ? AND (t.expires_at IS NULL OR t.expires_at > NOW())
    LIMIT 1
  ";
  $st = $pdo->prepare($sel);
  $st->execute([$token]);
  $row = $st->fetch(PDO::FETCH_ASSOC);
  if (!$row) return null;

  // تهيئة عرض الدولار
  $usd_cents = (int)($row['usd_balance_cents'] ?? 0);
  if ($usd_cents > 0) {
    $row['usd_display'] = number_format($usd_cents/100, 2, '.', '');
  } else {
    $row['usd_display'] = number_format((float)($row['usd_balance'] ?? 0), 2, '.', '');
  }
  return $row;
}

/** يجبر على تسجيل الدخول وإلا يُعيد التوجيه */
function require_agent_or_redirect(): array {
  $token = agent_get_token();
  if (!$token) { header('Location: login.php'); exit; }

  $agent = agent_fetch_by_token($token);
  if (!$agent) {
    agent_clear_token();
    header('Location: login.php'); exit;
  }
  return ['token'=>$token, 'agent'=>$agent];
}

/** إعادة تحميل بيانات الوكيل من DB */
function agent_reload(string $token): ?array {
  return agent_fetch_by_token($token);
}
