<?php
// admin/users.php
declare(strict_types=1);
require_once __DIR__ . '/inc/auth.php';
require_login();

/* ---------------- CSRF ---------------- */
if (!isset($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
function csrf_token(): string { return $_SESSION['csrf']; }
function csrf_ok(?string $t): bool { return $t && hash_equals($_SESSION['csrf'], $t); }

/* -------------- Helpers --------------- */
function codepoint_to_utf8(int $cp): string {
  if ($cp <= 0x7F)    return chr($cp);
  if ($cp <= 0x7FF)   return chr(0xC0|($cp>>6)).chr(0x80|($cp&0x3F));
  if ($cp <= 0xFFFF)  return chr(0xE0|($cp>>12)).chr(0x80|(($cp>>6)&0x3F)).chr(0x80|($cp&0x3F));
  if ($cp <= 0x10FFFF)return chr(0xF0|($cp>>18)).chr(0x80|(($cp>>12)&0x3F)).chr(0x80|(($cp>>6)&0x3F)).chr(0x80|($cp&0x3F));
  return '';
}
function flag_emoji(?string $cc): string {
  $cc = strtoupper(trim((string)$cc));
  if (!preg_match('/^[A-Z]{2}$/',$cc)) return '';
  $a = 0x1F1E6 + (ord($cc[0]) - 65);
  $b = 0x1F1E6 + (ord($cc[1]) - 65);
  return codepoint_to_utf8($a).codepoint_to_utf8($b);
}
function fmt_date(?string $s): string {
  if (!$s) return '';
  $t = strtotime($s); if ($t<=0) return e($s);
  return date('Y-m-d H:i', $t);
}
function gender_badge(?string $g): string {
  $g = strtolower(trim((string)$g));
  if ($g==='male')   return '<span class="badge gender-m" title="ذكر">ذكر</span>';
  if ($g==='female') return '<span class="badge gender-f" title="أنثى">أنثى</span>';
  if ($g!=='')       return '<span class="badge gender-o">'.e($g).'</span>';
  return '<span class="badge" title="غير محدد">غير محدد</span>';
}

/* -------------- Actions ---------------- */
$pdo = db();
$msg = null; $err = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $action = $_POST['action'] ?? '';
  $id     = (int)($_POST['id'] ?? 0);
  if (!csrf_ok($_POST['csrf'] ?? '')) {
    $err = 'رمز الأمان غير صالح. أعد المحاولة.';
  } elseif ($id <= 0) {
    $err = 'مُعرّف غير صالح.';
  } else {
    try {
      if ($action === 'clear_avatar') {
        $pdo->prepare("UPDATE users SET avatar_url=NULL WHERE id=? LIMIT 1")->execute([$id]);
        $msg = 'تم تفريغ الصورة الشخصية.';
      } elseif ($action === 'delete_user') {
        $pdo->prepare("DELETE FROM users WHERE id=? LIMIT 1")->execute([$id]);
        $msg = 'تم حذف المستخدم نهائيًا.';
      }
    } catch (Throwable $e) {
      $err = 'حدث خطأ أثناء تنفيذ العملية.';
    }
  }
}

/* -------------- Search & Page ---------- */
$q       = trim($_GET['q'] ?? '');
$page    = max(1, (int)($_GET['page'] ?? 1));
$perPage = 50;                       // ← 50 مستخدم في الصفحة
$offset  = ($page - 1) * $perPage;

$where  = "1=1"; $params = [];
if ($q !== '') {
  $where .= " AND (username LIKE ? OR email LIKE ? OR uid LIKE ? OR CAST(userid AS CHAR) LIKE ?)";
  $kw = "%$q%"; array_push($params,$kw,$kw,$kw,$kw);
}

$st = $pdo->prepare("SELECT COUNT(*) FROM users WHERE $where");
$st->execute($params);
$total = (int)$st->fetchColumn();
$pages = max(1, (int)ceil($total / $perPage));

$sql = "SELECT id, uid, userid, email, username, avatar_url, gender, country, country_code, birth_date, created_at, coins
        FROM users
        WHERE $where
        ORDER BY created_at ASC, id ASC   -- ← الأقدم أولاً (من أعلى لأسفل)
        LIMIT $perPage OFFSET $offset";
$st = $pdo->prepare($sql); $st->execute($params);
$rows = $st->fetchAll();

/* -------------- Render ------------------ */
$PAGE_TITLE = "المستخدمون";
require __DIR__ . '/inc/header.php';
require __DIR__ . '/inc/sidebar.php';
?>

<!-- ====== تنسيق خاص بالصفحة ====== -->
<style>
.users-page .table{
  width:100%;
  border-collapse:separate;
  border-spacing:0;
  background:var(--panel);
  border:1px solid var(--border);
  border-radius:12px;
  overflow:hidden;
  table-layout: fixed;             /* ← يضمن اتّساق الأعمدة */
}
.users-page .table td, .users-page .table th{
  padding:10px 12px;
  border-bottom:1px solid var(--border);
  vertical-align:middle;
  text-align:start;
}
.users-page .table thead th{
  background:#0b1220;color:var(--text);font-weight:700;
  position:sticky;top:0;z-index:3;
  box-shadow:inset 0 -1px 0 var(--border);
}
.users-page .table tbody tr:nth-child(even) td{background:rgba(255,255,255,.015)}
.users-page .table tbody tr:hover td{background:rgba(59,130,246,.08)}

.users-page .td-num{text-align:center;opacity:.9}
.users-page .td-center{text-align:center}

.users-page .ellipsis{max-width:220px;display:inline-block;overflow:hidden;white-space:nowrap;text-overflow:ellipsis;vertical-align:bottom}
.users-page .ellipsis.small{max-width:140px}

.users-page .user-cell{display:flex;align-items:center;gap:10px;min-width:180px}
.users-page .user-avatar{width:40px;height:40px;border-radius:999px;object-fit:cover;border:1px solid var(--border)}

.users-page .badge{display:inline-block;padding:2px 8px;border-radius:999px;border:1px solid var(--border);font-size:12px;opacity:.95}
.users-page .badge.gender-m{border-color:#3b82f6}
.users-page .badge.gender-f{border-color:#ec4899}
.users-page .badge.gender-o{border-color:#a78bfa}
.users-page .badge.coin{border-color:#22c55e}

.users-page .actions{display:flex;gap:8px;flex-wrap:nowrap;align-items:center;justify-content:flex-start}
.users-page .actions form{display:inline-block;margin:0}     /* ← يمنع كسر الصف */
.users-page .actions .btn{white-space:nowrap}

.users-page .btn.sm{padding:6px 10px;font-size:14px}

/* أعمدة بعرض ثابت */
.users-page .col-id{width:60px}.users-page .col-av{width:64px}.users-page .col-name{width:220px}.users-page .col-email{width:240px}
.users-page .col-uid{width:180px}.users-page .col-userid{width:110px}.users-page .col-gender{width:90px}
.users-page .col-country{width:160px}.users-page .col-birth{width:120px}.users-page .col-coins{width:90px}
.users-page .col-created{width:160px}.users-page .col-actions{width:280px}   /* ← زوّدنا عرض الإجراءات */

/* غلاف الجدول لتمرير داخلي وهيدر ثابت */
.users-page .table-wrap{max-height:70vh;overflow:auto;position:relative}
</style>

<div class="users-page">
  <div class="card" style="margin-bottom:16px;">
    <form method="get" class="form" style="display:flex; gap:12px; align-items:flex-end; flex-wrap: wrap;">
      <div class="input" style="min-width:260px; flex:1;">
        <label>ابحث بالاسم/الإيميل/UID/USERID</label>
        <input type="text" name="q" value="<?= e($q) ?>" placeholder="مثال: campa أو user@domain.com">
      </div>
      <button class="btn sm" type="submit">بحث</button>
      <?php if ($q!==''): ?>
        <a class="btn sm secondary" href="users.php">مسح البحث</a>
      <?php endif; ?>
      <div style="margin-inline-start:auto; opacity:.9;">إجمالي: <b><?= $total ?></b> نتيجة</div>
    </form>
    <?php if ($msg): ?><div class="alert" style="margin-top:12px; border-color:#22c55e;"><?= e($msg) ?></div><?php endif; ?>
    <?php if ($err): ?><div class="alert" style="margin-top:12px;"><?= e($err) ?></div><?php endif; ?>
  </div>

  <div class="card">
    <div class="table-wrap">
      <table class="table">
        <colgroup>
          <col class="col-id"><col class="col-av"><col class="col-name"><col class="col-email">
          <col class="col-uid"><col class="col-userid"><col class="col-gender">
          <col class="col-country"><col class="col-birth"><col class="col-coins">
          <col class="col-created"><col class="col-actions">
        </colgroup>
        <thead>
          <tr>
            <th class="td-center">#</th>
            <th class="td-center">الصورة</th>
            <th>الاسم</th>
            <th>الإيميل</th>
            <th>UID</th>
            <th>UserID</th>
            <th>الجنس</th>
            <th>الدولة</th>
            <th>الميلاد</th>
            <th>الكوينز</th>
            <th>أُنشيء</th>
            <th class="col-actions">إجراءات</th>
          </tr>
        </thead>
        <tbody>
          <?php foreach ($rows as $r): ?>
            <tr>
              <td class="td-num"><?= (int)$r['id'] ?></td>
              <td class="td-center">
                <?php if (!empty($r['avatar_url'])): ?>
                  <img src="<?= e($r['avatar_url']) ?>" alt="" class="user-avatar">
                <?php else: ?>
                  <span class="badge">بدون</span>
                <?php endif; ?>
              </td>
              <td>
                <div class="user-cell">
                  <div>
                    <div style="font-weight:700" title="<?= e($r['username'] ?? '') ?>">
                      <span class="ellipsis"><?= e($r['username'] ?? '') ?></span>
                    </div>
                    <?php if (!empty($r['country']) || !empty($r['country_code'])): ?>
                      <div style="opacity:.8;font-size:12px">
                        <?= e($r['country'] ?? '') ?>
                        <?php if (!empty($r['country_code'])): ?>
                          &nbsp;<?= flag_emoji($r['country_code']) ?> (<?= e($r['country_code']) ?>)
                        <?php endif; ?>
                      </div>
                    <?php endif; ?>
                  </div>
                </div>
              </td>
              <td title="<?= e($r['email'] ?? '') ?>"><span class="ellipsis"><?= e($r['email'] ?? '') ?></span></td>
              <td title="<?= e($r['uid'] ?? '') ?>"><span class="ellipsis small" style="font-family:monospace;"><?= e($r['uid'] ?? '') ?></span></td>
              <td><span class="ellipsis small"><?= e((string)$r['userid']) ?></span></td>
              <td><?= gender_badge($r['gender'] ?? '') ?></td>
              <td>
                <?php if (!empty($r['country'])): ?>
                  <span class="ellipsis small" title="<?= e($r['country']) ?>"><?= e($r['country']) ?></span>
                <?php else: ?>
                  <span class="badge">—</span>
                <?php endif; ?>
              </td>
              <td><?= e($r['birth_date'] ?? '') ?></td>
              <td><span class="badge coin" title="Coins"><?= (int)$r['coins'] ?></span></td>
              <td title="<?= e($r['created_at'] ?? '') ?>"><?= fmt_date($r['created_at'] ?? '') ?></td>
              <td class="actions">
                <a class="btn sm secondary" href="users_edit.php?id=<?= (int)$r['id'] ?>">تعديل</a>

                <form method="post" onsubmit="return confirm('تفريغ صورة المستخدم؟');">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                  <input type="hidden" name="action" value="clear_avatar">
                  <button class="btn sm" type="submit" title="تفريغ الصورة">تفريغ الصورة</button>
                </form>

                <form method="post" onsubmit="return confirm('حذف نهائي؟ لا يمكن التراجع.');">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                  <input type="hidden" name="action" value="delete_user">
                  <button class="btn sm" type="submit" style="background:var(--danger);">حذف</button>
                </form>
              </td>
            </tr>
          <?php endforeach; ?>
          <?php if (!$rows): ?>
            <tr><td colspan="12" style="text-align:center; opacity:.8;">لا توجد بيانات</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>

    <?php if ($pages > 1): ?>
      <div style="display:flex; gap:8px; margin-top:12px; flex-wrap: wrap;">
        <?php for($p=1;$p<=$pages;$p++):
          $url = 'users.php?page='.$p.($q!=='' ? '&q='.urlencode($q) : '');
          $is = $p===$page; ?>
          <a href="<?= e($url) ?>" class="btn sm <?= $is ? '' : 'secondary' ?>"<?= $is?' style="pointer-events:none; opacity:.8;"':''; ?>><?= $p ?></a>
        <?php endfor; ?>
      </div>
    <?php endif; ?>
  </div>
</div>

<?php require __DIR__ . '/inc/footer.php'; ?>
