<?php
// admin/store_categories.php
declare(strict_types=1);

$CFG_PATH = __DIR__ . '/config.php';
$INC_DIR  = __DIR__ . '/inc';
require_once $CFG_PATH;
require_once $INC_DIR . '/auth.php';
require_login();

/* CSRF */
if (!isset($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
function csrf_token(): string { return $_SESSION['csrf']; }
function csrf_ok(?string $t): bool { return $t && hash_equals($_SESSION['csrf'], $t); }

/* Helpers */
if (!function_exists('e')) { function e(string $s): string { return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); } }
function fmt_dt(?string $s): string { if(!$s) return ''; $t=strtotime($s); return $t>0?date('Y-m-d H:i',$t):e($s); }
function to_en_digits(string $s): string {
  $ar=['٠','١','٢','٣','٤','٥','٦','٧','٨','٩']; $fa=['۰','۱','۲','۳','۴','۵','۶','۷','۸','۹']; $en=['0','1','2','3','4','5','6','7','8','9'];
  return str_replace($fa,$en,str_replace($ar,$en,$s));
}
function to_int($v): int { $v=to_en_digits(trim((string)$v)); if($v==='')return 0; $v=preg_replace('/[^0-9\-]/','',$v); return (int)$v; }
function make_slug(string $s): string {
  $s = to_en_digits($s); $s = strtolower($s);
  $s = preg_replace('/\s+/', '-', $s);
  $s = preg_replace('/[^a-z0-9_-]/', '', $s);
  return trim($s,'-_');
}
function col_exists(PDO $pdo, string $table, string $col): bool {
  $st=$pdo->prepare("SHOW COLUMNS FROM `$table` LIKE ?"); $st->execute([$col]);
  return (bool)$st->fetchColumn();
}

/* DB */
$pdo = db();
$msg=null; $err=null; $dbg = isset($_GET['dbg']);

/* ===== Actions: toggle / delete ===== */
if ($_SERVER['REQUEST_METHOD']==='POST') {
  if (!csrf_ok($_POST['csrf'] ?? '')) {
    $err='رمز الأمان غير صالح.';
  } else {
    $act = $_POST['action'] ?? '';
    $id  = (int)($_POST['id'] ?? 0);
    try {
      if ($act==='toggle' && $id>0) {
        $pdo->prepare("UPDATE store_categories SET is_active = IF(is_active=1,0,1) WHERE id=? LIMIT 1")->execute([$id]);
        $msg = 'تم تبديل الحالة.';
      } elseif ($act==='delete' && $id>0) {
        // حماية: لا نحذف إن كان هناك عناصر مرتبطة
        $st = $pdo->prepare("SELECT slug FROM store_categories WHERE id=? LIMIT 1");
        $st->execute([$id]);
        $slug = (string)($st->fetchColumn() ?: '');
        $hasCatId = col_exists($pdo,'store_items','category_id');
        $hasCat   = col_exists($pdo,'store_items','category');

        $linked = 0;
        if ($hasCatId) {
          $st=$pdo->prepare("SELECT COUNT(*) FROM store_items WHERE category_id=?"); $st->execute([$id]); $linked += (int)$st->fetchColumn();
        }
        if ($hasCat && $slug!=='') {
          $st=$pdo->prepare("SELECT COUNT(*) FROM store_items WHERE category=?"); $st->execute([$slug]); $linked += (int)$st->fetchColumn();
        }

        if ($linked>0) {
          $err = 'لا يمكن حذف الصنف لوجود عناصر متجر مرتبطة به.';
        } else {
          $pdo->prepare("DELETE FROM store_categories WHERE id=? LIMIT 1")->execute([$id]);
          $msg='تم حذف الصنف.';
        }
      }
    } catch (Throwable $e) {
      $err = 'تعذّر تنفيذ العملية.' . ($dbg ? ' ('.$e->getMessage().')' : '');
    }
  }
}

/* ===== Filters ===== */
$q       = trim($_GET['q'] ?? '');
$active  = $_GET['active'] ?? '';  // '', '1', '0'
$page    = max(1,(int)($_GET['page'] ?? 1));
$perPage = 50;
$offset  = ($page-1)*$perPage;

$where = "1=1"; $params=[];
if ($q!=='') {
  $where .= " AND (name LIKE ? OR slug LIKE ?)";
  $kw = "%$q%"; array_push($params,$kw,$kw);
}
if ($active==='1' || $active==='0') { $where.=" AND is_active=?"; $params[]=$active; }

/* count */
$total=0;
try{ $st=$pdo->prepare("SELECT COUNT(*) FROM store_categories WHERE $where"); $st->execute($params); $total=(int)$st->fetchColumn(); }
catch(Throwable $e){ $err='تعذّر حساب الإجمالي.'; }
$pages=max(1,(int)ceil($total/$perPage));

/* fetch */
$rows=[];
try{
  $sql="SELECT id,slug,name,is_active,sort_order,meta_json,created_at
        FROM store_categories
        WHERE $where
        ORDER BY sort_order ASC, id DESC
        LIMIT $perPage OFFSET $offset";
  $st=$pdo->prepare($sql); $st->execute($params);
  $rows=$st->fetchAll(PDO::FETCH_ASSOC) ?: [];
}catch(Throwable $e){ $err='تعذّر جلب البيانات.' . ($dbg ? ' ('.$e->getMessage().')' : ''); }

/* precompute item counts (لطيف لكنه بعدد قليل من الاستعلامات) */
$hasCatId = col_exists($pdo,'store_items','category_id');
$hasCat   = col_exists($pdo,'store_items','category');
foreach ($rows as &$r) {
  $cnt=0;
  if ($hasCatId) { $st=$pdo->prepare("SELECT COUNT(*) FROM store_items WHERE category_id=?"); $st->execute([(int)$r['id']]); $cnt+=(int)$st->fetchColumn(); }
  if ($hasCat)   { $st=$pdo->prepare("SELECT COUNT(*) FROM store_items WHERE category=?");   $st->execute([(string)$r['slug']]); $cnt+=(int)$st->fetchColumn(); }
  $r['_items_count']=$cnt;
}
unset($r);

/* ===== Render ===== */
$PAGE_TITLE = "أصناف المتجر";
require $INC_DIR.'/header.php';
require $INC_DIR.'/sidebar.php';
?>
<style>
.cat-page .table{width:100%;border-collapse:separate;border-spacing:0;background:var(--panel);border:1px solid var(--border);border-radius:12px;overflow:hidden;table-layout:fixed}
.cat-page .table td,.cat-page .table th{padding:10px 12px;border-bottom:1px solid var(--border);vertical-align:middle;text-align:start}
.cat-page .table thead th{background:#0b1220;color:var(--text);font-weight:700;position:sticky;top:0;z-index:3;box-shadow:inset 0 -1px 0 var(--border)}
.cat-page .table tbody tr:nth-child(even) td{background:rgba(255,255,255,.015)}
.cat-page .table tbody tr:hover td{background:rgba(59,130,246,.08)}
.cat-page .table-wrap{max-height:70vh;overflow:auto;position:relative}
.cat-page .td-center{text-align:center}
.cat-page .badge{display:inline-block;padding:2px 8px;border-radius:999px;border:1px solid var(--border);font-size:12px}
.cat-page .ellipsis{max-width:300px;display:inline-block;overflow:hidden;white-space:nowrap;text-overflow:ellipsis}
.cat-page .col-id{width:70px}.cat-page .col-slug{width:180px}.cat-page .col-name{width:260px}.cat-page .col-items{width:120px}.cat-page .col-active{width:100px}.cat-page .col-sort{width:100px}.cat-page .col-created{width:170px}.cat-page .col-actions{width:220px}
</style>

<div class="cat-page">
  <div class="card" style="margin-bottom:16px;">
    <form method="get" class="form" style="display:flex; gap:12px; align-items:flex-end; flex-wrap:wrap;">
      <div class="input" style="min-width:260px; flex:1;">
        <label>بحث بالاسم/SLUG</label>
        <input type="text" name="q" value="<?= e($q) ?>" placeholder="frame / vip / خلفية">
      </div>
      <div class="input">
        <label>الحالة</label>
        <select name="active">
          <option value="">الكل</option>
          <option value="1"<?= $active==='1'?' selected':''; ?>>مفعّل</option>
          <option value="0"<?= $active==='0'?' selected':''; ?>>معطّل</option>
        </select>
      </div>
      <button class="btn sm" type="submit">بحث</button>
      <?php if ($q!=='' || $active!==''): ?><a class="btn sm secondary" href="store_categories.php">مسح الفلاتر</a><?php endif; ?>
      <a class="btn sm" style="margin-inline-start:auto;" href="store_categories_edit.php">+ إضافة صنف</a>
      <div style="opacity:.9;">إجمالي: <b><?= $total ?></b></div>
    </form>
    <?php if ($msg): ?><div class="alert" style="margin-top:12px;border-color:#22c55e;"><?= e($msg) ?></div><?php endif; ?>
    <?php if ($err): ?><div class="alert" style="margin-top:12px;"><?= e($err) ?></div><?php endif; ?>
  </div>

  <div class="card">
    <div class="table-wrap">
      <table class="table">
        <colgroup>
          <col class="col-id"><col class="col-slug"><col class="col-name"><col class="col-items"><col class="col-active"><col class="col-sort"><col class="col-created"><col class="col-actions">
        </colgroup>
        <thead>
          <tr>
            <th class="td-center">#</th>
            <th>SLUG</th>
            <th>الاسم</th>
            <th>عناصر مرتبطة</th>
            <th>الحالة</th>
            <th>الترتيب</th>
            <th>أُنشئ</th>
            <th>إجراءات</th>
          </tr>
        </thead>
        <tbody>
          <?php foreach($rows as $r): ?>
            <tr>
              <td class="td-center"><?= (int)$r['id'] ?></td>
              <td><span class="badge"><?= e($r['slug']) ?></span></td>
              <td><span class="ellipsis" title="<?= e($r['name']) ?>"><?= e($r['name']) ?></span></td>
              <td class="td-center"><?= (int)($r['_items_count'] ?? 0) ?></td>
              <td class="td-center"><?= ((int)$r['is_active']===1?'✅ مفعل':'⛔ معطّل') ?></td>
              <td class="td-center"><?= (int)$r['sort_order'] ?></td>
              <td><?= fmt_dt($r['created_at'] ?? '') ?></td>
              <td style="display:flex; gap:8px; flex-wrap:nowrap;">
                <a class="btn sm secondary" href="store_categories_edit.php?id=<?= (int)$r['id'] ?>">تعديل</a>
                <form method="post">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                  <input type="hidden" name="action" value="toggle">
                  <button class="btn sm" type="submit"><?= ((int)$r['is_active']===1?'تعطيل':'تفعيل') ?></button>
                </form>
                <form method="post" onsubmit="return confirm('حذف نهائي للصنف؟ سيتم التحقق من عدم وجود عناصر مرتبطة.');">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                  <input type="hidden" name="action" value="delete">
                  <button class="btn sm" type="submit" style="background:var(--danger);">حذف</button>
                </form>
              </td>
            </tr>
          <?php endforeach; ?>
          <?php if (!$rows): ?><tr><td colspan="8" style="text-align:center;opacity:.8;">لا توجد بيانات</td></tr><?php endif; ?>
        </tbody>
      </table>
    </div>

    <?php if ($pages>1): ?>
      <div style="display:flex; gap:8px; margin-top:12px; flex-wrap:wrap;">
        <?php for($p=1;$p<=$pages;$p++):
          $url='store_categories.php?page='.$p
            .($q!==''? '&q='.urlencode($q):'')
            .($active!==''? '&active='.urlencode($active):'');
          $is=$p===$page; ?>
          <a href="<?= e($url) ?>" class="btn sm <?= $is?'':'secondary' ?>"<?= $is?' style="pointer-events:none;opacity:.8;"':''; ?>><?= $p ?></a>
        <?php endfor; ?>
      </div>
    <?php endif; ?>
  </div>
</div>

<?php require $INC_DIR.'/footer.php'; ?>
