<?php
// admin/staff.php
declare(strict_types=1);
require_once __DIR__ . '/inc/auth.php';
require_login();

/* ===== CSRF ===== */
if (!isset($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
function csrf_token(): string { return $_SESSION['csrf']; }
function csrf_ok(?string $t): bool { return $t && hash_equals($_SESSION['csrf'], $t); }

/* ===== Helpers ===== */
if (!function_exists('e')) {
  function e(string $s): string { return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }
}
function fmt_dt(?string $s): string {
  if (!$s) return '';
  $t = strtotime((string)$s); return $t>0 ? date('Y-m-d H:i', $t) : '';
}

/* ===== DB ===== */
$pdo = db();
$msg = null; $err = null;

/* ===== Actions ===== */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $action = $_POST['action'] ?? '';
  $id     = (int)($_POST['id'] ?? 0);

  if (!csrf_ok($_POST['csrf'] ?? '')) {
    $err = 'رمز الأمان غير صالح.';
  } elseif ($id <= 0) {
    $err = 'مُعرّف غير صالح.';
  } else {
    try {
      if ($action === 'toggle_active') {
        $pdo->prepare("UPDATE `admincp` SET `is_active`=1-`is_active`, `updated_at`=NOW() WHERE `id`=? LIMIT 1")->execute([$id]);
        $msg = 'تم تبديل الحالة.';
      } elseif ($action === 'delete_staff') {
        $pdo->prepare("DELETE FROM `admincp` WHERE `id`=? LIMIT 1")->execute([$id]);
        $msg = 'تم حذف الموظف.';
      } elseif ($action === 'change_role') {
        $role = trim((string)($_POST['role'] ?? ''));
        if ($role==='') { $err = 'اختر صلاحية صحيحة.'; }
        else {
          $pdo->prepare("UPDATE `admincp` SET `role`=?, `updated_at`=NOW() WHERE `id`=? LIMIT 1")->execute([$role,$id]);
          $msg = 'تم تحديث الصلاحية.';
        }
      }
    } catch (Throwable $e) {
      $err = 'حدث خطأ أثناء تنفيذ العملية.';
    }
  }
}

/* ===== Filters / Paging ===== */
$q       = trim($_GET['q'] ?? '');
$roleF   = trim($_GET['role'] ?? ''); // '', 'superadmin','admin','editor'...
$activeF = trim($_GET['active'] ?? ''); // '', '1','0'
$page    = max(1,(int)($_GET['page'] ?? 1));
$per     = 50;
$off     = ($page-1)*$per;

$where = "1=1"; $prm = [];
if ($q!=='') {
  $where .= " AND (username LIKE ? OR email LIKE ? OR full_name LIKE ?)";
  $kw = "%$q%"; array_push($prm,$kw,$kw,$kw);
}
if ($roleF!=='')   { $where .= " AND role = ?"; $prm[] = $roleF; }
if ($activeF!=='') { $where .= " AND is_active = ?"; $prm[] = (int)$activeF; }

/* count */
$st = $pdo->prepare("SELECT COUNT(*) FROM `admincp` WHERE $where");
$st->execute($prm); $total=(int)$st->fetchColumn();
$pages = max(1,(int)ceil($total/$per));

/* page */
$sql = "SELECT id,username,email,full_name,role,is_active,last_login_at,created_at,updated_at
        FROM `admincp`
        WHERE $where
        ORDER BY created_at DESC, id DESC
        LIMIT $per OFFSET $off";
$st = $pdo->prepare($sql); $st->execute($prm);
$rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

/* ===== View ===== */
$PAGE_TITLE = "قائمة الموظفين";
require __DIR__ . '/inc/header.php';
require __DIR__ . '/inc/sidebar.php';
?>
<style>
.staff .table{width:100%;border-collapse:separate;border-spacing:0;background:var(--panel);border:1px solid var(--border);border-radius:12px;overflow:hidden;table-layout:fixed}
.staff .table td,.staff .table th{padding:10px 12px;border-bottom:1px solid var(--border);vertical-align:middle;text-align:start}
.staff .table thead th{background:#0b1220;color:var(--text);font-weight:700;position:sticky;top:0;z-index:3;box-shadow:inset 0 -1px 0 var(--border)}
.staff .table tbody tr:nth-child(even) td{background:rgba(255,255,255,.015)}
.staff .table tbody tr:hover td{background:rgba(59,130,246,.08)}
.staff .table-wrap{max-height:70vh;overflow:auto;position:relative}
.staff .td-center{text-align:center}
.staff .badge{display:inline-block;padding:2px 8px;border-radius:999px;border:1px solid var(--border);font-size:12px}
.staff .badge.ok{border-color:#22c55e}
.staff .ellipsis{max-width:260px;display:inline-block;overflow:hidden;white-space:nowrap;text-overflow:ellipsis}
.staff .col-id{width:60px}.staff .col-name{width:220px}.staff .col-email{width:260px}.staff .col-role{width:160px}.staff .col-active{width:100px}
.staff .col-login{width:170px}.staff .col-created{width:170px}.staff .col-actions{width:280px}
.staff .btn.sm{padding:6px 10px;font-size:14px}
</style>

<div class="staff">
  <div class="card" style="margin-bottom:16px;">
    <form method="get" class="form" style="display:flex;gap:12px;align-items:flex-end;flex-wrap:wrap;">
      <div class="input" style="min-width:260px;flex:1;">
        <label>ابحث بالاسم/اسم المستخدم/الإيميل</label>
        <input type="text" name="q" value="<?= e($q) ?>" placeholder="مثال: admin أو user@mail.com">
      </div>
      <div class="input">
        <label>الصلاحية</label>
        <select name="role" style="padding:10px 12px;border-radius:10px;border:1px solid var(--border);background:#0b1220;color:var(--text);">
          <option value="" <?= $roleF===''?'selected':'' ?>>الكل</option>
          <option value="superadmin" <?= $roleF==='superadmin'?'selected':'' ?>>مدير عام</option>
          <option value="admin"      <?= $roleF==='admin'?'selected':'' ?>>مدير</option>
          <option value="editor"     <?= $roleF==='editor'?'selected':'' ?>>محرر</option>
          <option value="viewer"     <?= $roleF==='viewer'?'selected':'' ?>>مشاهِد</option>
        </select>
      </div>
      <div class="input">
        <label>الحالة</label>
        <select name="active" style="padding:10px 12px;border-radius:10px;border:1px solid var(--border);background:#0b1220;color:var(--text);">
          <option value=""  <?= $activeF===''?'selected':'' ?>>الكل</option>
          <option value="1" <?= $activeF==='1'?'selected':'' ?>>مفعل</option>
          <option value="0" <?= $activeF==='0'?'selected':'' ?>>معطل</option>
        </select>
      </div>

      <button class="btn sm" type="submit">بحث</button>
      <?php if ($q!=='' || $roleF!=='' || $activeF!==''): ?>
        <a class="btn sm secondary" href="staff.php">مسح الفلاتر</a>
      <?php endif; ?>
      <div style="margin-inline-start:auto;opacity:.9;">إجمالي: <b><?= $total ?></b> موظف</div>
      <a class="btn sm" href="staff_add.php">+ إضافة موظف</a>
    </form>

    <?php if ($msg): ?><div class="alert" style="margin-top:12px;border-color:#22c55e;"><?= e($msg) ?></div><?php endif; ?>
    <?php if ($err): ?><div class="alert" style="margin-top:12px;"><?= e($err) ?></div><?php endif; ?>
  </div>

  <div class="card">
    <div class="table-wrap">
      <table class="table">
        <colgroup>
          <col class="col-id"><col class="col-name"><col class="col-email"><col class="col-role"><col class="col-active">
          <col class="col-login"><col class="col-created"><col class="col-actions">
        </colgroup>
        <thead>
          <tr>
            <th class="td-center">#</th>
            <th>الاسم</th>
            <th>الإيميل</th>
            <th>الصلاحية</th>
            <th>الحالة</th>
            <th>آخر دخول</th>
            <th>أُنشيء</th>
            <th>إجراءات</th>
          </tr>
        </thead>
        <tbody>
          <?php foreach ($rows as $r): ?>
            <tr>
              <td class="td-center"><?= (int)$r['id'] ?></td>
              <td title="<?= e($r['full_name'] ?? '') ?>">
                <div style="font-weight:700" class="ellipsis"><?= e($r['full_name'] ?? '') ?></div>
                <div style="opacity:.7" class="ellipsis"><?= e($r['username'] ?? '') ?></div>
              </td>
              <td title="<?= e($r['email'] ?? '') ?>"><span class="ellipsis"><?= e($r['email'] ?? '') ?></span></td>
              <td>
                <form method="post" style="display:flex;gap:6px;align-items:center;">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id"   value="<?= (int)$r['id'] ?>">
                  <input type="hidden" name="action" value="change_role">
                  <select name="role" style="padding:6px 8px;border-radius:10px;border:1px solid var(--border);background:#0b1220;color:var(--text);">
                    <?php foreach (['superadmin'=>'مدير عام','admin'=>'مدير','editor'=>'محرر','viewer'=>'مشاهِد'] as $k=>$v): ?>
                      <option value="<?= e($k) ?>" <?= ($r['role']===$k)?'selected':'' ?>><?= e($v) ?></option>
                    <?php endforeach; ?>
                  </select>
                  <button class="btn sm" type="submit">حفظ</button>
                </form>
              </td>
              <td class="td-center">
                <?= ((int)$r['is_active']===1)?'<span class="badge ok">مفعل</span>':'<span class="badge">معطل</span>' ?>
              </td>
              <td><?= fmt_dt($r['last_login_at'] ?? '') ?></td>
              <td><?= fmt_dt($r['created_at'] ?? '') ?></td>
              <td style="display:flex;gap:8px;flex-wrap:wrap;align-items:center;">
                <form method="post" onsubmit="return confirm('تبديل حالة التفعيل؟');">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                  <input type="hidden" name="action" value="toggle_active">
                  <button class="btn sm" type="submit"><?= ((int)$r['is_active']===1)?'تعطيل':'تفعيل' ?></button>
                </form>
                <form method="post" onsubmit="return confirm('حذف نهائي لهذا الموظف؟');">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                  <input type="hidden" name="action" value="delete_staff">
                  <button class="btn sm" type="submit" style="background:var(--danger);">حذف</button>
                </form>
              </td>
            </tr>
          <?php endforeach; ?>
          <?php if (!$rows): ?><tr><td colspan="8" style="text-align:center;opacity:.8;">لا توجد بيانات</td></tr><?php endif; ?>
        </tbody>
      </table>
    </div>

    <?php if ($pages>1): ?>
      <div style="display:flex;gap:8px;margin-top:12px;flex-wrap:wrap;">
        <?php for($p=1;$p<=$pages;$p++):
          $url='staff.php?page='.$p
            .($q!==''?'&q='.urlencode($q):'')
            .($roleF!==''?'&role='.urlencode($roleF):'')
            .($activeF!==''?'&active='.$activeF:'');
          $is=$p===$page; ?>
          <a href="<?= e($url) ?>" class="btn sm <?= $is?'':'secondary' ?>"<?= $is?' style="pointer-events:none;opacity:.8;"':''; ?>><?= $p ?></a>
        <?php endfor; ?>
      </div>
    <?php endif; ?>
  </div>
</div>
<?php require __DIR__ . '/inc/footer.php'; ?>
