<?php
// admin/post_reports.php
declare(strict_types=1);
require_once __DIR__ . '/inc/auth.php';
require_login();

/* ---------------- CSRF ---------------- */
if (!isset($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
function csrf_token(): string { return $_SESSION['csrf']; }
function csrf_ok(?string $t): bool { return $t && hash_equals($_SESSION['csrf'], $t); }

/* -------------- Helpers --------------- */
if (!function_exists('e')) {
  function e(string $s): string { return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }
}
function fmt_date(?string $s): string {
  if (!$s) return '';
  $t = strtotime($s); if ($t<=0) return e($s);
  return date('Y-m-d H:i', $t);
}
function badge(string $text, string $class=''): string {
  $cls = 'badge'.($class ? ' '.$class : '');
  return '<span class="'.$cls.'">'.e($text).'</span>';
}
function status_label(string $st): string {
  $map = [
    'new'         => 'جديد',
    'in_review'   => 'قيد المراجعة',
    'resolved'    => 'تم الإجراء',
    'rejected'    => 'مرفوض',
  ];
  return $map[$st] ?? $st;
}
function status_badge(string $st): string {
  $class = match($st){
    'new' => 'st-new',
    'in_review' => 'st-review',
    'resolved' => 'st-ok',
    'rejected' => 'st-bad',
    default => '',
  };
  return badge(status_label($st), $class);
}
function reason_label(?string $code): string {
  $code = (string)$code;
  $map = [
    'abuse'       => 'محتوى مسيء',
    'harassment'  => 'تنمّر/تحرّش',
    'sexual'      => 'محتوى جنسي غير لائق',
    'spam'        => 'سبام/ترويج',
    'impersonate' => 'انتحال شخصية',
  ];
  return $map[$code] ?? $code;
}

/* -------------- Actions ---------------- */
$pdo = db();
$msg = null; $err = null;

$ALLOWED_STATUSES = ['new','in_review','resolved','rejected'];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $action = $_POST['action'] ?? '';
  $id     = (int)($_POST['id'] ?? 0);

  if (!csrf_ok($_POST['csrf'] ?? '')) {
    $err = 'رمز الأمان غير صالح. أعد المحاولة.';
  } elseif ($id <= 0) {
    $err = 'مُعرّف غير صالح.';
  } else {
    try {
      if ($action === 'set_status') {
        $st = trim((string)($_POST['status'] ?? ''));
        if (!in_array($st, $ALLOWED_STATUSES, true)) throw new RuntimeException('حالة غير مسموح بها');
        $stmt = $pdo->prepare("UPDATE post_reports SET status=?, updated_at=NOW() WHERE id=? LIMIT 1");
        $stmt->execute([$st, $id]);
        $msg = 'تم تحديث الحالة.';
      } elseif ($action === 'delete_report') {
        $pdo->prepare("DELETE FROM post_reports WHERE id=? LIMIT 1")->execute([$id]);
        $msg = 'تم حذف البلاغ نهائيًا.';
      }
    } catch (Throwable $e) {
      $err = 'حدث خطأ أثناء تنفيذ العملية.';
    }
  }
}

/* -------------- Filters & Page --------- */
$q        = trim($_GET['q'] ?? '');                       // بحث
$f_status = trim($_GET['status'] ?? '');                  // فلترة بالحالة
$f_reason = trim($_GET['reason'] ?? '');                  // فلترة بالسبب (الكود)
$page     = max(1, (int)($_GET['page'] ?? 1));
$perPage  = 50;
$offset   = ($page - 1) * $perPage;

$where  = "1=1"; $params = [];

if ($q !== '') {
  // بحث في عدة أعمدة
  $where .= " AND (
      CAST(id AS CHAR) LIKE ?
      OR CAST(post_id AS CHAR) LIKE ?
      OR CAST(post_user_id AS CHAR) LIKE ?
      OR CAST(reporter_user_id AS CHAR) LIKE ?
      OR reason_text LIKE ?
      OR ip LIKE ?
      OR user_agent LIKE ?
    )";
  $kw = "%$q%";
  array_push($params, $kw,$kw,$kw,$kw,$kw,$kw,$kw);
}
if ($f_status !== '') {
  $where .= " AND status = ?";
  $params[] = $f_status;
}
if ($f_reason !== '') {
  $where .= " AND reason_code = ?";
  $params[] = $f_reason;
}

/* إجمالي وعدد الصفحات */
$st = $pdo->prepare("SELECT COUNT(*) FROM post_reports WHERE $where");
$st->execute($params);
$total = (int)$st->fetchColumn();
$pages = max(1, (int)ceil($total / $perPage));

/* إحضار السجلات */
$sql = "SELECT id, post_id, post_user_id, reporter_user_id, reason_code, reason_text, details, status, ip, user_agent, created_at, updated_at
        FROM post_reports
        WHERE $where
        ORDER BY created_at DESC, id DESC
        LIMIT $perPage OFFSET $offset";
$st = $pdo->prepare($sql); $st->execute($params);
$rows = $st->fetchAll();

/* ملخصات سريعة حسب الحالة (للعرض فقط) */
$counts = [];
try {
  $counts = $pdo->query("SELECT status, COUNT(*) c FROM post_reports GROUP BY status")->fetchAll(PDO::FETCH_KEY_PAIR) ?: [];
} catch(Throwable $e) { $counts = []; }

/* -------------- View single ------------- */
$viewId = (int)($_GET['view'] ?? 0);
$viewRow = null;
if ($viewId > 0) {
  $st = $pdo->prepare("SELECT * FROM post_reports WHERE id=? LIMIT 1");
  $st->execute([$viewId]);
  $viewRow = $st->fetch();
}

/* -------------- Render ------------------ */
$PAGE_TITLE = "مراجعة التبليغات";
require __DIR__ . '/inc/header.php';
require __DIR__ . '/inc/sidebar.php';
?>
<!-- ====== تنسيق خاص بالصفحة ====== -->
<style>
.reports-page .table{
  width:100%;
  border-collapse:separate;
  border-spacing:0;
  background:var(--panel);
  border:1px solid var(--border);
  border-radius:12px;
  overflow:hidden;
  table-layout: fixed;
}
.reports-page .table td, .reports-page .table th{
  padding:10px 12px;
  border-bottom:1px solid var(--border);
  vertical-align:middle;
  text-align:start;
}
.reports-page .table thead th{
  background:#0b1220;color:var(--text);font-weight:700;
  position:sticky;top:0;z-index:3;
  box-shadow:inset 0 -1px 0 var(--border);
}
.reports-page .table tbody tr:nth-child(even) td{background:rgba(255,255,255,.015)}
.reports-page .table tbody tr:hover td{background:rgba(59,130,246,.08)}

.reports-page .td-center{text-align:center}
.reports-page .td-num{text-align:center;opacity:.9}
.reports-page .ellipsis{max-width:240px;display:inline-block;overflow:hidden;white-space:nowrap;text-overflow:ellipsis;vertical-align:bottom}
.reports-page .ellipsis.small{max-width:150px}
.reports-page .mono{font-family:monospace}

.reports-page .badge{display:inline-block;padding:2px 8px;border-radius:999px;border:1px solid var(--border);font-size:12px;opacity:.95}
.reports-page .badge.st-new{border-color:#3b82f6}
.reports-page .badge.st-review{border-color:#a78bfa}
.reports-page .badge.st-ok{border-color:#22c55e}
.reports-page .badge.st-bad{border-color:#ef4444}
.reports-page .badge.small{font-size:11px;padding:1px 6px}

.reports-page .actions{display:flex;gap:8px;flex-wrap:nowrap;align-items:center}
.reports-page .actions form{display:inline-block;margin:0}
.reports-page .btn.sm{padding:6px 10px;font-size:14px}

.reports-page .col-id{width:64px}.reports-page .col-post{width:110px}
.reports-page .col-users{width:220px}.reports-page .col-reason{width:170px}
.reports-page .col-rtext{width:240px}.reports-page .col-ip{width:140px}
.reports-page .col-ua{width:230px}.reports-page .col-status{width:150px}
.reports-page .col-created{width:150px}.reports-page .col-actions{width:300px}

.reports-page .table-wrap{max-height:70vh;overflow:auto;position:relative}
.reports-page .filters{display:flex; gap:12px; align-items:flex-end; flex-wrap:wrap}
.reports-page .summary{display:flex; gap:8px; flex-wrap:wrap}
</style>

<div class="reports-page">
  <!-- شريط التصفية والبحث -->
  <div class="card" style="margin-bottom:16px;">
    <form method="get" class="filters">
      <div class="input" style="min-width:260px;flex:1;">
        <label>ابحث في (المعرفات/النص/الآي بي/اليوزر إيجنت)</label>
        <input type="text" name="q" value="<?= e($q) ?>" placeholder="مثال: 154.190.* أو Dart">
      </div>

      <div class="input">
        <label>الحالة</label>
        <select name="status">
          <option value="">الكل</option>
          <?php foreach ($ALLOWED_STATUSES as $s): ?>
            <option value="<?= e($s) ?>" <?= $f_status===$s?'selected':''; ?>><?= e(status_label($s)) ?></option>
          <?php endforeach; ?>
        </select>
      </div>

      <div class="input">
        <label>السبب (الكود)</label>
        <select name="reason">
          <option value="">الكل</option>
          <?php foreach (['abuse','harassment','sexual','spam','impersonate'] as $rc): ?>
            <option value="<?= e($rc) ?>" <?= $f_reason===$rc?'selected':''; ?>><?= e(reason_label($rc)) ?> (<?= e($rc) ?>)</option>
          <?php endforeach; ?>
        </select>
      </div>

      <button class="btn sm" type="submit">تطبيق</button>
      <?php if ($q!=='' || $f_status!=='' || $f_reason!==''): ?>
        <a class="btn sm secondary" href="post_reports.php">تفريغ الفلاتر</a>
      <?php endif; ?>

      <div style="margin-inline-start:auto; opacity:.9;">إجمالي: <b><?= $total ?></b> بلاغ</div>
    </form>

    <?php if ($msg): ?><div class="alert" style="margin-top:12px; border-color:#22c55e;"><?= e($msg) ?></div><?php endif; ?>
    <?php if ($err): ?><div class="alert" style="margin-top:12px;"><?= e($err) ?></div><?php endif; ?>

    <!-- ملخص سريع حسب الحالة -->
    <?php if ($counts): ?>
      <div class="summary" style="margin-top:10px;">
        <?php foreach($counts as $stt=>$c): ?>
          <div><?= status_badge((string)$stt) ?> &nbsp;<b><?= (int)$c ?></b></div>
        <?php endforeach; ?>
      </div>
    <?php endif; ?>
  </div>

  <!-- بطاقة تفاصيل عند view=ID -->
  <?php if ($viewRow): ?>
    <div class="card" style="margin-bottom:16px;">
      <div style="display:flex;justify-content:space-between;align-items:center;gap:12px;">
        <h3 style="margin:0;">تفاصيل البلاغ #<?= (int)$viewRow['id'] ?></h3>
        <a class="btn sm secondary" href="post_reports.php">إغلاق</a>
      </div>
      <div style="margin-top:10px;display:grid;grid-template-columns:1fr 1fr;gap:12px;">
        <div><b>المعرف:</b> <?= (int)$viewRow['id'] ?></div>
        <div><b>الحالة:</b> <?= status_badge((string)$viewRow['status']) ?></div>
        <div><b>Post ID:</b> <span class="mono"><?= e((string)$viewRow['post_id']) ?></span></div>
        <div><b>صاحب المنشور (ID):</b> <a class="mono" href="users_edit.php?id=<?= (int)$viewRow['post_user_id'] ?>">#<?= (int)$viewRow['post_user_id'] ?></a></div>
        <div><b>المُبلّغ (ID):</b> <a class="mono" href="users_edit.php?id=<?= (int)$viewRow['reporter_user_id'] ?>">#<?= (int)$viewRow['reporter_user_id'] ?></a></div>
        <div><b>السبب:</b> <?= e(reason_label($viewRow['reason_code'])) ?> (<?= e((string)$viewRow['reason_code']) ?>)</div>
        <div style="grid-column:1/-1;"><b>نص السبب:</b> <?= e((string)$viewRow['reason_text']) ?></div>
        <div style="grid-column:1/-1;"><b>تفاصيل إضافية:</b> <pre style="white-space:pre-wrap;background:rgba(255,255,255,.03);padding:10px;border:1px solid var(--border);border-radius:8px;margin:0"><?= e((string)($viewRow['details'] ?? '')) ?></pre></div>
        <div><b>IP:</b> <span class="mono"><?= e((string)$viewRow['ip']) ?></span></div>
        <div><b>User-Agent:</b> <span class="ellipsis mono" title="<?= e((string)$viewRow['user_agent']) ?>"><?= e((string)$viewRow['user_agent']) ?></span></div>
        <div><b>أُنشيء:</b> <?= fmt_date($viewRow['created_at'] ?? '') ?></div>
        <div><b>آخر تحديث:</b> <?= fmt_date($viewRow['updated_at'] ?? '') ?></div>
      </div>
    </div>
  <?php endif; ?>

  <!-- الجدول -->
  <div class="card">
    <div class="table-wrap">
      <table class="table">
        <colgroup>
          <col class="col-id"><col class="col-post"><col class="col-users"><col class="col-reason">
          <col class="col-rtext"><col class="col-ip"><col class="col-ua"><col class="col-status">
          <col class="col-created"><col class="col-actions">
        </colgroup>
        <thead>
          <tr>
            <th class="td-center">#</th>
            <th>Post</th>
            <th>الأطراف</th>
            <th>السبب</th>
            <th>نص السبب</th>
            <th>IP</th>
            <th>User-Agent</th>
            <th>الحالة</th>
            <th>أُنشيء</th>
            <th class="col-actions">إجراءات</th>
          </tr>
        </thead>
        <tbody>
          <?php foreach ($rows as $r): ?>
            <tr>
              <td class="td-num"><?= (int)$r['id'] ?></td>
              <td>
                <div title="Post ID">
                  <span class="mono"><?= e((string)$r['post_id']) ?></span>
                </div>
              </td>
              <td>
                <div style="display:flex;flex-direction:column;gap:4px;">
                  <div title="صاحب المنشور">
                    صاحب المنشور: <a class="mono" href="users_edit.php?id=<?= (int)$r['post_user_id'] ?>">#<?= (int)$r['post_user_id'] ?></a>
                  </div>
                  <div title="المُبلّغ">
                    المُبلّغ: <a class="mono" href="users_edit.php?id=<?= (int)$r['reporter_user_id'] ?>">#<?= (int)$r['reporter_user_id'] ?></a>
                  </div>
                </div>
              </td>
              <td>
                <div><?= e(reason_label($r['reason_code'])) ?></div>
                <div class="mono" style="opacity:.7;font-size:12px;"><?= e((string)$r['reason_code']) ?></div>
              </td>
              <td title="<?= e((string)$r['reason_text']) ?>"><span class="ellipsis"><?= e((string)$r['reason_text']) ?></span></td>
              <td><span class="mono"><?= e((string)$r['ip']) ?></span></td>
              <td title="<?= e((string)$r['user_agent']) ?>"><span class="ellipsis small mono"><?= e((string)$r['user_agent']) ?></span></td>
              <td><?= status_badge((string)$r['status']) ?></td>
              <td title="<?= e($r['created_at'] ?? '') ?>"><?= fmt_date($r['created_at'] ?? '') ?></td>
              <td class="actions">
                <a class="btn sm secondary" href="post_reports.php?view=<?= (int)$r['id'] ?>">تفاصيل</a>

                <!-- تحديث الحالة -->
                <form method="post">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                  <input type="hidden" name="action" value="set_status">
                  <select name="status" class="btn sm" style="padding:6px 8px;">
                    <?php foreach ($ALLOWED_STATUSES as $s): ?>
                      <option value="<?= e($s) ?>" <?= ((string)$r['status']===$s)?'selected':''; ?>><?= e(status_label($s)) ?></option>
                    <?php endforeach; ?>
                  </select>
                  <button class="btn sm" type="submit">حفظ</button>
                </form>

                <!-- حذف -->
                <form method="post" onsubmit="return confirm('حذف البلاغ نهائيًا؟ لا يمكن التراجع.');">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                  <input type="hidden" name="action" value="delete_report">
                  <button class="btn sm" type="submit" style="background:var(--danger);">حذف</button>
                </form>
              </td>
            </tr>
          <?php endforeach; ?>
          <?php if (!$rows): ?>
            <tr><td colspan="10" style="text-align:center; opacity:.8;">لا توجد تبليغات</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>

    <!-- صفحات -->
    <?php if ($pages > 1): ?>
      <div style="display:flex; gap:8px; margin-top:12px; flex-wrap: wrap;">
        <?php
          $base = 'post_reports.php?';
          $qs = [];
          if ($q!=='')        $qs[] = 'q='.urlencode($q);
          if ($f_status!=='') $qs[] = 'status='.urlencode($f_status);
          if ($f_reason!=='') $qs[] = 'reason='.urlencode($f_reason);
          $base .= implode('&', $qs);
          if ($base && !str_ends_with($base,'&') && !str_ends_with($base,'?')) $base .= '&';
        ?>
        <?php for($p=1;$p<=$pages;$p++):
          $url = $base.'page='.$p;
          $is = $p===$page; ?>
          <a href="<?= e($url) ?>" class="btn sm <?= $is ? '' : 'secondary' ?>"<?= $is?' style="pointer-events:none; opacity:.8;"':''; ?>><?= $p ?></a>
        <?php endfor; ?>
      </div>
    <?php endif; ?>
  </div>
</div>

<?php require __DIR__ . '/inc/footer.php'; ?>
