<?php
// admin/agents_stats.php
declare(strict_types=1);
require_once __DIR__ . '/inc/auth.php';
require_login();

/* ===== Helpers ===== */
if (!function_exists('e')) {
  function e(string $s): string { return htmlspecialchars($s, ENT_QUOTES|ENT_SUBSTITUTE, 'UTF-8'); }
}
function usd_cols(PDO $pdo): array {
  $has_cents = $has_dec = false;
  try { $s=$pdo->query("SHOW COLUMNS FROM agents LIKE 'usd_balance_cents'"); $has_cents=(bool)$s->fetchColumn(); } catch(Throwable $e){}
  try { $s=$pdo->query("SHOW COLUMNS FROM agents LIKE 'usd_balance'");       $has_dec  =(bool)$s->fetchColumn(); } catch(Throwable $e){}
  return [$has_cents,$has_dec];
}
function usd_display_from_row(array $r, bool $has_cents, bool $has_dec): string {
  if ($has_cents && isset($r['usd_balance_cents'])) return number_format(((int)$r['usd_balance_cents'])/100, 2, '.', '');
  if ($has_dec   && isset($r['usd_balance']))       return number_format((float)$r['usd_balance'], 2, '.', '');
  return '—';
}

$pdo = db();

/* ===== بحث بسيط ===== */
$q = trim((string)($_GET['q'] ?? ''));

/* ===== حساب الحدود الزمنية ===== */
// اليوم
$todayStart = new DateTime('today');
$todayEnd   = (clone $todayStart)->modify('+1 day');
// الأسبوع: من السبت 00:00 إلى السبت التالي 00:00 (يغطي السبت→الجمعة)
$w = (int)$todayStart->format('w');         // 0=الأحد .. 6=السبت
$diffToSat = ($w - 6 + 7) % 7;
$weekStart = (clone $todayStart)->modify("-{$diffToSat} days");
$weekEnd   = (clone $weekStart)->modify('+7 days');

$params = [
  ':t0' => $todayStart->format('Y-m-d H:i:s'),
  ':t1' => $todayEnd->format('Y-m-d H:i:s'),
  ':w0' => $weekStart->format('Y-m-d H:i:s'),
  ':w1' => $weekEnd->format('Y-m-d H:i:s'),
];

list($has_usd_cents, $has_usd_dec) = usd_cols($pdo);

/* ===== جلب البيانات مجمّعة ===== */
$where = "1=1";
$qParams = [];
if ($q !== '') {
  $where .= " AND (a.username LIKE :kw OR a.display_name LIKE :kw)";
  $qParams[':kw'] = "%$q%";
}

$sql = "
  SELECT
    a.id, a.username, a.display_name, a.status, a.main_balance" .
    ($has_usd_cents ? ", a.usd_balance_cents" : "") .
    ($has_usd_dec   ? ", a.usd_balance"       : "") . ",
    COALESCE(SUM(CASE WHEN c.status='success' AND c.created_at>=:t0 AND c.created_at<:t1 THEN c.coins END),0) AS coins_today,
    COALESCE(SUM(CASE WHEN c.status='success' AND c.created_at>=:w0 AND c.created_at<:w1 THEN c.coins END),0) AS coins_week,
    COALESCE(SUM(CASE WHEN c.status='success' THEN c.coins END),0) AS coins_total,
    COUNT(CASE WHEN c.status='success' THEN 1 END) AS ops_total
  FROM agents a
  LEFT JOIN agent_charges c ON c.agent_id = a.id
  WHERE $where
  GROUP BY a.id
  ORDER BY coins_today DESC, coins_week DESC, a.id DESC
";
$st = $pdo->prepare($sql);
$st->execute($params + $qParams);
$rows = $st->fetchAll(PDO::FETCH_ASSOC);

/* ===== مجاميع عامة ===== */
$sum_today = $sum_week = $sum_total = 0;
foreach ($rows as $r) {
  $sum_today += (int)$r['coins_today'];
  $sum_week  += (int)$r['coins_week'];
  $sum_total += (int)$r['coins_total'];
}

/* ===== Render ===== */
$PAGE_TITLE = 'إحصائيات الوكلاء';
require __DIR__ . '/inc/header.php';
require __DIR__ . '/inc/sidebar.php';
?>
<style>
.agents-stats .table{
  width:100%;border-collapse:separate;border-spacing:0;background:var(--panel);
  border:1px solid var(--border);border-radius:12px;overflow:hidden;table-layout:fixed;
}
.agents-stats .table td,.agents-stats .table th{padding:10px 12px;border-bottom:1px solid var(--border);vertical-align:middle}
.agents-stats .table thead th{background:#0b1220;color:var(--text);font-weight:700;position:sticky;top:0;z-index:3;box-shadow:inset 0 -1px 0 var(--border)}
.agents-stats .table tbody tr:nth-child(even) td{background:rgba(255,255,255,.015)}
.agents-stats .badge{display:inline-block;padding:2px 8px;border-radius:999px;border:1px solid var(--border);font-size:12px}
.agents-stats .badge.ok{border-color:#22c55e}.agents-stats .badge.bad{border-color:#ef4444}
.agents-stats .mono{font-family:ui-monospace,Menlo,Consolas,monospace}
.agents-stats .grid{display:grid;gap:12px;grid-template-columns:repeat(3,1fr)}
@media (max-width:900px){ .agents-stats .grid{grid-template-columns:1fr} }
.agents-stats .stat{background:var(--panel);border:1px solid var(--border);border-radius:12px;padding:12px}
.agents-stats .stat h4{margin:0 0 6px;font-size:14px;color:var(--sub)}
.agents-stats .stat .val{font-size:22px;font-weight:800}
.agents-stats .col-id{width:72px}.agents-stats .col-user{width:200px}.agents-stats .col-name{width:220px}
.agents-stats .col-status{width:110px}.agents-stats .col-bal{width:150px}
.agents-stats .col-today{width:140px}.agents-stats .col-week{width:160px}.agents-stats .col-total{width:160px}
.agents-stats .actions{width:240px}
.agents-stats .btn.sm{padding:6px 10px;font-size:14px}
</style>

<div class="agents-stats">
  <div class="card" style="margin-bottom:12px;">
    <form method="get" class="form" style="display:flex;gap:12px;align-items:flex-end;flex-wrap:wrap">
      <div class="input" style="min-width:260px;flex:1">
        <label>بحث بالاسم/اللقب</label>
        <input type="text" name="q" value="<?= e($q) ?>" placeholder="مثال: ali أو alex">
      </div>
      <button class="btn sm" type="submit">بحث</button>
      <?php if ($q!==''): ?><a class="btn sm secondary" href="agents_stats.php">مسح البحث</a><?php endif; ?>
      <div style="margin-inline-start:auto;opacity:.9">
        الفترة الأسبوعية الحالية: <b class="mono"><?= e($weekStart->format('Y-m-d')) ?></b> ⟶ <b class="mono"><?= e($weekEnd->modify('-1 second')->format('Y-m-d')) ?></b>
      </div>
    </form>
  </div>

  <!-- مجاميع -->
  <div class="grid" style="margin-bottom:12px;">
    <div class="stat">
      <h4>إجمالي شحن اليوم لجميع الوكلاء</h4>
      <div class="val mono"><?= number_format($sum_today) ?> كوين</div>
    </div>
    <div class="stat">
      <h4>إجمالي شحن الأسبوع (السبت ⟶ الجمعة)</h4>
      <div class="val mono"><?= number_format($sum_week) ?> كوين</div>
    </div>
    <div class="stat">
      <h4>إجمالي الشحن (طوال الفترة)</h4>
      <div class="val mono"><?= number_format($sum_total) ?> كوين</div>
    </div>
  </div>

  <div class="card">
    <div style="margin-bottom:8px;opacity:.9">إجمالي النتائج: <b><?= count($rows) ?></b></div>
    <div class="table-wrap" style="max-height:70vh;overflow:auto;position:relative">
      <table class="table">
        <colgroup>
          <col class="col-id"><col class="col-user"><col class="col-name"><col class="col-status">
          <col class="col-bal"><col class="col-today"><col class="col-week"><col class="col-total"><col class="actions">
        </colgroup>
        <thead>
          <tr>
            <th>#</th>
            <th>اسم المستخدم</th>
            <th>الاسم الظاهر</th>
            <th>الحالة</th>
            <th>الأرصدة</th>
            <th>شحن اليوم</th>
            <th>شحن الأسبوع</th>
            <th>إجمالي الشحن</th>
            <th>إجراءات</th>
          </tr>
        </thead>
        <tbody>
          <?php if ($rows): foreach($rows as $r): ?>
            <tr>
              <td class="mono"><?= (int)$r['id'] ?></td>
              <td><?= e((string)$r['username']) ?></td>
              <td><?= e((string)$r['display_name']) ?></td>
              <td>
                <?php if (($r['status'] ?? '')==='active'): ?>
                  <span class="badge ok">نشط</span>
                <?php else: ?>
                  <span class="badge bad">موقوف</span>
                <?php endif; ?>
              </td>
              <td>
                <div>🪙 <b class="mono"><?= (int)$r['main_balance'] ?></b> كوين</div>
                <div>💵 <b class="mono"><?= e(usd_display_from_row($r, $has_usd_cents, $has_usd_dec)) ?></b> $</div>
              </td>
              <td class="mono"><?= number_format((int)$r['coins_today']) ?></td>
              <td class="mono"><?= number_format((int)$r['coins_week']) ?></td>
              <td class="mono"><?= number_format((int)$r['coins_total']) ?></td>
              <td style="display:flex;gap:8px;flex-wrap:wrap">
                <a class="btn sm" href="agents_credit.php?id=<?= (int)$r['id'] ?>">شحن/خصم</a>
                <a class="btn sm secondary" href="agents_edit.php?id=<?= (int)$r['id'] ?>">تعديل</a>
              </td>
            </tr>
          <?php endforeach; else: ?>
            <tr><td colspan="9" style="text-align:center;opacity:.8">لا توجد بيانات</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>

<?php require __DIR__ . '/inc/footer.php'; ?>
