<?php
// admin/agents_credit.php
declare(strict_types=1);
require_once __DIR__ . '/inc/auth.php';
require_login();

if (!isset($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
function csrf_token(): string { return $_SESSION['csrf']; }
function csrf_ok(?string $t): bool { return $t && hash_equals($_SESSION['csrf'], $t); }
if (!function_exists('e')) { function e(string $s): string { return htmlspecialchars($s, ENT_QUOTES|ENT_SUBSTITUTE, 'UTF-8'); } }

$pdo = db();

/* Helpers */
function col_exists(PDO $pdo, string $table, string $col): bool {
  try {
    $st = $pdo->prepare("SHOW COLUMNS FROM `$table` LIKE ?");
    $st->execute([$col]);
    return (bool)$st->fetchColumn();
  } catch (Throwable $e) { return false; }
}
function table_exists(PDO $pdo, string $table): bool {
  try {
    $st = $pdo->prepare("SHOW TABLES LIKE ?");
    $st->execute([$table]);
    return (bool)$st->fetchColumn();
  } catch (Throwable $e) { return false; }
}

/* جلب الوكيل */
$id  = (int)($_GET['id'] ?? 0);
$st  = $pdo->prepare("SELECT * FROM agents WHERE id=? LIMIT 1");
$st->execute([$id]);
$agent = $st->fetch();
if (!$agent) redirect('agents.php');

/* الكشف عن أعمدة الدولار */
$has_usd_cents = col_exists($pdo, 'agents', 'usd_balance_cents');
$has_usd_dec   = col_exists($pdo, 'agents', 'usd_balance');

/* حساب رصيد الدولار الحالي للعرض */
$usd_display = null;
if ($has_usd_cents && array_key_exists('usd_balance_cents', $agent)) {
  $usd_display = number_format(((int)$agent['usd_balance_cents'])/100, 2, '.', '');
} elseif ($has_usd_dec && array_key_exists('usd_balance', $agent)) {
  $usd_display = number_format((float)$agent['usd_balance'], 2, '.', '');
}

$msg=null; $err=null;

/* معالجة POST */
if ($_SERVER['REQUEST_METHOD']==='POST') {
  if (!csrf_ok($_POST['csrf'] ?? '')) {
    $err='رمز الأمان غير صالح.';
  } else {
    $kind = ($_POST['kind'] ?? 'coins') === 'usd' ? 'usd' : 'coins'; // coins | usd
    $type = ($_POST['type'] ?? 'credit') === 'debit' ? 'debit' : 'credit'; // credit | debit

    try {
      $pdo->beginTransaction();

      if ($kind === 'coins') {
        $amt = (int)($_POST['amount_coins'] ?? 0);
        if ($amt <= 0) throw new RuntimeException('bad_amount');

        // قفل صف الوكيل
        $st=$pdo->prepare("SELECT id, main_balance FROM agents WHERE id=? FOR UPDATE");
        $st->execute([$id]);
        $row = $st->fetch(PDO::FETCH_ASSOC);
        if (!$row) throw new RuntimeException('agent_not_found');

        $bal = (int)$row['main_balance'];
        $new = $type==='credit' ? $bal + $amt : $bal - $amt;
        if ($new < 0) throw new RuntimeException('insufficient_balance');

        $u=$pdo->prepare("UPDATE agents SET main_balance=? WHERE id=? LIMIT 1");
        $u->execute([$new, $id]);

        // Ledger
        $reason  = $type==='credit' ? 'admin_credit' : 'admin_debit';
        $adminId = function_exists('admin_id') ? (int)admin_id() : null;

        $has_currency_col = col_exists($pdo, 'agent_ledger', 'currency');
        if ($has_currency_col) {
          $ins=$pdo->prepare(
            "INSERT INTO agent_ledger (agent_id, direction, reason, amount, balance_after, ref_admin_id, currency)
             VALUES (?,?,?,?,?,?,?)"
          );
          $ins->execute([$id, $type==='credit'?'in':'out', $reason, $amt, $new, $adminId, 'coins']);
        } else {
          $ins=$pdo->prepare(
            "INSERT INTO agent_ledger (agent_id, direction, reason, amount, balance_after, ref_admin_id)
             VALUES (?,?,?,?,?,?)"
          );
          $ins->execute([$id, $type==='credit'?'in':'out', $reason, $amt, $new, $adminId]);
        }

        $msg='تم تنفيذ عملية الكوين بنجاح.';
      } else { // USD
        if (!$has_usd_cents && !$has_usd_dec) throw new RuntimeException('usd_column_missing');

        // قراءة المبلغ بالدولار مع دعم الفاصلة العشرية
        $amt_usd_raw = trim((string)($_POST['amount_usd'] ?? '0'));
        $amt_usd     = (float) str_replace(',', '.', $amt_usd_raw);
        if ($amt_usd <= 0) throw new RuntimeException('bad_amount');

        // قفل صف الوكيل
        $sel_cols = "id, main_balance";
        if ($has_usd_cents) $sel_cols .= ", IFNULL(usd_balance_cents,0) AS usd_balance_cents";
        if ($has_usd_dec)   $sel_cols .= ", IFNULL(usd_balance,0) AS usd_balance";
        $st=$pdo->prepare("SELECT $sel_cols FROM agents WHERE id=? FOR UPDATE");
        $st->execute([$id]);
        $row = $st->fetch(PDO::FETCH_ASSOC);
        if (!$row) throw new RuntimeException('agent_not_found');

        if ($has_usd_cents) {
          $current_cents = (int)$row['usd_balance_cents'];
          $delta_cents   = (int) round($amt_usd * 100);
          $new_cents     = $type==='credit' ? $current_cents + $delta_cents : $current_cents - $delta_cents;
          if ($new_cents < 0) throw new RuntimeException('insufficient_balance');

          $u=$pdo->prepare("UPDATE agents SET usd_balance_cents=? WHERE id=? LIMIT 1");
          $u->execute([$new_cents, $id]);

          $usd_after = $new_cents / 100.0;
        } else { // DECIMAL
          $current = (float)$row['usd_balance'];
          $new     = $type==='credit' ? $current + $amt_usd : $current - $amt_usd;
          if ($new < -1e-9) throw new RuntimeException('insufficient_balance');

          $u=$pdo->prepare("UPDATE agents SET usd_balance=? WHERE id=? LIMIT 1");
          $u->execute([$new, $id]);

          $usd_after = $new;
        }

        // Ledger
        $reason  = $type==='credit' ? 'admin_usd_credit' : 'admin_usd_debit';
        $adminId = function_exists('admin_id') ? (int)admin_id() : null;

        $has_currency_col = col_exists($pdo, 'agent_ledger', 'currency');
        if ($has_currency_col) {
          $ins=$pdo->prepare(
            "INSERT INTO agent_ledger (agent_id, direction, reason, amount, balance_after, ref_admin_id, currency)
             VALUES (?,?,?,?,?,?,?)"
          );
          $ins->execute([$id, $type==='credit'?'in':'out', $reason, $amt_usd, $usd_after, $adminId, 'usd']);
        } else {
          $ins=$pdo->prepare(
            "INSERT INTO agent_ledger (agent_id, direction, reason, amount, balance_after, ref_admin_id)
             VALUES (?,?,?,?,?,?)"
          );
          $ins->execute([$id, $type==='credit'?'in':'out', $reason, $amt_usd, $usd_after, $adminId]);
        }

        $msg='تم تنفيذ عملية الدولار بنجاح.';
      }

      $pdo->commit();

      // تحديث الوكيل للعرض
      $st=$pdo->prepare("SELECT * FROM agents WHERE id=? LIMIT 1");
      $st->execute([$id]);
      $agent = $st->fetch() ?: $agent;

      // إعادة حساب عرض الدولار
      if ($has_usd_cents && array_key_exists('usd_balance_cents', $agent)) {
        $usd_display = number_format(((int)$agent['usd_balance_cents'])/100, 2, '.', '');
      } elseif ($has_usd_dec && array_key_exists('usd_balance', $agent)) {
        $usd_display = number_format((float)$agent['usd_balance'], 2, '.', '');
      }

    } catch (Throwable $e) {
      if ($pdo->inTransaction()) { try { $pdo->rollBack(); } catch (Throwable $e2) {} }
      $code = $e->getMessage();
      if ($code === 'bad_amount')                $err='المبلغ غير صالح.';
      elseif ($code === 'agent_not_found')       $err='الوكيل غير موجود.';
      elseif ($code === 'insufficient_balance')  $err='الرصيد غير كافٍ.';
      elseif ($code === 'usd_column_missing')    $err='لا يوجد عمود رصيد الدولار في جدول الوكلاء.';
      else                                       $err='تعذّر تنفيذ العملية.';
    }
  }
}

$PAGE_TITLE = 'شحن/خصم رصيد وكيل';
require __DIR__ . '/inc/header.php';
require __DIR__ . '/inc/sidebar.php';
?>
<style>
.agent-credit .grid{display:grid;grid-template-columns: 1fr 1fr; gap:16px}
@media (max-width: 900px){ .agent-credit .grid{grid-template-columns: 1fr} }
.agent-credit .chips{display:flex;gap:8px;flex-wrap:wrap;margin-top:6px}
.agent-credit .chip{display:inline-flex;align-items:center;gap:8px;border:1px solid var(--border);background:var(--panel);padding:6px 10px;border-radius:999px}
</style>

<div class="agent-credit">
  <div class="card" style="margin-bottom:16px;">
    <h2 style="margin-top:0">
      وكيل: <?= e($agent['display_name'] ?? $agent['username'] ?? ('#'.$agent['id'])) ?>
      <small style="opacity:.8;">(<?= e($agent['username'] ?? '') ?>)</small>
    </h2>

    <div class="chips">
      <div class="chip">🪙 الرصيد (كوين): <b><?= (int)$agent['main_balance'] ?></b></div>
      <div class="chip">💵 الرصيد ($): <b><?= $usd_display !== null ? e($usd_display) : '—' ?></b></div>
    </div>

    <?php if ($msg): ?><div class="alert" style="margin-top:12px;border-color:#22c55e;"><?= e($msg) ?></div><?php endif; ?>
    <?php if ($err): ?><div class="alert" style="margin-top:12px;"><?= e($err) ?></div><?php endif; ?>

    <div class="grid" style="margin-top:12px;">
      <!-- ===== نموذج الكوين ===== -->
      <form method="post" class="form" style="background:var(--panel);padding:12px;border-radius:12px;border:1px solid var(--border)">
        <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
        <input type="hidden" name="kind" value="coins">
        <h3 style="margin:0 0 8px">شحن/خصم كوين</h3>

        <div class="input">
          <label>نوع العملية</label>
          <div style="display:flex; gap:16px; align-items:center; padding-top:6px;">
            <label><input type="radio" name="type" value="credit" checked> شحن</label>
            <label><input type="radio" name="type" value="debit"> خصم</label>
          </div>
        </div>

        <div class="input">
          <label>المبلغ (كوين)</label>
          <input name="amount_coins" type="number" min="1" step="1" required>
        </div>

        <div style="margin-top:10px; display:flex; gap:8px; align-items:center;">
          <button class="btn" type="submit">تنفيذ</button>
          <a class="btn secondary" href="agents.php">رجوع</a>
          <a class="btn sm secondary" href="agents_edit.php?id=<?= (int)$agent['id'] ?>">فتح بطاقة الوكيل</a>
        </div>
      </form>

      <!-- ===== نموذج الدولار ===== -->
      <form method="post" class="form" style="background:var(--panel);padding:12px;border-radius:12px;border:1px solid var(--border)">
        <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
        <input type="hidden" name="kind" value="usd">
        <h3 style="margin:0 0 8px">شحن/خصم دولار</h3>

        <?php if (!$has_usd_cents && !$has_usd_dec): ?>
          <div class="alert">لا يوجد عمود رصيد الدولار في جدول <code>agents</code>. أضف أحدهما:
            <div style="margin-top:6px;opacity:.9">
              <code>ALTER TABLE agents ADD COLUMN usd_balance_cents INT NOT NULL DEFAULT 0;</code><br>
              أو<br>
              <code>ALTER TABLE agents ADD COLUMN usd_balance DECIMAL(14,2) NOT NULL DEFAULT 0;</code>
            </div>
          </div>
        <?php else: ?>
          <div class="input">
            <label>نوع العملية</label>
            <div style="display:flex; gap:16px; align-items:center; padding-top:6px;">
              <label><input type="radio" name="type" value="credit" checked> شحن</label>
              <label><input type="radio" name="type" value="debit"> خصم</label>
            </div>
          </div>

          <div class="input">
            <label>المبلغ ($)</label>
            <input name="amount_usd" type="text" inputmode="decimal" placeholder="مثال: 10.50 أو 10,50" required>
          </div>

          <div style="margin-top:10px; display:flex; gap:8px; align-items:center;">
            <button class="btn" type="submit">تنفيذ</button>
            <a class="btn secondary" href="agents.php">رجوع</a>
            <a class="btn sm secondary" href="agents_edit.php?id=<?=
              (int)$agent['id'] ?>">فتح بطاقة الوكيل</a>
          </div>
        <?php endif; ?>
      </form>
    </div>
  </div>

  <div class="card">
    <h3 style="margin-top:0;">ملاحظات</h3>
    <ul style="margin:0 0 0 1rem; opacity:.9; line-height:1.8">
      <li>لا يمكن الخصم إذا كان الرصيد الناتج سالبًا (سواء للكوين أو الدولار).</li>
      <li>يتم تسجيل كل عملية في <code>agent_ledger</code>:
        <ul>
          <li>كوين: <code>admin_credit</code> / <code>admin_debit</code> (currency = <code>coins</code> إذا متوفر).</li>
          <li>دولار: <code>admin_usd_credit</code> / <code>admin_usd_debit</code> (currency = <code>usd</code> إذا متوفر).</li>
        </ul>
      </li>
      <li>الدولار يُخزَّن وفق المتوفر: <code>usd_balance_cents</code> (بالسنت) أو <code>usd_balance</code> (كسور عشرية).</li>
    </ul>
  </div>
</div>

<?php require __DIR__ . '/inc/footer.php'; ?>
