<?php
// admin/agents.php
declare(strict_types=1);
require_once __DIR__ . '/inc/auth.php';
require_login();

/* ---------------- CSRF ---------------- */
if (!isset($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
function csrf_token(): string { return $_SESSION['csrf']; }
function csrf_ok(?string $t): bool { return $t && hash_equals($_SESSION['csrf'], $t); }

/* -------------- Helpers --------------- */
if (!function_exists('e')) {
  function e(string $s): string { return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }
}
function fmt_date(?string $s): string {
  if (!$s) return '';
  $t = strtotime($s); if ($t<=0) return e($s);
  return date('Y-m-d H:i', $t);
}
function fmt_ip(?string $bin): string {
  if ($bin===null || $bin==='') return '';
  $ip = @inet_ntop($bin);
  return $ip ?: '';
}
function col_exists(PDO $pdo, string $table, string $col): bool {
  try {
    $st = $pdo->prepare("SHOW COLUMNS FROM `$table` LIKE ?");
    $st->execute([$col]);
    return (bool)$st->fetchColumn();
  } catch (Throwable $e) { return false; }
}

/* -------------- Actions ---------------- */
$pdo = db();
$msg = null; $err = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $action = $_POST['action'] ?? '';
  $id     = (int)($_POST['id'] ?? 0);
  if (!csrf_ok($_POST['csrf'] ?? '')) {
    $err = 'رمز الأمان غير صالح. أعد المحاولة.';
  } elseif ($id <= 0) {
    $err = 'مُعرّف غير صالح.';
  } else {
    try {
      if ($action === 'toggle_status') {
        $st = $pdo->prepare("SELECT status FROM agents WHERE id=? LIMIT 1");
        $st->execute([$id]);
        $row = $st->fetch();
        if ($row) {
          $new = ($row['status'] === 'active') ? 'suspended' : 'active';
          $pdo->prepare("UPDATE agents SET status=? WHERE id=? LIMIT 1")->execute([$new, $id]);
          $msg = 'تم تحديث حالة الوكيل.';
        } else {
          $err = 'الوكيل غير موجود.';
        }
      } elseif ($action === 'delete_agent') {
        // احتياطيًا: لا تحذف إذا لديه رصيد
        $st = $pdo->prepare("SELECT main_balance FROM agents WHERE id=? LIMIT 1");
        $st->execute([$id]);
        $row = $st->fetch();
        if (!$row) { $err='الوكيل غير موجود.'; }
        elseif ((int)$row['main_balance'] !== 0) { $err='لا يمكن الحذف ورصيد الوكيل غير صفري.'; }
        else {
          $pdo->prepare("DELETE FROM agents WHERE id=? LIMIT 1")->execute([$id]);
          $msg = 'تم حذف الوكيل نهائيًا.';
        }
      }
    } catch (Throwable $e) {
      $err = 'حدث خطأ أثناء تنفيذ العملية.';
    }
  }
}

/* -------------- Search & Page ---------- */
$q       = trim($_GET['q'] ?? '');
$page    = max(1, (int)($_GET['page'] ?? 1));
$perPage = 50;
$offset  = ($page - 1) * $perPage;

$where  = "1=1"; $params = [];
if ($q !== '') {
  $where .= " AND (username LIKE ? OR display_name LIKE ?)";
  $kw = "%$q%"; array_push($params,$kw,$kw);
}

/* كشف أعمدة الدولار لاستخدامها في الاستعلام والعرض */
$has_usd_cents = col_exists($pdo, 'agents', 'usd_balance_cents');
$has_usd_dec   = col_exists($pdo, 'agents', 'usd_balance');

$st = $pdo->prepare("SELECT COUNT(*) FROM agents WHERE $where");
$st->execute($params);
$total = (int)$st->fetchColumn();
$pages = max(1, (int)ceil($total / $perPage));

$selectCols = "id, username, display_name, status, main_balance, created_at, last_login_at, last_login_ip";
if ($has_usd_cents) $selectCols .= ", IFNULL(usd_balance_cents,0) AS usd_balance_cents";
if ($has_usd_dec)   $selectCols .= ", IFNULL(usd_balance,0) AS usd_balance";

$sql = "SELECT $selectCols
        FROM agents
        WHERE $where
        ORDER BY id DESC
        LIMIT $perPage OFFSET $offset";
$st = $pdo->prepare($sql); $st->execute($params);
$rows = $st->fetchAll();

/* -------------- Render ------------------ */
$PAGE_TITLE = "وكلاء الشحن";
require __DIR__ . '/inc/header.php';
require __DIR__ . '/inc/sidebar.php';
?>
<style>
.agents-page .table{
  width:100%;
  border-collapse:separate;
  border-spacing:0;
  background:var(--panel);
  border:1px solid var(--border);
  border-radius:12px;
  overflow:hidden;
  table-layout: fixed;
}
.agents-page .table td, .agents-page .table th{
  padding:10px 12px;
  border-bottom:1px solid var(--border);
  vertical-align:middle;
  text-align:start;
}
.agents-page .table thead th{
  background:#0b1220;color:var(--text);font-weight:700;
  position:sticky;top:0;z-index:3;
  box-shadow:inset 0 -1px 0 var(--border);
}
.agents-page .table tbody tr:nth-child(even) td{background:rgba(255,255,255,.015)}
.agents-page .table tbody tr:hover td{background:rgba(59,130,246,.08)}
.agents-page .td-center{text-align:center}
.agents-page .badge{display:inline-block;padding:2px 8px;border-radius:999px;border:1px solid var(--border);font-size:12px;opacity:.95}
.agents-page .badge.coin{border-color:#22c55e}
.agents-page .badge.usd{border-color:#2e7d32}
.agents-page .badge.active{border-color:#22c55e}
.agents-page .badge.suspended{border-color:#ef4444}
.agents-page .actions{display:flex;gap:8px;flex-wrap:nowrap;align-items:center;justify-content:flex-start}
.agents-page .actions form{display:inline-block;margin:0}
.agents-page .btn.sm{padding:6px 10px;font-size:14px}
.agents-page .table-wrap{max-height:70vh;overflow:auto;position:relative}
.agents-page .col-id{width:70px}
.agents-page .col-user{width:200px}
.agents-page .col-name{width:200px}
.agents-page .col-status{width:120px}
.agents-page .col-balance{width:130px}
.agents-page .col-usd{width:130px}
.agents-page .col-created{width:160px}
.agents-page .col-lastlogin{width:220px}
.agents-page .col-actions{width:340px}
</style>

<div class="agents-page">
  <div class="card" style="margin-bottom:16px;">
    <form method="get" class="form" style="display:flex; gap:12px; align-items:flex-end; flex-wrap: wrap;">
      <div class="input" style="min-width:260px; flex:1;">
        <label>ابحث بالاسم/اللقب</label>
        <input type="text" name="q" value="<?= e($q) ?>" placeholder="مثال: agent1 أو أحمد">
      </div>
      <button class="btn sm" type="submit">بحث</button>
      <?php if ($q!==''): ?>
        <a class="btn sm secondary" href="agents.php">مسح البحث</a>
      <?php endif; ?>
      <div style="margin-inline-start:auto; display:flex; gap:8px; align-items:center;">
        <a class="btn sm" href="agents_edit.php">+ إضافة وكيل</a>
        <span style="opacity:.9;">إجمالي: <b><?= $total ?></b> نتيجة</span>
      </div>
    </form>
    <?php if ($msg): ?><div class="alert" style="margin-top:12px; border-color:#22c55e;"><?= e($msg) ?></div><?php endif; ?>
    <?php if ($err): ?><div class="alert" style="margin-top:12px;"><?= e($err) ?></div><?php endif; ?>
  </div>

  <div class="card">
    <div class="table-wrap">
      <table class="table">
        <colgroup>
          <col class="col-id"><col class="col-user"><col class="col-name"><col class="col-status">
          <col class="col-balance"><col class="col-usd"><col class="col-created"><col class="col-lastlogin"><col class="col-actions">
        </colgroup>
        <thead>
          <tr>
            <th class="td-center">#</th>
            <th>اسم المستخدم</th>
            <th>الاسم الظاهر</th>
            <th>الحالة</th>
            <th>الرصيد (كوين)</th>
            <th>الدولار ($)</th>
            <th>أُنشيء</th>
            <th>آخر دخول</th>
            <th class="col-actions">إجراءات</th>
          </tr>
        </thead>
        <tbody>
          <?php foreach ($rows as $r): ?>
            <?php
              // تحضير عرض الدولار حسب العمود المتوفر
              $usd_disp = null;
              if ($has_usd_cents && array_key_exists('usd_balance_cents', $r)) {
                $usd_disp = number_format(((int)$r['usd_balance_cents'])/100, 2, '.', '');
              } elseif ($has_usd_dec && array_key_exists('usd_balance', $r)) {
                $usd_disp = number_format((float)$r['usd_balance'], 2, '.', '');
              }
            ?>
            <tr>
              <td class="td-center"><?= (int)$r['id'] ?></td>
              <td><?= e($r['username']) ?></td>
              <td><?= e($r['display_name']) ?></td>
              <td>
                <?php if ($r['status']==='active'): ?>
                  <span class="badge active">نشط</span>
                <?php else: ?>
                  <span class="badge suspended">موقوف</span>
                <?php endif; ?>
              </td>
              <td><span class="badge coin"><?= (int)$r['main_balance'] ?> كوين</span></td>
              <td>
                <?php if ($usd_disp !== null): ?>
                  <span class="badge usd">$ <?= e($usd_disp) ?></span>
                <?php else: ?>
                  <span class="badge usd" style="opacity:.6">$ —</span>
                <?php endif; ?>
              </td>
              <td title="<?= e($r['created_at'] ?? '') ?>"><?= fmt_date($r['created_at'] ?? '') ?></td>
              <td>
                <div style="font-size:12px; opacity:.9;">
                  <div><?= fmt_date($r['last_login_at'] ?? '') ?></div>
                  <?php $ip = fmt_ip($r['last_login_ip'] ?? null); if ($ip): ?>
                    <div style="font-family:monospace;"><?= e($ip) ?></div>
                  <?php endif; ?>
                </div>
              </td>
              <td class="actions">
                <a class="btn sm secondary" href="agents_edit.php?id=<?= (int)$r['id'] ?>">تعديل</a>
                <a class="btn sm" href="agents_credit.php?id=<?= (int)$r['id'] ?>">شحن/خصم</a>

                <form method="post" onsubmit="return confirm('تغيير حالة هذا الوكيل؟');">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                  <input type="hidden" name="action" value="toggle_status">
                  <button class="btn sm" type="submit" title="تبديل الحالة">
                    <?= $r['status']==='active' ? 'إيقاف' : 'تنشيط' ?>
                  </button>
                </form>

                <form method="post" onsubmit="return confirm('حذف نهائي؟ لا يمكن التراجع. سيُمنع الحذف إذا كان الرصيد غير صفري.');">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                  <input type="hidden" name="action" value="delete_agent">
                  <button class="btn sm" type="submit" style="background:var(--danger);">حذف</button>
                </form>
              </td>
            </tr>
          <?php endforeach; ?>
          <?php if (!$rows): ?>
            <tr><td colspan="9" style="text-align:center; opacity:.8;">لا توجد بيانات</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>

    <?php if ($pages > 1): ?>
      <div style="display:flex; gap:8px; margin-top:12px; flex-wrap: wrap;">
        <?php for($p=1;$p<=$pages;$p++):
          $url = 'agents.php?page='.$p.($q!=='' ? '&q='.urlencode($q) : '');
          $is = $p===$page; ?>
          <a href="<?= e($url) ?>" class="btn sm <?= $is ? '' : 'secondary' ?>"<?= $is?' style="pointer-events:none; opacity:.8;"':''; ?>><?= $p ?></a>
        <?php endfor; ?>
      </div>
    <?php endif; ?>
  </div>
</div>

<?php require __DIR__ . '/inc/footer.php'; ?>
