<?php
// voicechat/api/user_public.php
require_once __DIR__ . '/config.php';

// ============ Helpers محلية خفيفة ============

/** هل الجدول يحتوي على عمود معيّن؟ */
function table_has_col(PDO $pdo, string $table, string $col): bool {
  try {
    $st = $pdo->prepare("SHOW COLUMNS FROM `$table` LIKE ?");
    $st->execute([$col]);
    return (bool)$st->fetchColumn();
  } catch (Throwable $e) { return false; }
}

/**
 * تحويل رابط نسبي إلى مطلق + https
 * - إن كان الرابط يبدأ بـ http/https يُعاد كما هو مع إجبار https.
 * - إن كان نسبيًا، يُسبق بـ BASE_URL إن كانت معرفة في config.php وإلا يُستنتج من الطلب.
 */
function absolute_url(?string $raw): ?string {
  $s = trim((string)$raw);
  if ($s === '') return null;

  if (stripos($s, 'http://') === 0 || stripos($s, 'https://') === 0) {
    return preg_replace('#^http://#i', 'https://', $s);
  }

  $base = defined('BASE_URL') ? rtrim((string)BASE_URL, '/') : null;
  if (!$base) {
    $proto = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    if (!empty($_SERVER['HTTP_X_FORWARDED_PROTO'])) $proto = $_SERVER['HTTP_X_FORWARDED_PROTO'];
    $host = $_SERVER['HTTP_X_FORWARDED_HOST'] ?? ($_SERVER['HTTP_HOST'] ?? 'localhost');
    $base = $proto . '://' . $host;
  }
  if ($s[0] !== '/') $s = '/' . $s;
  return $base . $s;
}

/** اختيار أعمدة users ديناميكيًا (حسب وجود الأعمدة) */
function select_user_cols(PDO $pdo): string {
  $cols = [
    'id',
    'COALESCE(userid,"")     AS userid',
    'COALESCE(username,"")   AS username',
    'COALESCE(avatar_url,"") AS avatar_url',
  ];
  if (table_has_col($pdo, 'users', 'cover_url')) {
    $cols[] = 'COALESCE(cover_url,"")  AS cover_url';
  }
  if (table_has_col($pdo, 'users', 'bio')) {
    $cols[] = 'COALESCE(bio,"")        AS bio';
  }
  // 👇 دعم الإطار مع أسماء أعمدة بديلة — كلّها تُعاد بالاسم frame_pag_url
  if (table_has_col($pdo, 'users', 'frame_pag_url')) {
    $cols[] = 'COALESCE(frame_pag_url,"") AS frame_pag_url';
  } elseif (table_has_col($pdo, 'users', 'frame_url')) {
    $cols[] = 'COALESCE(frame_url,"")     AS frame_pag_url';
  } elseif (table_has_col($pdo, 'users', 'frame')) {
    $cols[] = 'COALESCE(frame,"")         AS frame_pag_url';
  } elseif (table_has_col($pdo, 'users', 'ring_url')) {
    $cols[] = 'COALESCE(ring_url,"")      AS frame_pag_url';
  }
  return implode(",\n      ", $cols);
}

/** توحيد شكل كائن المستخدم + إصلاح الروابط + مفاتيح مضمونة */
function normalize_user_row(array $row): array {
  $u = [
    'id'            => isset($row['id']) ? (int)$row['id'] : null,
    'userid'        => (string)($row['userid']        ?? ''),
    'username'      => (string)($row['username']      ?? ''),
    'avatar_url'    => (string)($row['avatar_url']    ?? ''),
    'cover_url'     => (string)($row['cover_url']     ?? ''),
    'bio'           => (string)($row['bio']           ?? ''),
    'frame_pag_url' => (string)($row['frame_pag_url'] ?? ''), // قد تأتي من أعمدة بديلة كما في select_user_cols
  ];

  // إصلاح الروابط إلى مطلقة/https
  $u['avatar_url']    = absolute_url($u['avatar_url']);
  $u['cover_url']     = absolute_url($u['cover_url']);
  $u['frame_pag_url'] = absolute_url($u['frame_pag_url']);

  return $u;
}

function fetch_user_by_id(PDO $pdo, int $id): ?array {
  $cols = select_user_cols($pdo);
  $st = $pdo->prepare("
    SELECT
      $cols
    FROM users
    WHERE id = ?
    LIMIT 1
  ");
  $st->execute([$id]);
  $row = $st->fetch();
  return $row ? normalize_user_row($row) : null;
}

function fetch_user_by_uid_text(PDO $pdo, string $uidText): ?array {
  // نجرب فقط لو عندنا عمود uid نصّي
  if (!table_has_col($pdo, 'users', 'uid')) return null;
  $cols = select_user_cols($pdo);
  $st = $pdo->prepare("
    SELECT
      $cols
    FROM users
    WHERE uid = ?
    LIMIT 1
  ");
  $st->execute([$uidText]);
  $row = $st->fetch();
  return $row ? normalize_user_row($row) : null;
}

function fetch_user_by_userid(PDO $pdo, string $code): ?array {
  $cols = select_user_cols($pdo);
  $st = $pdo->prepare("
    SELECT
      $cols
    FROM users
    WHERE userid = ?
    LIMIT 1
  ");
  $st->execute([$code]);
  $row = $st->fetch();
  return $row ? normalize_user_row($row) : null;
}

// ============ التنفيذ ============

try {
  $pdo = db();

  // نقرأ كل الأسماء الشائعة للبارامترات
  $idParam     = trim((string)($_GET['id'] ?? $_POST['id'] ?? $_GET['user_id'] ?? $_POST['user_id'] ?? ''));
  $uidParam    = trim((string)($_GET['uid'] ?? $_POST['uid'] ?? ''));         // قد يكون id رقمي أو uid نصّي
  $useridParam = trim((string)($_GET['userid'] ?? $_POST['userid'] ?? ''));

  // لو مفيش ولا حاجة → رجِّع "أنا" لو فيه توكن
  if ($idParam === '' && $uidParam === '' && $useridParam === '') {
    $me = bearer_user_id(); // من config.php
    if ($me <= 0) {
      json_out(['ok'=>false,'error'=>'missing_uid'], 400);
    }
    $u = fetch_user_by_id($pdo, (int)$me);
    if ($u) json_out(['ok'=>true,'user'=>$u]);
    json_out(['ok'=>false,'error'=>'not_found'], 404);
  }

  // 1) أولوية id/user_id إن كان رقمًا صالحًا
  if ($idParam !== '' && ctype_digit($idParam)) {
    $u = fetch_user_by_id($pdo, (int)$idParam);
    if ($u) json_out(['ok'=>true,'user'=>$u]);
  }

  // 2) uidParam: لو رقم نتعامل معه كـ id، لو نص نجرب عمود uid
  if ($uidParam !== '') {
    if (ctype_digit($uidParam)) {
      $u = fetch_user_by_id($pdo, (int)$uidParam);
      if ($u) json_out(['ok'=>true,'user'=>$u]);
    } else {
      $u = fetch_user_by_uid_text($pdo, $uidParam);
      if ($u) json_out(['ok'=>true,'user'=>$u]);
    }
  }

  // 3) userid (الكود)
  if ($useridParam !== '') {
    $u = fetch_user_by_userid($pdo, $useridParam);
    if ($u) json_out(['ok'=>true,'user'=>$u]);
  }

  // لم نجد شيئًا
  json_out(['ok'=>false,'error'=>'not_found'], 404);

} catch (Throwable $e) {
  json_out(['ok'=>false,'error'=>'server_error','msg'=>$e->getMessage()], 500);
}
