<?php
declare(strict_types=1);

require __DIR__ . '/../config.php';
require_once __DIR__ . '/../vendor/autoload.php';

use Firebase\JWT\JWT;
use Firebase\JWT\Key;

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Authorization, Content-Type');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }

if (function_exists('opcache_invalidate')) { @opcache_invalidate(__FILE__, true); }

function json_fail($e,$c=400){ http_response_code($c); echo json_encode(['ok'=>false,'error'=>$e], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES); exit; }
function json_ok($d=[]){ echo json_encode(['ok'=>true] + $d, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES); exit; }
function bearer(): ?string {
  $h = $_SERVER['HTTP_AUTHORIZATION'] ?? $_SERVER['Authorization'] ?? $_SERVER['REDIRECT_HTTP_AUTHORIZATION'] ?? '';
  if (!$h && function_exists('getallheaders')) {
    foreach (getallheaders() as $k => $v) { if (strcasecmp($k,'Authorization')===0){ $h=$v; break; } }
  }
  return stripos($h, 'Bearer ') === 0 ? trim(substr($h, 7)) : null;
}
function input_all(): array {
  $ct = $_SERVER['CONTENT_TYPE'] ?? $_SERVER['HTTP_CONTENT_TYPE'] ?? '';
  if (stripos($ct,'application/json')!==false) {
    $raw = file_get_contents('php://input') ?: '';
    $j = json_decode($raw, true);
    if (is_array($j)) return $j + $_POST + $_GET;
  }
  return $_POST + $_GET;
}
function auth_user_id(PDO $pdo, object $payload): int {
  $id = 0;
  foreach (['id','user_id','uid'] as $k) {
    if (isset($payload->$k) && is_numeric($payload->$k)) { $id = (int)$payload->$k; break; }
  }
  if ($id > 0) return $id;
  $fuid = (string)($payload->firebase_uid ?? '');
  if ($fuid !== '') {
    $st = $pdo->prepare("SELECT id FROM users WHERE uid = ? LIMIT 1");
    $st->execute([$fuid]);
    $id = (int)$st->fetchColumn();
  }
  return $id;
}

$jwt = bearer(); if (!$jwt) json_fail('missing authorization bearer token', 401);
try {
  $payload = JWT::decode($jwt, new Key($JWT_SECRET, 'HS256'));
} catch (Throwable $e) {
  json_fail('invalid token: '.$e->getMessage(), 401);
}

$pdo = db();
$uid = auth_user_id($pdo, $payload);
if ($uid <= 0) json_fail('user_not_found', 401);

// ====== المدخلات ======
$in       = input_all();
$threadId = (int)($in['thread_id'] ?? $in['thread'] ?? 0);

$textRaw = $in['message_text'] ?? $in['message'] ?? $in['text'] ?? $in['body'] ?? $in['content'] ?? '';
$text    = trim((string)$textRaw);

$attachment = trim((string)($in['attachment_url'] ?? $in['attachment'] ?? $in['file_url'] ?? ''));

// لا تقبل رسالة فارغة تمامًا
if ($text === '' && $attachment === '') json_fail('empty_message');

try {
  $pdo->beginTransaction();

  // ✅ شات واحد لكل مستخدم: استخدم آخر محادثة له، أو أنشئ واحدة إذا لا يوجد أي محادثة
  if ($threadId <= 0) {
    $st = $pdo->prepare("SELECT id FROM support_threads WHERE user_id = ? ORDER BY updated_at DESC, id DESC LIMIT 1");
    $st->execute([$uid]);
    $threadId = (int)$st->fetchColumn();

    if ($threadId <= 0) {
      $subject = 'محادثة دعم';
      $pdo->prepare("INSERT INTO support_threads (user_id, subject, status, created_at, updated_at) VALUES (?, ?, 'open', NOW(), NOW())")
          ->execute([$uid, $subject]);
      $threadId = (int)$pdo->lastInsertId();
    }
  } else {
    // أمان: تأكد أن المحادثة تخص المستخدم نفسه
    $st = $pdo->prepare("SELECT id FROM support_threads WHERE id = ? AND user_id = ? LIMIT 1");
    $st->execute([$threadId, $uid]);
    if (!$st->fetchColumn()) { throw new RuntimeException('thread_not_found'); }
  }

  // إضافة الرسالة
  $pdo->prepare("
    INSERT INTO support_messages
      (thread_id, sender_type, user_id, message_text, attachment_url, created_at, read_by_user, read_by_staff)
    VALUES
      (?, 'user', ?, ?, ?, NOW(), 1, 0)
  ")->execute([$threadId, $uid, $text, $attachment !== '' ? $attachment : null]);

  $messageId = (int)$pdo->lastInsertId();

  // تحديث بيانات المحادثة
  $pdo->prepare("UPDATE support_threads SET status='open', last_user_msg_at=NOW(), updated_at=NOW() WHERE id=?")
      ->execute([$threadId]);

  $pdo->commit();

  json_ok([
    'thread_id'  => $threadId,
    'message_id' => $messageId,
    'message'    => [
      'id' => $messageId,
      'sender_type' => 'user',
      'message_text'=> $text,
      'attachment_url' => ($attachment !== '' ? $attachment : null),
      'created_at' => date('Y-m-d H:i:s'),
    ],
  ]);

} catch (Throwable $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  if ($e instanceof RuntimeException && $e->getMessage() === 'thread_not_found') {
    json_fail('thread_not_found', 404);
  }
  json_fail('db_error: '.$e->getMessage(), 500);
}
