<?php
// admin/support.php
declare(strict_types=1);

require_once __DIR__ . '/inc/auth.php';
require_login();

/* ---------------- CSRF ---------------- */
if (!isset($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
function csrf_token(): string { return $_SESSION['csrf']; }
function csrf_ok(?string $t): bool { return $t && hash_equals($_SESSION['csrf'], $t); }

/* ---------------- Helpers ---------------- */
if (!function_exists('e')) {
  function e(string $s): string { return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }
}
function admin_display_name(): string {
  return (string)($_SESSION['admin_name'] ?? $_SESSION['username'] ?? 'الدعم');
}

/* ---------------- Ajax API ---------------- */
$pdo = db();

if (isset($_GET['ajax']) && $_GET['ajax'] === '1') {
  header('Content-Type: application/json; charset=utf-8');
  $action = $_GET['action'] ?? $_POST['action'] ?? '';

  try {
    if ($action === 'list_threads') {
      $status = $_GET['status'] ?? 'open'; // open|closed|all
      $where  = "1=1";
      $params = [];
      if ($status === 'open')   $where .= " AND t.status='open'";
      if ($status === 'closed') $where .= " AND t.status='closed'";

      $q = trim($_GET['q'] ?? '');
      if ($q !== '') {
        $where .= " AND (t.subject LIKE ? OR u.username LIKE ? OR u.userid LIKE ?)";
        $kw = "%$q%"; array_push($params, $kw, $kw, $kw);
      }

      $sql = "
        SELECT
          t.id, t.user_id, t.subject, t.status, t.updated_at,
          t.last_user_msg_at, t.last_staff_msg_at,
          u.username, u.userid, u.avatar_url,
          (
            SELECT COUNT(*) FROM support_messages m
            WHERE m.thread_id=t.id AND m.sender_type='user' AND m.read_by_staff=0
          ) AS unread
        FROM support_threads t
        JOIN users u ON u.id=t.user_id
        WHERE $where
        ORDER BY (CASE WHEN t.status='open' THEN 0 ELSE 1 END), t.updated_at DESC, t.id DESC
        LIMIT 200";
      $st = $pdo->prepare($sql);
      $st->execute($params);
      $rows = $st->fetchAll(PDO::FETCH_ASSOC);

      echo json_encode(['ok'=>true,'threads'=>$rows], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
      exit;
    }

    if ($action === 'messages') {
      $threadId = (int)($_GET['thread_id'] ?? 0);
      $afterId  = (int)($_GET['after_id'] ?? 0);
      if ($threadId <= 0) { echo json_encode(['ok'=>false,'error'=>'missing thread_id']); exit; }

      if ($afterId > 0) {
        $st = $pdo->prepare("
          SELECT id, thread_id, sender_type, user_id, staff_name, message_text, attachment_url, created_at
          FROM support_messages
          WHERE thread_id=? AND id>?
          ORDER BY id ASC
          LIMIT 500
        ");
        $st->execute([$threadId, $afterId]);
      } else {
        $st = $pdo->prepare("
          SELECT id, thread_id, sender_type, user_id, staff_name, message_text, attachment_url, created_at
          FROM support_messages
          WHERE thread_id=?
          ORDER BY id ASC
          LIMIT 500
        ");
        $st->execute([$threadId]);
      }
      $msgs = $st->fetchAll(PDO::FETCH_ASSOC);

      // علّم رسائل المستخدم كمقروءة للطاقم
      $pdo->prepare("UPDATE support_messages SET read_by_staff=1 WHERE thread_id=? AND sender_type='user' AND read_by_staff=0")
          ->execute([$threadId]);

      echo json_encode(['ok'=>true,'messages'=>$msgs], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
      exit;
    }

    if ($action === 'send') {
      if (!csrf_ok($_POST['csrf'] ?? '')) { echo json_encode(['ok'=>false,'error'=>'bad_csrf']); exit; }
      $threadId = (int)($_POST['thread_id'] ?? 0);
      $body     = trim((string)($_POST['body'] ?? ''));
      if ($threadId <= 0) { echo json_encode(['ok'=>false,'error'=>'missing thread_id']); exit; }
      if ($body === '')   { echo json_encode(['ok'=>false,'error'=>'empty']); exit; }

      $exists = $pdo->prepare("SELECT id FROM support_threads WHERE id=? LIMIT 1");
      $exists->execute([$threadId]);
      if (!$exists->fetchColumn()) { echo json_encode(['ok'=>false,'error'=>'thread_not_found']); exit; }

      $staff = admin_display_name();
      $pdo->beginTransaction();
      $pdo->prepare("
        INSERT INTO support_messages
          (thread_id, sender_type, user_id, staff_name, message_text, created_at, read_by_user, read_by_staff)
        VALUES
          (?, 'staff', NULL, ?, ?, NOW(), 0, 1)
      ")->execute([$threadId, $staff, $body]);
      $msgId = (int)$pdo->lastInsertId();
      $pdo->prepare("UPDATE support_threads SET last_staff_msg_at=NOW(), updated_at=NOW(), status='open' WHERE id=?")
          ->execute([$threadId]);
      $pdo->commit();

      echo json_encode(['ok'=>true,'message'=>[
        'id'=>$msgId, 'thread_id'=>$threadId, 'sender_type'=>'staff', 'staff_name'=>$staff,
        'message_text'=>$body, 'created_at'=>date('Y-m-d H:i:s'),
      ]], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
      exit;
    }

    if ($action === 'toggle_status') {
      if (!csrf_ok($_POST['csrf'] ?? '')) { echo json_encode(['ok'=>false,'error'=>'bad_csrf']); exit; }
      $threadId = (int)($_POST['thread_id'] ?? 0);
      $new = $_POST['to'] ?? '';
      if (!in_array($new, ['open','closed'], true)) { echo json_encode(['ok'=>false,'error'=>'bad_status']); exit; }
      $pdo->prepare("UPDATE support_threads SET status=?, updated_at=NOW() WHERE id=?")->execute([$new, $threadId]);
      echo json_encode(['ok'=>true]); exit;
    }

    echo json_encode(['ok'=>false,'error'=>'unknown_action']); exit;
  } catch (Throwable $e) {
    echo json_encode(['ok'=>false,'error'=>'server_error']); exit;
  }
}

/* ---------------- Render ---------------- */
$PAGE_TITLE = "مركز دعم العملاء";
require __DIR__ . '/inc/header.php';
require __DIR__ . '/inc/sidebar.php';
?>
<style>
.support-wrap{display:flex;gap:12px;align-items:stretch;min-height:calc(100vh - 140px)}
.support-threads{width:340px;min-width:300px;max-width:420px;display:flex;flex-direction:column;gap:12px}
.support-threads .card{padding:0;overflow:hidden}
.support-threads .filters{display:flex;gap:8px;padding:10px;border-bottom:1px solid var(--border)}
.support-threads .list{max-height:calc(100vh - 220px);overflow:auto}
.thread-item{display:flex;gap:10px;padding:12px;border-bottom:1px solid var(--border);align-items:center;cursor:pointer;text-decoration:none;color:inherit}
.thread-item:hover{background:rgba(59,130,246,.08)}
.thread-item.active{background:rgba(59,130,246,.15)}
.thread-item .avatar{width:42px;height:42px;border-radius:50%;object-fit:cover;background:#eceff4;flex:0 0 42px}
.thread-item .meta{flex:1 1 auto;min-width:0}
.thread-item .meta .top{display:flex;align-items:center;gap:6px}
.thread-item .meta .name{font-weight:700}
.thread-item .meta .time{margin-inline-start:auto;font-size:12px;opacity:.7}
.thread-item .subject{font-size:12px;opacity:.9;white-space:nowrap;overflow:hidden;text-overflow:ellipsis}
.badge-dot{display:inline-block;min-width:20px;padding:2px 6px;text-align:center;border-radius:999px;background:#ef4444;color:#fff;font-size:11px}

.support-chat{flex:1 1 auto;display:flex;flex-direction:column;gap:12px}
.support-chat .chat-card{display:flex;flex-direction:column;overflow:hidden}
.chat-header{display:flex;align-items:center;justify-content:space-between;padding:12px 14px;border-bottom:1px solid var(--border)}
.chat-body{padding:14px;overflow:auto;height:calc(100vh - 280px);background:linear-gradient(180deg,rgba(255,255,255,.02),rgba(255,255,255,.0))}
.msg{max-width:70%;padding:10px 12px;border-radius:12px;margin:6px 0;position:relative;box-shadow:0 1px 3px rgba(0,0,0,.06)}
.msg.user{background:#fff;border:1px solid var(--border);margin-inline-start:auto}
.msg.staff{background:#eff6ff;border:1px solid #c7dbff;margin-inline-end:auto}
.msg .time{font-size:11px;opacity:.7;margin-top:4px}
.chat-input{border-top:1px solid var(--border);padding:12px;background:var(--panel);display:flex;gap:8px;align-items:flex-end}
.chat-input textarea{width:100%;resize:vertical;min-height:42px;max-height:160px;border:1px solid var(--border);border-radius:10px;padding:10px 12px;background:#0b1220;color:var(--text)}
</style>

<div class="support-wrap" dir="rtl">
  <!-- القائمة اليمنى -->
  <div class="support-threads">
    <div class="card">
      <div class="filters">
        <input class="input" id="q" placeholder="بحث باسم المستخدم / العنوان" style="flex:1" />
        <select id="status" class="input" style="width:140px">
          <option value="open">المفتوحة</option>
          <option value="all">الكل</option>
          <option value="closed">المغلقة</option>
        </select>
        <button class="btn sm" id="btn-refresh">تحديث</button>
      </div>
      <div id="threads" class="list">
        <div style="padding:12px;opacity:.7">جارٍ التحميل…</div>
      </div>
    </div>
  </div>

  <!-- منطقة الدردشة -->
  <div class="support-chat">
    <div class="card chat-card">
      <div class="chat-header">
        <div>
          <div id="chat-title" style="font-weight:700">اختر محادثة من القائمة</div>
          <div id="chat-sub" style="font-size:12px;opacity:.8"></div>
        </div>
        <div>
          <button id="btn-open" class="btn sm secondary" style="display:none" onclick="toggleStatus('open')">إعادة فتح</button>
          <button id="btn-close" class="btn sm" style="display:none;background:var(--danger)" onclick="toggleStatus('closed')">إغلاق</button>
        </div>
      </div>
      <div id="chat" class="chat-body"></div>
      <div class="chat-input">
        <textarea id="body" placeholder="اكتب ردّك للعميل…"></textarea>
        <button id="sendBtn" class="btn sm">إرسال</button>
      </div>
    </div>
  </div>
</div>

<script>
(function(){
  const CSRF   = "<?= e(csrf_token()) ?>";
  const AVATAR_FALLBACK = 'data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" width="42" height="42"><rect width="100%" height="100%" fill="%23e5e7eb"/></svg>';

  let CURRENT_ID = null;
  let LAST_MSG_ID = 0;
  let POLLER = null;

  const $ = (id)=>document.getElementById(id);
  const safe = (t)=>{ const d=document.createElement('div'); d.textContent=(t??''); return d.innerHTML; };

  $('btn-refresh').addEventListener('click', loadThreads);
  $('sendBtn').addEventListener('click', sendMsg);

  async function loadThreads(){
    const q = $('q').value.trim();
    const status = $('status').value;
    const box = $('threads');
    box.innerHTML = '<div style="padding:12px;opacity:.7">جارٍ التحميل…</div>';
    try{
      const res = await fetch(`support.php?ajax=1&action=list_threads&status=${encodeURIComponent(status)}&q=${encodeURIComponent(q)}`, {cache:'no-store'});
      const j = await res.json();
      if(!j.ok){ box.innerHTML = '<div style="padding:12px;opacity:.8">تعذّر التحميل</div>'; return; }
      if(!j.threads.length){ box.innerHTML = '<div style="padding:12px;opacity:.8">لا توجد محادثات</div>'; return; }

      box.innerHTML = '';
      j.threads.forEach(t=>{
        const a = document.createElement('a');
        a.className = 'thread-item' + (t.id===CURRENT_ID ? ' active':'');
        a.href = '#';
        a.addEventListener('click', (e)=>{ e.preventDefault(); openThread(t); });

        const img = document.createElement('img');
        img.className = 'avatar';
        img.src = t.avatar_url || AVATAR_FALLBACK;
        img.onerror = ()=>{ img.src = AVATAR_FALLBACK; };

        const meta = document.createElement('div');
        meta.className = 'meta';
        meta.innerHTML = `
          <div class="top">
            <div class="name">${safe(t.username || 'مستخدم')}</div>
            <div class="time">${safe((t.updated_at||'').slice(11,16))}</div>
          </div>
          <div class="subject">${safe(t.subject || '(بدون عنوان)')}</div>
        `;

        a.appendChild(img);
        a.appendChild(meta);
        if (t.unread > 0) {
          const b = document.createElement('span');
          b.className = 'badge-dot';
          b.textContent = t.unread;
          a.appendChild(b);
        }
        box.appendChild(a);
      });
    }catch(err){
      console.error(err);
      box.innerHTML = '<div style="padding:12px;opacity:.8">خطأ في الاتصال</div>';
    }
  }

  async function openThread(t){
    CURRENT_ID  = t.id;
    LAST_MSG_ID = 0;

    $('chat').innerHTML   = '';
    $('chat-title').textContent = (t.username || 'مستخدم') + ' — #' + t.id;
    $('chat-sub').textContent   = (t.subject || '');
    $('btn-open').style.display  = t.status === 'closed' ? 'inline-block' : 'none';
    $('btn-close').style.display = t.status !== 'closed' ? 'inline-block' : 'none';

    // أعد تمييز العنصر النشط
    document.querySelectorAll('.thread-item').forEach(el=>el.classList.remove('active'));
    const list = document.querySelectorAll('.thread-item');
    if (list.length) list[0].parentElement.querySelectorAll('.thread-item').forEach(el=>{
      // لا توجد هوية فريدة داخل العناصر، سنعيد تحميل القائمة بعد فتح الرسائل
    });

    await loadMessages();

    if (POLLER) clearInterval(POLLER);
    POLLER = setInterval(loadNew, 5000);

    // أعِد تحميل قائمة المواضيع لضبط الشارات/النشاط
    loadThreads();
  }

  function msgNode(m){
    const isStaff = (m.sender_type === 'staff');
    const div = document.createElement('div');
    div.className = 'msg ' + (isStaff ? 'staff' : 'user');
    const txt  = safe(m.message_text || '');
    const time = safe((m.created_at||'').slice(11,16));
    const who  = (isStaff && m.staff_name) ? ' · '+safe(m.staff_name) : '';
    div.innerHTML = `<div>${txt}</div><div class="time">${time}${who}</div>`;
    return div;
  }

  async function loadMessages(){
    if (!CURRENT_ID) return;
    try{
      const res = await fetch(`support.php?ajax=1&action=messages&thread_id=${CURRENT_ID}`, {cache:'no-store'});
      const j = await res.json();
      if (!j.ok){ return; }
      const box = $('chat');
      box.innerHTML = '';
      j.messages.forEach(m=>{
        box.appendChild(msgNode(m));
        if (m.id > LAST_MSG_ID) LAST_MSG_ID = m.id;
      });
      box.scrollTop = box.scrollHeight + 200;
    }catch(err){ console.error(err); }
  }

  async function loadNew(){
    if (!CURRENT_ID || !LAST_MSG_ID) return;
    try{
      const res = await fetch(`support.php?ajax=1&action=messages&thread_id=${CURRENT_ID}&after_id=${LAST_MSG_ID}`, {cache:'no-store'});
      const j = await res.json();
      if (!j.ok || !j.messages.length) return;
      const box = $('chat');
      j.messages.forEach(m=>{
        box.appendChild(msgNode(m));
        if (m.id > LAST_MSG_ID) LAST_MSG_ID = m.id;
      });
      box.scrollTop = box.scrollHeight + 200;
      loadThreads(); // لتحديث الشارات والترتيب
    }catch(err){ console.error(err); }
  }

  async function sendMsg(){
    if (!CURRENT_ID) return;
    const body = $('body').value.trim();
    if (!body) return;
    $('sendBtn').disabled = true;
    try{
      const form = new FormData();
      form.set('action','send');
      form.set('csrf', CSRF);
      form.set('thread_id', CURRENT_ID);
      form.set('body', body);
      const res = await fetch('support.php?ajax=1', { method:'POST', body: form });
      const j = await res.json();
      if (!j.ok){ alert('تعذّر الإرسال'); return; }
      $('body').value = '';
      await loadNew();
    }catch(err){
      console.error(err);
      alert('خطأ في الاتصال');
    }finally{
      $('sendBtn').disabled = false;
    }
  }

  window.toggleStatus = async function(to){
    if (!CURRENT_ID) return;
    const form = new FormData();
    form.set('action','toggle_status');
    form.set('csrf', CSRF);
    form.set('thread_id', CURRENT_ID);
    form.set('to', to);
    const res = await fetch('support.php?ajax=1', { method:'POST', body: form });
    const j = await res.json();
    if (j.ok){
      $('btn-open').style.display  = (to === 'open')   ? 'none' : 'inline-block';
      $('btn-close').style.display = (to === 'closed') ? 'none' : 'inline-block';
      loadThreads();
    }
  }

  // أول تحميل
  loadThreads();
})();
</script>

<?php require __DIR__ . '/inc/footer.php'; ?>
