<?php
// voicechat/api/rooms_my.php
declare(strict_types=1);

require __DIR__ . '/config.php';

$pdo   = db();

// يتحقق من التوكن ويعيد معرّف المستخدم (int) حسب config.php
$userId = require_user();

// نحاول جلب UID النصي للمستخدم (إن كان جدول users يحتويه)
$userUid = null;
try {
  $st = $pdo->prepare('SELECT uid FROM users WHERE id = ? LIMIT 1');
  $st->execute([$userId]);
  $u = $st->fetch();
  if ($u && isset($u['uid']) && $u['uid'] !== '' && $u['uid'] !== null) {
    $userUid = (string)$u['uid'];
  }
} catch (Throwable $e) {
  // بعض الجداول قد لا تحتوي عمود uid — نتجاهل
}

/**
 * دالة مساعدة: تنفيذ SELECT مع تجاهل "Unknown column" بهدوء
 */
function try_select_first(PDO $pdo, string $sql, array $params): ?array {
  try {
    $st = $pdo->prepare($sql);
    $st->execute($params);
    $row = $st->fetch(PDO::FETCH_ASSOC);
    return $row ?: null;
  } catch (Throwable $e) {
    // لو العمود غير موجود أو أي خطأ SQL؛ نعتبرها محاولة فاشلة ونجرّب غيرها
    return null;
  }
}

/**
 * نحاول إيجاد غرفة المستخدم وفق ترتيب الأولوية (open ثم الأحدث)
 */
$roomRow = null;

// 1) owner_id + مفتوحة
$roomRow = try_select_first($pdo, "SELECT * FROM rooms WHERE owner_id = ? AND status = 'open' ORDER BY id DESC LIMIT 1", [$userId]);

// 2) owner_uid + مفتوحة (لو عندنا userUid نصي)
if (!$roomRow && $userUid !== null) {
  $roomRow = try_select_first($pdo, "SELECT * FROM rooms WHERE owner_uid = ? AND status = 'open' ORDER BY id DESC LIMIT 1", [$userUid]);
}

// 3) owner_id بدون شرط الحالة (أحدث غرفة)
if (!$roomRow) {
  $roomRow = try_select_first($pdo, "SELECT * FROM rooms WHERE owner_id = ? ORDER BY id DESC LIMIT 1", [$userId]);
}

// 4) owner_uid بدون شرط الحالة (أحدث غرفة)
if (!$roomRow && $userUid !== null) {
  $roomRow = try_select_first($pdo, "SELECT * FROM rooms WHERE owner_uid = ? ORDER BY id DESC LIMIT 1", [$userUid]);
}

// لو ما في غرفة
if (!$roomRow) {
  json_out(['room' => null], 200);
}

// حساب عدد المتصلين (اختياري)
$onlineCount = 0;
try {
  // نحاول معرفة اسم العمود الخاص بـ UID الغرفة
  $roomUidCol = array_key_exists('uid', $roomRow) ? 'uid' :
                (array_key_exists('room_uid', $roomRow) ? 'room_uid' : null);
  if ($roomUidCol !== null) {
    $st2 = $pdo->prepare("SELECT COUNT(*) FROM room_users WHERE room_uid = ?");
    $st2->execute([$roomRow[$roomUidCol]]);
    $onlineCount = (int)$st2->fetchColumn();
  }
} catch (Throwable $e) {
  // جدول room_users قد لا يكون موجودًا — تجاهل
}

// تجهيز الحقول التي يتوقعها التطبيق
$roomUid = '';
if (isset($roomRow['uid'])) {
  $roomUid = (string)$roomRow['uid'];
} elseif (isset($roomRow['room_uid'])) {
  $roomUid = (string)$roomRow['room_uid'];
}

// تحديد owner_uid النصي الذي يحتاجه التطبيق للمقارنة
$ownerUidForJson = '';
if (isset($roomRow['owner_uid']) && $roomRow['owner_uid'] !== '') {
  $ownerUidForJson = (string)$roomRow['owner_uid'];
} elseif ($userUid !== null) {
  $ownerUidForJson = $userUid; // نستعمل uid من جدول users
} else {
  $ownerUidForJson = (string)$userId; // fallback
}

// بناء الاستجابة
$room = [
  'id'           => isset($roomRow['id']) ? (int)$roomRow['id'] : 0,
  'uid'          => $roomUid,
  'name'         => (string)($roomRow['name'] ?? ''),
  'description'  => (string)($roomRow['description'] ?? ''),
  'owner_uid'    => $ownerUidForJson,
  // نُدرج owner_id إن كان موجودًا (قد لا يستخدمه التطبيق، لكنه مفيد):
  'owner_id'     => isset($roomRow['owner_id']) ? (int)$roomRow['owner_id'] : $userId,
  'image_url'    => $roomRow['image_url'] ?? null,
  'status'       => (string)($roomRow['status'] ?? 'open'),
  'is_private'   => (int)($roomRow['is_private'] ?? 0),
  'online_count' => $onlineCount,
];

// يمكنك إضافة حقول أخرى موجودة في جدولك (مثل: created_at، category_id…)
json_out(['room' => $room], 200);
