<?php
// backend/api/room_ban.php
declare(strict_types=1);
require_once __DIR__ . '/config.php';
header('Content-Type: application/json; charset=utf-8');

/* ====== DEBUG SWITCH (يفيد بالمتصفح) ====== */
$__DEBUG = isset($_GET['debug']) && $_GET['debug'] === '1';
if ($__DEBUG) {
  ini_set('display_errors', '1');
  error_reporting(E_ALL);
}

/* التزم بالـ POST */
if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST' && !$__DEBUG) {
  json_out(['ok'=>false,'error'=>'method_not_allowed'], 405);
}

/* Helpers */
function in_json(): array {
  $raw = file_get_contents('php://input');
  if ($raw === false || $raw === '') return [];
  $d = json_decode($raw, true);
  return is_array($d) ? $d : [];
}
function assert_room_exists(PDO $pdo, string $room_uid): void {
  $st = $pdo->prepare("SELECT 1 FROM rooms WHERE uid=? LIMIT 1");
  $st->execute([$room_uid]);
  if (!$st->fetchColumn()) json_out(['ok'=>false,'error'=>'room_not_found'], 404);
}
function create_bans_table_if_needed(PDO $pdo): void {
  try { $pdo->query("SELECT 1 FROM room_bans LIMIT 1"); }
  catch (\Throwable $e) {
    $sql = "
    CREATE TABLE IF NOT EXISTS room_bans (
      id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
      room_uid      VARCHAR(64) NOT NULL,
      target_uid    VARCHAR(64) NOT NULL,
      banned_by     VARCHAR(64) NOT NULL,
      is_permanent  TINYINT(1) NOT NULL DEFAULT 0,
      banned_until  DATETIME    NOT NULL,
      reason        TEXT NULL,
      created_at    DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
      PRIMARY KEY (id),
      UNIQUE KEY uniq_room_target (room_uid, target_uid),
      KEY idx_room (room_uid),
      KEY idx_target (target_uid)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";
    $pdo->exec($sql);
  }
}

/* === NEW: تأكد من وجود جدول الأدمن (اختياري لكن مفيد) === */
function create_admins_table_if_needed(PDO $pdo): void {
  try { $pdo->query("SELECT 1 FROM room_admins LIMIT 1"); }
  catch (\Throwable $e) {
    $sql = "
    CREATE TABLE IF NOT EXISTS room_admins (
      id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
      room_uid   VARCHAR(64) NOT NULL,
      admin_uid  VARCHAR(64) NOT NULL,
      created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
      PRIMARY KEY (id),
      UNIQUE KEY uniq_room_admin (room_uid, admin_uid),
      KEY idx_room (room_uid),
      KEY idx_admin (admin_uid)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";
    $pdo->exec($sql);
  }
}

function is_room_owner(PDO $pdo, string $room_uid, string $actor_uid): bool {
  $st = $pdo->prepare("SELECT owner_uid FROM rooms WHERE uid=? LIMIT 1");
  $st->execute([$room_uid]);
  $owner = (string)($st->fetchColumn() ?: '');
  if ($owner === '') return false;
  if ($owner === $actor_uid) return true;

  // الممثل داخلي → طابق مع supabase uid المحتمل
  try {
    $st = $pdo->prepare("
      SELECT COALESCE(supabase_uid, supa_uid, sb_uid, auth_uid, external_uid)
      FROM users WHERE uid=? LIMIT 1
    ");
    $st->execute([$actor_uid]);
    $actorSupa = (string)($st->fetchColumn() ?: '');
    if ($actorSupa !== '' && $owner === $actorSupa) return true;
  } catch (\Throwable $e) {}

  // الممثل قد يكون supabase uid → طابق مع uid الداخلي
  try {
    $st = $pdo->prepare("
      SELECT uid FROM users
      WHERE supabase_uid=? OR supa_uid=? OR sb_uid=? OR auth_uid=? OR external_uid=?
      LIMIT 1
    ");
    $st->execute([$actor_uid, $actor_uid, $actor_uid, $actor_uid, $actor_uid]);
    $actorLocal = (string)($st->fetchColumn() ?: '');
    if ($actorLocal !== '' && $owner === $actorLocal) return true;
  } catch (\Throwable $e) {}

  return false;
}

/* === NEW: فاحص هل المستخدم أدمن في هذه الغرفة === */
function is_room_admin(PDO $pdo, string $room_uid, string $uid): bool {
  // فحص مباشر
  $st = $pdo->prepare("SELECT 1 FROM room_admins WHERE room_uid=? AND admin_uid=? LIMIT 1");
  $st->execute([$room_uid, $uid]);
  if ($st->fetchColumn()) return true;

  // uid داخلي وربما له supabase_uid
  try {
    $st = $pdo->prepare("
      SELECT COALESCE(supabase_uid, supa_uid, sb_uid, auth_uid, external_uid)
      FROM users WHERE uid=? LIMIT 1
    ");
    $st->execute([$uid]);
    $supa = (string)($st->fetchColumn() ?: '');
    if ($supa !== '') {
      $st = $pdo->prepare("SELECT 1 FROM room_admins WHERE room_uid=? AND admin_uid=? LIMIT 1");
      $st->execute([$room_uid, $supa]);
      if ($st->fetchColumn()) return true;
    }
  } catch (\Throwable $e) {}

  // uid قد يكون supabase_uid ويحتاج تحويله لـ uid داخلي
  try {
    $st = $pdo->prepare("
      SELECT uid FROM users
      WHERE supabase_uid=? OR supa_uid=? OR sb_uid=? OR auth_uid=? OR external_uid=?
      LIMIT 1
    ");
    $st->execute([$uid, $uid, $uid, $uid, $uid]);
    $local = (string)($st->fetchColumn() ?: '');
    if ($local !== '') {
      $st = $pdo->prepare("SELECT 1 FROM room_admins WHERE room_uid=? AND admin_uid=? LIMIT 1");
      $st->execute([$room_uid, $local]);
      if ($st->fetchColumn()) return true;
    }
  } catch (\Throwable $e) {}

  return false;
}

function force_kick_member(PDO $pdo, string $room_uid, string $target_uid): void {
  // حرر المايك + كرسي
  $trySqlsDel = [
    "DELETE FROM room_mics WHERE room_uid=? AND user_uid=?",
  ];
  $trySqlsUpd = [
    "UPDATE room_mics SET user_uid=NULL, username=NULL, avatar_url=NULL WHERE room_uid=? AND user_uid=?",
    "UPDATE room_mic_state SET user_uid=NULL, username=NULL, avatar_url=NULL WHERE room_uid=? AND user_uid=?",
    "UPDATE room_seats SET user_uid=NULL, username=NULL, avatar_url=NULL, taken=0 WHERE room_uid=? AND user_uid=?",
  ];
  foreach ($trySqlsDel as $sql) { try { $st=$pdo->prepare($sql); $st->execute([$room_uid,$target_uid]); } catch (\Throwable $e) {} }
  foreach ($trySqlsUpd as $sql) { try { $st=$pdo->prepare($sql); $st->execute([$room_uid,$target_uid]); } catch (\Throwable $e) {} }

  // أخرج العضو من الغرفة
  try {
    $st=$pdo->prepare("DELETE FROM room_members WHERE room_uid=? AND user_uid=?");
    $st->execute([$room_uid,$target_uid]);
  } catch (\Throwable $e) {}
}

/* ====== Main ====== */
try {
  $pdo = db();
  try { $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION); } catch (\Throwable $e) {}

  // مصادقة
  $auth = require_auth();
  $actorUid = (string)($auth['uid'] ?? '');
  if ($actorUid === '') json_out(['ok'=>false,'error'=>'unauthorized'], 401);

  // بارس
  $in = in_json();
  if ($__DEBUG && empty($in)) {
    $in = [
      'room_uid'   => $_GET['room_uid']   ?? '',
      'target_uid' => $_GET['target_uid'] ?? '',
      'type'       => $_GET['type']       ?? 'temp', // temp|permanent
      'minutes'    => (int)($_GET['minutes'] ?? 15),
      'reason'     => $_GET['reason']     ?? '',
    ];
  }

  $room_uid   = trim((string)($in['room_uid'] ?? ''));
  $target_uid = trim((string)($in['target_uid'] ?? ''));
  $type       = strtolower(trim((string)($in['type'] ?? 'temp'))); // temp|permanent
  $minutes    = (int)($in['minutes'] ?? 15);
  $reason     = trim((string)($in['reason'] ?? ''));

  if ($room_uid === '' || $target_uid === '') {
    json_out(['ok'=>false,'error'=>'missing_params'], 422);
  }
  assert_room_exists($pdo, $room_uid);

  if ($target_uid === $actorUid) {
    json_out(['ok'=>false,'error'=>'cannot_ban_self'], 400);
  }

  // تأكد من الجداول المطلوبة
  create_bans_table_if_needed($pdo);
  create_admins_table_if_needed($pdo); // NEW

  // صلاحيات: مالك أو أدمن
  $actorIsOwner = is_room_owner($pdo, $room_uid, $actorUid);
  $actorIsAdmin = !$actorIsOwner && is_room_admin($pdo, $room_uid, $actorUid);
  if (!$actorIsOwner && !$actorIsAdmin) {
    json_out(['ok'=>false,'error'=>'forbidden','hint'=>'owner_or_admin_only'], 403);
  }

  // لا يمكن طرد المالك بأي حال
  if (is_room_owner($pdo, $room_uid, $target_uid)) {
    json_out(['ok'=>false,'error'=>'cannot_ban_owner'], 400);
  }

  // لا يمكن للأدمن طرد أدمن مثله (المالك فقط يطرد الأدمن)
  if (!$actorIsOwner && is_room_admin($pdo, $room_uid, $target_uid)) {
    json_out(['ok'=>false,'error'=>'cannot_ban_admin'], 400);
  }

  $isPermanent = ($type === 'permanent');
  if (!$isPermanent) {
    if ($minutes <= 0 || $minutes > 1440) $minutes = 15;
  }

  // banned_until: قيمة بعيدة للطرد الدائم (لتفادي NOT NULL)
  if ($isPermanent) {
    $bannedUntil = '2099-12-31 23:59:59';
  } else {
    $minutes = (int)$minutes;
    $q = $pdo->query("SELECT DATE_FORMAT(NOW() + INTERVAL $minutes MINUTE, '%Y-%m-%d %H:%i:%s')");
    $bannedUntil = (string)$q->fetchColumn();
  }

  $pdo->beginTransaction();

  // UPDATE ثم INSERT
  $st = $pdo->prepare("
    UPDATE room_bans
       SET banned_by=?, is_permanent=?, banned_until=?, reason=?
     WHERE room_uid=? AND target_uid=?
  ");
  $st->execute([
    $actorUid,
    $isPermanent ? 1 : 0,
    $bannedUntil,
    ($reason !== '' ? $reason : null),
    $room_uid,
    $target_uid
  ]);

  if ($st->rowCount() === 0) {
    $st = $pdo->prepare("
      INSERT INTO room_bans
        (room_uid, target_uid, banned_by, is_permanent, banned_until, reason, created_at)
      VALUES (?,?,?,?,?,?, NOW())
    ");
    $st->execute([
      $room_uid,
      $target_uid,
      $actorUid,
      $isPermanent ? 1 : 0,
      $bannedUntil,
      ($reason !== '' ? $reason : null)
    ]);
  }

  // طرد فوري من الغرفة + تحرير المايك
  force_kick_member($pdo, $room_uid, $target_uid);

  $pdo->commit();

  json_out([
    'ok'    => true,
    'by'    => $actorIsOwner ? 'owner' : 'admin',
    'type'  => $isPermanent ? 'permanent' : 'temp',
    'until' => $bannedUntil,
  ]);

} catch (\Throwable $e) {
  if (isset($pdo) && $pdo instanceof PDO && $pdo->inTransaction()) {
    $pdo->rollBack();
  }
  if ($__DEBUG) {
    header('Content-Type: text/plain; charset=utf-8');
    echo "ERROR 500\n";
    echo $e->getMessage() . "\n\n";
    echo $e->getTraceAsString();
    exit;
  }
  json_out(['ok'=>false, 'error'=>'server_error', 'msg'=>$e->getMessage()], 500);
}
