<?php
// voicechat/api/request.php
require_once __DIR__ . '/config.php';

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Authorization, Content-Type');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }

$pdo = db();

/* ======================= Helpers ======================= */

// هل يملك جدول users عمود uid؟
function users_has_uid_col(PDO $pdo): bool {
  try {
    $st = $pdo->prepare("SHOW COLUMNS FROM `users` LIKE 'uid'");
    $st->execute();
    return (bool)$st->fetch();
  } catch (Throwable $e) { return false; }
}

// يرجع users.id من التوكن؛ يقبل id مباشرة أو يبحث بـ uid النصّي
function require_user_id(PDO $pdo): int {
  try {
    $claims = require_auth();                 // من config.php
  } catch (Throwable $e) {
    json_out(['ok'=>false,'error'=>'auth_required'], 401);
  }

  // إن كان الـ JWT يحمل id صريحًا
  if (!empty($claims['id']) && (int)$claims['id'] > 0) {
    return (int)$claims['id'];
  }

  // fallback: لو يحمل uid نصّي (Firebase/uid) حاول تحويله إلى users.id
  if (!empty($claims['uid'])) {
    $uid = (string)$claims['uid'];

    // لو uid يبدو رقمًا، جرّبه كـ id أولًا
    if (ctype_digit($uid)) {
      $st = $pdo->prepare('SELECT id FROM users WHERE id=? LIMIT 1');
      $st->execute([$uid]);
      $id = (int)$st->fetchColumn();
      if ($id > 0) return $id;
    }

    // وإلا إن كان لدينا عمود uid نصّي
    if (users_has_uid_col($pdo)) {
      $st = $pdo->prepare('SELECT id FROM users WHERE uid=? LIMIT 1');
      $st->execute([$uid]);
      $id = (int)$st->fetchColumn();
      if ($id > 0) return $id;
    }
  }

  json_out(['ok'=>false,'error'=>'auth_bad_uid'], 401);
  return 0; // لن يصل
}

/** يحوّل أي peer_uid إلى users.id (يقبل id أو userid أو uid النصّي) */
function resolve_user_id(PDO $pdo, $raw) : int {
  if ($raw === null) return 0;
  $s = trim((string)$raw);
  if ($s === '') return 0;

  // 1) جرّبه كـ id
  if (ctype_digit($s)) {
    $q = $pdo->prepare('SELECT id FROM users WHERE id=? LIMIT 1');
    $q->execute([$s]);
    $id = (int)$q->fetchColumn();
    if ($id > 0) return $id;
  }

  // 2) userid (الكود العام)
  $q = $pdo->prepare('SELECT id FROM users WHERE userid=? LIMIT 1');
  $q->execute([$s]);
  $id = (int)$q->fetchColumn();
  if ($id > 0) return $id;

  // 3) uid النصّي (إن وُجد العمود)
  if (users_has_uid_col($pdo)) {
    $q = $pdo->prepare('SELECT id FROM users WHERE uid=? LIMIT 1');
    $q->execute([$s]);
    $id = (int)$q->fetchColumn();
    if ($id > 0) return $id;
  }

  return 0;
}

function other_id($row, $me) {
  return ((int)$row['uid1'] === (int)$me) ? (int)$row['uid2'] : (int)$row['uid1'];
}

/* ======================= Auth ======================= */

$me = require_user_id($pdo);  // دائمًا users.id

/* ======================= Routing ======================= */

$path = trim($_SERVER['PATH_INFO'] ?? '', '/');
if ($path === '') $path = $_GET['action'] ?? '';

try {
  switch ($path) {

    // ---------------------------------------------
    // GET /request.php/status?peer_uid=...
    // يرجّع: none | outgoing_pending | incoming_pending | friends
    // ---------------------------------------------
    case 'status': {
      $peerRaw = $_GET['peer_uid'] ?? null;
      $peer    = resolve_user_id($pdo, $peerRaw);
      if ($peer <= 0 || $peer === $me) json_out(['ok'=>false,'error'=>'bad_peer'], 400);

      $st = $pdo->prepare("
        SELECT * FROM friends
        WHERE (uid1=:a AND uid2=:b) OR (uid1=:b AND uid2=:a)
        ORDER BY id DESC
        LIMIT 1
      ");
      $st->execute([':a'=>$me, ':b'=>$peer]);
      $row = $st->fetch();

      $state = 'none';
      if ($row) {
        if ($row['status'] === 'accepted') {
          $state = 'friends';
        } elseif ($row['status'] === 'pending') {
          $state = ((int)$row['requested_by'] === $me) ? 'outgoing_pending' : 'incoming_pending';
        }
      }

      json_out(['ok'=>true,'state'=>$state]);
    }

    // ---------------------------------------------
    // POST /request.php/request   body: { peer_uid }
    // ---------------------------------------------
    case 'request': {
      $in   = json_in();
      $peer = resolve_user_id($pdo, $in['peer_uid'] ?? null);
      if ($peer <= 0 || $peer === $me) json_out(['ok'=>false,'error'=>'bad_peer'], 400);

      $st = $pdo->prepare("
        SELECT * FROM friends
        WHERE (uid1=:a AND uid2=:b) OR (uid1=:b AND uid2=:a)
        ORDER BY id DESC
        LIMIT 1
      ");
      $st->execute([':a'=>$me, ':b'=>$peer]);
      $row = $st->fetch();

      if ($row) {
        if ($row['status'] === 'accepted') {
          json_out(['ok'=>true,'state'=>'friends']);
        } elseif ($row['status'] === 'pending') {
          if ((int)$row['requested_by'] === $me) {
            json_out(['ok'=>true,'state'=>'outgoing_pending']);
          } else {
            json_out(['ok'=>true,'state'=>'incoming_pending']);
          }
        } else {
          // مُلغى/مرفوض → أعد استخدام السجل
          $st = $pdo->prepare("
            UPDATE friends
            SET uid1=:me, uid2=:peer, status='pending',
                requested_by=:me, updated_at=NOW()
            WHERE id=:id
          ");
          $st->execute([':me'=>$me, ':peer'=>$peer, ':id'=>$row['id']]);
          json_out(['ok'=>true,'state'=>'outgoing_pending']);
        }
      }

      // لا يوجد سجل: أنشئ واحدًا
      $st = $pdo->prepare("
        INSERT INTO friends (uid1, uid2, status, requested_by, created_at, updated_at)
        VALUES (:me, :peer, 'pending', :me, NOW(), NOW())
      ");
      $st->execute([':me'=>$me, ':peer'=>$peer]);
      json_out(['ok'=>true,'state'=>'outgoing_pending']);
    }

    // ---------------------------------------------
    // POST /request.php/respond   body:{ peer_uid, accept:true|false }
    // ---------------------------------------------
    case 'respond': {
      $in     = json_in();
      $peer   = resolve_user_id($pdo, $in['peer_uid'] ?? null);
      $accept = !empty($in['accept']);
      if ($peer <= 0 || $peer === $me) json_out(['ok'=>false,'error'=>'bad_peer'], 400);

      $st = $pdo->prepare("
        SELECT * FROM friends
        WHERE ((uid1=:peer AND uid2=:me) OR (uid1=:me AND uid2=:peer))
          AND status='pending'
        ORDER BY id DESC
        LIMIT 1
      ");
      $st->execute([':me'=>$me, ':peer'=>$peer]);
      $row = $st->fetch();
      if (!$row) json_out(['ok'=>false,'error'=>'no_pending'], 404);

      if ((int)$row['requested_by'] === $me) {
        json_out(['ok'=>false,'error'=>'you_are_requester'], 400);
      }

      $newStatus = $accept ? 'accepted' : 'rejected';
      $st = $pdo->prepare("UPDATE friends SET status=:s, updated_at=NOW() WHERE id=:id");
      $st->execute([':s'=>$newStatus, ':id'=>$row['id']]);

      json_out(['ok'=>true,'status'=>$newStatus]);
    }

    // ---------------------------------------------
    // POST /request.php/cancel   body:{ peer_uid }
    // ---------------------------------------------
    case 'cancel': {
      $in   = json_in();
      $peer = resolve_user_id($pdo, $in['peer_uid'] ?? null);
      if ($peer <= 0) json_out(['ok'=>false,'error'=>'bad_peer'], 400);

      $st = $pdo->prepare("
        DELETE FROM friends
        WHERE status='pending' AND requested_by=:me
          AND ((uid1=:me AND uid2=:peer) OR (uid1=:peer AND uid2=:me))
        LIMIT 1
      ");
      $st->execute([':me'=>$me, ':peer'=>$peer]);
      json_out(['ok'=>true]);
    }

    // ---------------------------------------------
    // POST /request.php/remove   body:{ peer_uid }
    // ---------------------------------------------
    case 'remove': {
      $in   = json_in();
      $peer = resolve_user_id($pdo, $in['peer_uid'] ?? null);
      if ($peer <= 0) json_out(['ok'=>false,'error'=>'bad_peer'], 400);

      $st = $pdo->prepare("
        DELETE FROM friends
        WHERE status='accepted'
          AND ((uid1=:me AND uid2=:peer) OR (uid1=:peer AND uid2=:me))
        LIMIT 1
      ");
      $st->execute([':me'=>$me, ':peer'=>$peer]);
      json_out(['ok'=>true]);
    }

    // ---------------------------------------------
    // GET /request.php/list?type=incoming|outgoing|friends
    // ---------------------------------------------
    case 'list': {
      $type = $_GET['type'] ?? 'incoming';
      if (!in_array($type, ['incoming','outgoing','friends'], true)) $type = 'incoming';

      if ($type === 'incoming') {
        $st = $pdo->prepare("
          SELECT * FROM friends
          WHERE status='pending'
            AND (uid1=:me OR uid2=:me)
            AND requested_by<>:me
          ORDER BY id DESC
        ");
      } elseif ($type === 'outgoing') {
        $st = $pdo->prepare("
          SELECT * FROM friends
          WHERE status='pending'
            AND requested_by=:me
          ORDER BY id DESC
        ");
      } else { // friends
        $st = $pdo->prepare("
          SELECT * FROM friends
          WHERE status='accepted'
            AND (uid1=:me OR uid2=:me)
          ORDER BY id DESC
        ");
      }
      $st->execute([':me'=>$me]);
      $rows = $st->fetchAll();

      $items = [];
      foreach ($rows as $r) {
        $peerId = other_id($r, $me);
        $items[] = [
          'id'           => (int)$r['id'],
          'status'       => $r['status'],
          'requested_by' => (int)$r['requested_by'],
          'peer_id'      => $peerId,
          'peer'         => user_summary($peerId),
          'created_at'   => $r['created_at'],
          'updated_at'   => $r['updated_at'],
        ];
      }
      json_out(['ok'=>true,'items'=>$items]);
    }

    default:
      json_out(['ok'=>false,'error'=>'bad_action','hint'=>'status|request|respond|cancel|remove|list'], 404);
  }
} catch (Throwable $e) {
  json_out(['ok'=>false,'error'=>'server_error','detail'=>$e->getMessage()], 500);
}
