<?php
require __DIR__ . '/config.php';

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Authorization, Content-Type');
header('Access-Control-Allow-Methods: POST, OPTIONS');

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }

// مصادقة (قد تُرجع id/uid)
$payload  = require_auth();
$auth_uid = $payload['uid']      ?? $payload['user_uid'] ?? null;
$auth_id  = $payload['id']       ?? $payload['user_id']  ?? null;

// --- اجمع جسم الطلب من JSON و POST (يدمجهم) ---
function in_all() {
  $j = json_in();            // ترجع مصفوفة إن كان JSON
  if (!is_array($j)) $j = [];
  foreach ($_POST as $k => $v) { $j[$k] = $v; } // POST يغلب JSON
  return $j;
}
$in = in_all();

// --- التقاط الحقول مع دعم أسماء شائعة بديلة ---
$username   = null;
if (isset($in['username']))         $username = trim((string)$in['username']);
elseif (isset($in['name']))         $username = trim((string)$in['name']);
elseif (isset($in['display_name'])) $username = trim((string)$in['display_name']);

$avatar_url = null;
if (isset($in['avatar_url']))       $avatar_url = trim((string)$in['avatar_url']);
elseif (isset($in['avatar']))       $avatar_url = trim((string)$in['avatar']);

$gender_raw = null;
if (isset($in['gender']))           $gender_raw = $in['gender'];
elseif (isset($in['sex']))          $gender_raw = $in['sex'];
elseif (isset($in['gendar']))       $gender_raw = $in['gendar'];
elseif (isset($in['gen']))          $gender_raw = $in['gen'];

// 👇 تاريخ الميلاد (يدعم عدة مفاتيح)
$birth_raw = null;
if (isset($in['birth_date']))        $birth_raw = $in['birth_date'];
elseif (isset($in['birthdate']))     $birth_raw = $in['birthdate'];
elseif (isset($in['dob']))           $birth_raw = $in['dob'];
elseif (isset($in['date_of_birth'])) $birth_raw = $in['date_of_birth'];

// 👇 البايو (نبذة) مع مفاتيح احتياطية شائعة
$bio = null;
if (array_key_exists('bio', $in))            $bio = trim((string)$in['bio']);
elseif (array_key_exists('about', $in))      $bio = trim((string)$in['about']);
elseif (array_key_exists('description', $in))$bio = trim((string)$in['description']);
elseif (array_key_exists('profile', $in))    $bio = trim((string)$in['profile']);
elseif (array_key_exists('bio_text', $in))   $bio = trim((string)$in['bio_text']);

// uid/id صريحة من العميل (تتجاوز ما جاء من التوكن)
$uid_param = isset($in['uid']) ? trim((string)$in['uid']) : null;
$id_param  = isset($in['id'])  ? (int)$in['id'] : null;

// توحيد المعرّفات
$uid = $uid_param ?: $auth_uid;
$id  = $id_param  ?: $auth_id;

// تثبيت/تصحيح رابط الصورة إلى https وكامل
function fix_url($s) {
  if (!isset($s)) return $s;
  $s = trim((string)$s);
  if ($s === '') return null;
  if (strpos($s, '//') === 0) $s = 'https:' . $s;
  if (stripos($s, 'http://') === 0) $s = 'https://' . substr($s, 7);
  if (stripos($s, 'http') !== 0) {
    $origin = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' ? 'https' : 'http')
            . '://' . ($_SERVER['HTTP_HOST'] ?? 'localhost');
    if ($s[0] !== '/') $s = '/' . $s;
    $s = $origin . $s;
  }
  return $s;
}
if ($avatar_url !== null && $avatar_url !== '') {
  $avatar_url = fix_url($avatar_url);
}

// --- تطبيع قيمة النوع إلى male/female ---
function normalize_gender($g) {
  if (!isset($g)) return null;
  $g = trim(mb_strtolower((string)$g, 'UTF-8'));
  $male_map   = ['male','m','ذكر','دكر','1','true','boy','man','ذكر ♂'];
  $female_map = ['female','f','أنثى','انثى','0','false','girl','woman','أنثى ♀'];
  if (in_array($g, $male_map, true))   return 'male';
  if (in_array($g, $female_map, true)) return 'female';
  $g2 = preg_replace('/\s+/', '', $g);
  if ($g2 === 'ذكر') return 'male';
  if ($g2 === 'انثى' || $g2 === 'أنثى') return 'female';
  return null;
}
$gender = normalize_gender($gender_raw);

// --- تطبيع/التحقق من تاريخ الميلاد إلى YYYY-MM-DD ---
function normalize_birth($d) {
  if (!isset($d)) return null;
  $s = trim((string)$d);
  if ($s === '') return null;

  // 1) جاهز بصيغة YYYY-MM-DD
  if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $s)) return $s;

  // 2) صيغ شائعة: DD/MM/YYYY أو DD-MM-YYYY
  if (preg_match('/^(\d{2})[\/\-](\d{2})[\/\-](\d{4})$/', $s, $m)) {
    $dd = (int)$m[1]; $mm = (int)$m[2]; $yy = (int)$m[3];
    if (checkdate($mm, $dd, $yy)) {
      return sprintf('%04d-%02d-%02d', $yy, $mm, $dd);
    }
  }

  // 3) محاولة أخيرة: strtotime
  $ts = strtotime($s);
  if ($ts !== false) {
    $yy = (int)date('Y', $ts);
    $mm = (int)date('m', $ts);
    $dd = (int)date('d', $ts);
    if (checkdate($mm, $dd, $yy)) {
      return sprintf('%04d-%02d-%02d', $yy, $mm, $dd);
    }
  }
  return false; // قيمة غير صالحة
}
$birth_date = normalize_birth($birth_raw);

// لو أُرسلت قيمة gender لكنها غير مفهومة
if (isset($gender_raw) && $gender_raw !== '' && $gender === null) {
  json_out(['ok'=>false,'error'=>'invalid_gender_value'], 422);
}
// لو أُرسل تاريخ ميلاد لكن غير صالح
if (isset($birth_raw) && $birth_raw !== '' && $birth_date === false) {
  json_out(['ok'=>false,'error'=>'invalid_birth_date'], 422);
}

// --- التحقق من البايو: حد أقصى 200 حرف (UTF-8) ---
if ($bio !== null) {
  // السماح بتفريغه بإرسال سلسلة فارغة؟ (حاليًا: السلسلة الفارغة تُحوَّل إلى NULL فلا تُحدِّث)
  if ($bio === '') {
    $bio = null;
  } else {
    if (mb_strlen($bio, 'UTF-8') > 200) {
      json_out(['ok'=>false,'error'=>'invalid_bio_length','max'=>200], 422);
    }
  }
}

// لا شيء للتحديث؟
if (($username   === null || $username   === '')
 && ($avatar_url === null || $avatar_url === '')
 && ($gender     === null || $gender     === '')
 && ($birth_date === null || $birth_date === '')
 && ($bio        === null /* يمكن السماح بسلسلة فارغة كمسح إن رغبت */)) {
  json_out(['ok'=>false,'error'=>'nothing_to_update'], 400);
}

try {
  $pdo = db();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

  $set  = [];
  $vals = [];

  if ($username   !== null && $username   !== '') { $set[] = 'username    = ?'; $vals[] = $username;   }
  if ($gender     !== null && $gender     !== '') { $set[] = 'gender      = ?'; $vals[] = $gender;     }
  if ($avatar_url !== null && $avatar_url !== '') { $set[] = 'avatar_url  = ?'; $vals[] = $avatar_url; }
  if ($birth_date !== null && $birth_date !== '') { $set[] = 'birth_date  = ?'; $vals[] = $birth_date; }
  if ($bio        !== null /* && $bio !== '' */ ) { $set[] = 'bio         = ?'; $vals[] = $bio;        }

  // إن كان لديك عمود updated_at
  $hasUpdatedAt = false; // غيّرها إلى true لو عندك العمود
  if ($hasUpdatedAt) { $set[] = 'updated_at = NOW()'; }

  if (!$set) {
    json_out(['ok'=>false,'error'=>'nothing_to_update'], 400);
  }

  // WHERE مرن: uid أو id
  $where = [];
  if ($uid) { $where[] = 'uid = ?'; $vals[] = $uid; }
  if ($id)  { $where[] = 'id  = ?'; $vals[] = (int)$id; }
  if (!$where) json_out(['ok'=>false,'error'=>'missing_user_identifier'], 401);

  $sql  = "UPDATE users SET ".implode(', ', $set)." WHERE ".implode(' OR ', $where)." LIMIT 1";
  $stmt = $pdo->prepare($sql);
  $stmt->execute($vals);

  // اعادة الصف بعد التحديث
  if ($uid) {
    $q = $pdo->prepare("SELECT id, uid, username, gender, avatar_url, birth_date, bio FROM users WHERE uid = ? LIMIT 1");
    $q->execute([$uid]);
  } else {
    $q = $pdo->prepare("SELECT id, uid, username, gender, avatar_url, birth_date, bio FROM users WHERE id = ? LIMIT 1");
    $q->execute([$id]);
  }
  $row = $q->fetch(PDO::FETCH_ASSOC) ?: [];

  json_out([
    'ok'   => true,
    'user' => [
      'id'         => isset($row['id']) ? (int)$row['id'] : ($id ?: null),
      'uid'        => $row['uid']        ?? $uid        ?? null,
      'username'   => $row['username']   ?? $username   ?? null,
      'gender'     => $row['gender']     ?? $gender     ?? null,
      'avatar_url' => $row['avatar_url'] ?? $avatar_url ?? null,
      'birth_date' => $row['birth_date'] ?? $birth_date ?? null,
      'bio'        => $row['bio']        ?? $bio        ?? null, // 👈 نرجّع البايو
    ],
  ]);

} catch (Throwable $e) {
  json_out(['ok'=>false,'error'=>'db_error','detail'=>$e->getMessage()], 500);
}
