<?php
header('Content-Type: application/json; charset=utf-8');

$errors = [];
$notes  = [];
$res    = [];

function ok($k,$v){ global $res; $res[$k]=$v; }

try {
  // 1) تحميل config.php
  $configPath = __DIR__ . '/../config.php';
  if (!file_exists($configPath)) throw new Exception("config_missing: $configPath");
  require_once $configPath;
  ok('config', 'loaded');

  // 2) التحقق من امتداد pdo_mysql
  if (!extension_loaded('pdo_mysql')) throw new Exception('pdo_mysql_extension_missing');
  ok('pdo_mysql', 'ok');

  // 3) اتصال قاعدة البيانات + نسخة MySQL
  if (!function_exists('db')) throw new Exception('db_function_missing_in_config');
  $pdo = db();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
  $v = $pdo->query('SELECT VERSION() as v')->fetchColumn();
  ok('mysql_version', $v);

  // 4) الدوال الموحّدة للـ Auth
  ok('auth_functions', [
    'get_bearer_token' => function_exists('get_bearer_token') ? 'ok' : 'missing',
    'bearer_user_id'   => function_exists('bearer_user_id')   ? 'ok' : 'missing',
    'require_user'     => function_exists('require_user')     ? 'ok' : 'missing',
  ]);

  // 5) الجداول المطلوبة (users بدلاً من tbl_users)
  $needTables = ['users','moment_posts','moment_post_likes','moment_comments','moment_categories'];
  $missing = [];
  foreach ($needTables as $t){
    $stmt = $pdo->prepare("SHOW TABLES LIKE ?");
    $stmt->execute([$t]);
    if (!$stmt->fetch()) $missing[] = $t;
  }
  if ($missing) throw new Exception('table_missing: '.implode(',', $missing));
  ok('tables', 'all_present');

  // 6) أعمدة أساسية سريعة (users, moment_posts)
  $needCols = [
    'users' => ['id','username','avatar_url'],
    'moment_posts' => ['id','user_id','category_id','content_text','image_url','created_at'],
  ];
  foreach ($needCols as $tbl => $cols) {
    $desc = $pdo->query("DESCRIBE `$tbl`")->fetchAll();
    $have = array_column($desc,'Field');
    $diff = array_diff($cols, $have);
    if ($diff) $notes[] = "missing_columns_in_$tbl: ".implode(',', $diff);
  }

  // 7) تعداد سريع للبوستات
  $cnt = (int)$pdo->query('SELECT COUNT(*) FROM moment_posts')->fetchColumn();
  ok('moment_posts_count', $cnt);

  // 8) تجربة JOIN بسيط users ← moment_posts
  $sqlJoin = "SELECT p.id, u.username
              FROM moment_posts p
              JOIN users u ON u.id = p.user_id
              ORDER BY p.id DESC
              LIMIT 1";
  $pdo->query($sqlJoin)->fetch();
  ok('join_users_check', 'ok');

  // 9) تجربة liked_by_me بطريقة LEFT JOIN (بدون الحاجة لتمرير uid فعلي)
  $sqlLikePreview = "EXPLAIN
    SELECT p.id,
           (me.user_id IS NOT NULL) AS liked_by_me
    FROM moment_posts p
    LEFT JOIN moment_post_likes me
      ON me.post_id = p.id AND me.user_id = :uid
    ORDER BY p.id DESC
    LIMIT 1";
  $st = $pdo->prepare($sqlLikePreview);
  $st->execute([':uid' => 1]); // اختبار شكلي
  ok('liked_by_me_plan', 'ok');

  // 10) تجربة استعلام التصنيفات باسم موحّد "name" (ar/en)
  $primary   = 'name_ar';
  $secondary = 'name_en';
  $cats = $pdo->query("
    SELECT id,
           COALESCE(NULLIF($primary,''), NULLIF($secondary,''), CONCAT('cat-',id)) AS name,
           COALESCE(name_ar,'') AS name_ar,
           COALESCE(name_en,'') AS name_en
    FROM moment_categories
    ORDER BY sort_order ASC, id ASC
    LIMIT 5
  ")->fetchAll();
  ok('categories_sample', $cats);

  // 11) صلاحية مجلد صور اللحظات من الكونفيج
  global $FILES_MOMENT_BASE_DIR, $FILES_MOMENT_BASE_URL;
  if (!isset($FILES_MOMENT_BASE_DIR) || !isset($FILES_MOMENT_BASE_URL)) {
    $notes[] = 'moment_files_paths_not_defined_in_config';
  } else {
    ok('files_moment_base_dir', $FILES_MOMENT_BASE_DIR);
    ok('files_moment_base_url', $FILES_MOMENT_BASE_URL);
    if (!is_dir($FILES_MOMENT_BASE_DIR)) {
      $notes[] = "files_dir_missing:$FILES_MOMENT_BASE_DIR";
    } else {
      $writable = is_writable($FILES_MOMENT_BASE_DIR);
      ok('files_dir_writable', $writable ? 'yes' : 'no');
      if (!$writable) $notes[] = "files_dir_not_writable:$FILES_MOMENT_BASE_DIR";
    }
  }

  // 12) رصد هيدر Authorization (اختياري)
  $authHdr = $_SERVER['HTTP_AUTHORIZATION'] ?? $_SERVER['Authorization'] ?? '';
  if (!$authHdr && function_exists('getallheaders')) {
    foreach (getallheaders() as $k => $v) {
      if (strcasecmp($k, 'Authorization') === 0) { $authHdr = $v; break; }
    }
  }
  ok('authorization_header_seen', $authHdr ? 'yes' : 'no');

  echo json_encode(['ok'=>1,'result'=>$res,'notes'=>$notes], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);

} catch (Throwable $e) {
  echo json_encode(['ok'=>0,'error'=>$e->getMessage(),'result'=>$res,'notes'=>$notes], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
}
