<?php
// voicechat/api/lib/fcm_lib.php
declare(strict_types=1);

use Firebase\JWT\JWT;

function fcm_get_access_token(string $service_json_path, array $scopes=["https://www.googleapis.com/auth/firebase.messaging"]): string {
  $sa = json_decode(file_get_contents($service_json_path), true);
  if (!$sa) throw new RuntimeException('Invalid service account JSON at '.$service_json_path);

  $now = time();
  $payload = [
    'iss' => $sa['client_email'],
    'sub' => $sa['client_email'],
    'aud' => 'https://oauth2.googleapis.com/token',
    'iat' => $now,
    'exp' => $now + 3600,
    'scope' => implode(' ', $scopes),
  ];
  $jwt = JWT::encode($payload, $sa['private_key'], 'RS256');

  $ch = curl_init('https://oauth2.googleapis.com/token');
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_POST => true,
    CURLOPT_POSTFIELDS => http_build_query([
      'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
      'assertion'  => $jwt,
    ]),
  ]);
  $out = curl_exec($ch);
  if ($out === false) throw new RuntimeException(curl_error($ch));
  $code = curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
  curl_close($ch);

  $resp = json_decode($out, true);
  if (!is_array($resp) || empty($resp['access_token'])) {
    throw new RuntimeException('Cannot get access token ('.$code.'): '.$out);
  }
  return $resp['access_token'];
}

function fcm_send_message_v1(string $project_id, array $message, string $access_token): array {
  $url = "https://fcm.googleapis.com/v1/projects/{$project_id}/messages:send";
  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_POST => true,
    CURLOPT_HTTPHEADER => [
      'Authorization: Bearer '.$access_token,
      'Content-Type: application/json; charset=UTF-8',
    ],
    CURLOPT_POSTFIELDS => json_encode(['message' => $message], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES),
  ]);
  $out = curl_exec($ch);
  if ($out === false) throw new RuntimeException(curl_error($ch));
  $code = curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
  curl_close($ch);

  $js = json_decode($out, true);
  if (!is_array($js)) $js = ['raw' => $out, 'status' => $code];
  return $js;
}
