<?php
// inc/room_access.php
require_once __DIR__ . '/config.php';

function is_room_locked(PDO $pdo, string $room_uid): bool {
  $st = $pdo->prepare("SELECT pass_enabled FROM rooms WHERE uid = ?");
  $st->execute([$room_uid]);
  $row = $st->fetch(PDO::FETCH_ASSOC);
  return $row && intval($row['pass_enabled']) === 1;
}

function set_room_password(PDO $pdo, string $owner_uid, string $room_uid, bool $enable, ?string $pin): array {
  // تحقق ملكية الغرفة
  $st = $pdo->prepare("SELECT owner_uid FROM rooms WHERE uid = ?");
  $st->execute([$room_uid]);
  $row = $st->fetch(PDO::FETCH_ASSOC);
  if (!$row) return ['ok'=>false,'error'=>'room_not_found'];
  if ($row['owner_uid'] !== $owner_uid) return ['ok'=>false,'error'=>'forbidden'];

  if (!$enable) {
    $st = $pdo->prepare("UPDATE rooms SET pass_enabled=0, pass_hash=NULL, pass_updated_at=NOW() WHERE uid=?");
    $st->execute([$room_uid]);
    return ['ok'=>true,'enabled'=>false];
  }

  if (!preg_match('/^\d{4}$/', $pin ?? '')) return ['ok'=>false,'error'=>'invalid_pin'];
  $hash = password_hash($pin, PASSWORD_BCRYPT);
  $st = $pdo->prepare("UPDATE rooms SET pass_enabled=1, pass_hash=?, pass_updated_at=NOW() WHERE uid=?");
  $st->execute([$hash, $room_uid]);
  return ['ok'=>true,'enabled'=>true];
}

function verify_room_pin(PDO $pdo, string $room_uid, string $pin): bool {
  $st = $pdo->prepare("SELECT pass_enabled, pass_hash FROM rooms WHERE uid=?");
  $st->execute([$room_uid]);
  $row = $st->fetch(PDO::FETCH_ASSOC);
  if (!$row || intval($row['pass_enabled']) !== 1) return true; // الغرفة مفتوحة
  return $row['pass_hash'] && password_verify($pin, $row['pass_hash']);
}

function grant_room_access(PDO $pdo, string $room_uid, string $user_uid): string {
  $token = bin2hex(random_bytes(16));
  $st = $pdo->prepare("INSERT INTO room_pass_tokens (room_uid, user_uid, token, expires_at)
                       VALUES (?, ?, ?, DATE_ADD(NOW(), INTERVAL 2 HOUR))");
  $st->execute([$room_uid, $user_uid, $token]);
  return $token;
}

function has_room_access(PDO $pdo, string $room_uid, string $user_uid, ?string $token): bool {
  if (!$token) return false;
  $st = $pdo->prepare("SELECT 1 FROM room_pass_tokens WHERE room_uid=? AND user_uid=? AND token=? AND expires_at>NOW() LIMIT 1");
  $st->execute([$room_uid, $user_uid, $token]);
  return (bool)$st->fetchColumn();
}
