<?php
// voicechat/api/gifts/feed.php
declare(strict_types=1);
require_once __DIR__ . '/../config.php';

// ============ Helpers ============
function app_origin(): string {
  $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
  $host   = $_SERVER['HTTP_HOST'] ?? 'battle-party.com';
  return $scheme . '://' . $host;
}
function abs_url(string $u): string {
  $u = trim($u);
  if ($u === '') return '';
  if (preg_match('#^https?://#i', $u)) return $u;
  if ($u[0] !== '/') $u = '/'.$u;     // تأكد من بداية بـ /
  return app_origin() . $u;
}

// ============ Params ============
$page = max(1, (int)($_GET['page'] ?? 1));
$size = (int)($_GET['page_size'] ?? 20);
if ($size < 1)   $size = 20;
if ($size > 100) $size = 100;

$offset = ($page - 1) * $size;

$pdo = db();

// ============ Query ============
// نجلب العمليات مرتبة تنازلياً بالأحدث، ونربط بمعلومات الهدية
$sql = "
SELECT
  gt.id,
  gt.gift_id,
  gt.sender_user_id,
  gt.receiver_user_id,
  gt.room_uid,
  gt.quantity,
  gt.unit_price,
  gt.total_price,
  gt.created_at,
  gi.name       AS gift_name,
  gi.image_url  AS gift_image_url,
  gi.price      AS gift_price
FROM gift_transactions gt
LEFT JOIN gift_items gi ON gi.id = gt.gift_id
ORDER BY gt.created_at DESC, gt.id DESC
LIMIT :lim OFFSET :off
";
$st = $pdo->prepare($sql);
$st->bindValue(':lim', $size, PDO::PARAM_INT);
$st->bindValue(':off', $offset, PDO::PARAM_INT);
$st->execute();
$rows = $st->fetchAll();

// كاش بسيط لمعلومات المستخدمين لتقليل الاستعلامات
$userCache = [];
$getUser = function(int $uid) use (&$userCache) : array {
  if ($uid <= 0) return ['id'=>0,'userid'=>'','username'=>'','avatar_url'=>''];
  if (!isset($userCache[$uid])) {
    $u = user_summary($uid);
    // تأكد من وجود userid إن كان في جدول users
    $u['avatar_url'] = abs_url((string)($u['avatar_url'] ?? ''));
    $userCache[$uid] = [
      'id'        => (int)($u['id'] ?? $uid),
      'userid'    => (string)($u['userid'] ?? ''),     // 👈 يرجع userid إن كان موجوداً
      'username'  => (string)($u['username'] ?? ''),
      'avatar_url'=> (string)($u['avatar_url'] ?? '')
    ];
  }
  return $userCache[$uid];
};

// صياغة العناصر للإخراج
$items = [];
foreach ($rows as $r) {
  $items[] = [
    'id'          => (int)$r['id'],
    'created_at'  => (string)$r['created_at'],
    'room_uid'    => (string)($r['room_uid'] ?? ''),
    'quantity'    => (int)$r['quantity'],
    'unit_price'  => (int)$r['unit_price'],
    'total_price' => (int)$r['total_price'],
    'sender'      => $getUser((int)$r['sender_user_id']),
    'receiver'    => $getUser((int)$r['receiver_user_id']),
    'gift'        => [
      'id'        => (int)$r['gift_id'],
      'name'      => (string)($r['gift_name'] ?? ''),
      'image_url' => abs_url((string)($r['gift_image_url'] ?? '')),
      'price'     => (int)($r['gift_price'] ?? $r['unit_price']),
    ],
  ];
}

// هل يوجد المزيد؟
$hasMore = (count($items) === $size);
$out = [
  'ok'         => true,
  'page'       => $page,
  'page_size'  => $size,
  'has_more'   => $hasMore,
  'next_page'  => $hasMore ? ($page + 1) : null,
  'items'      => $items,
];
json_out($out);
