<?php
// backend/api/auth_firebase_exchange.php
require __DIR__ . '/config.php';

// تحميل مكتبة firebase/php-jwt (ثبّتها بـ: composer require firebase/php-jwt:^6.9)
$autoload1 = __DIR__ . '/vendor/autoload.php';
$autoload2 = dirname(__DIR__) . '/vendor/autoload.php';
if (file_exists($autoload1)) {
  require_once $autoload1;
} elseif (file_exists($autoload2)) {
  require_once $autoload2;
} else {
  http_response_code(500);
  header('Content-Type: application/json; charset=utf-8');
  echo json_encode(['error' => 'jwt_library_missing']);
  exit;
}

use Firebase\JWT\JWT;
use Firebase\JWT\JWK;

const FB_PROJECT_ID = 'wesal-chat-26277'; // ← تأكد أنها تخص مشروعك

// ======= مساعدة للتسجيل التشخيصي =======
function logx($msg) {
  @file_put_contents(__DIR__ . '/auth_exchange_debug.log',
    '['.date('Y-m-d H:i:s').'] '.$msg."\n", FILE_APPEND);
}

// ======= جلب مفاتيح Google (JWKS) مع كاش =======
function fetch_jwks_cached(): array {
  $cache = __DIR__ . '/.jwks_cache.json';
  if (file_exists($cache) && (time() - filemtime($cache) < 3600)) {
    $j = file_get_contents($cache);
    $arr = json_decode($j, true);
    if (is_array($arr)) return $arr;
  }
  $url = 'https://www.googleapis.com/service_accounts/v1/jwk/securetoken@system.gserviceaccount.com';
  if (function_exists('curl_init')) {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
      CURLOPT_RETURNTRANSFER => true,
      CURLOPT_FOLLOWLOCATION => true,
      CURLOPT_SSL_VERIFYPEER => true,
      CURLOPT_SSL_VERIFYHOST => 2,
      CURLOPT_TIMEOUT        => 8,
      CURLOPT_CONNECTTIMEOUT => 6,
      CURLOPT_HTTPHEADER     => ['Accept: application/json'],
    ]);
    $out  = curl_exec($ch);
    $err  = curl_error($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    if ($out === false || $code >= 400) {
      logx("JWKS fetch failed: ".($err ?: "HTTP $code"));
      throw new Exception('jwks_fetch_failed');
    }
  } else {
    $ctx = stream_context_create(['http'=>['timeout'=>8, 'header'=>"Accept: application/json\r\n"]]);
    $out = @file_get_contents($url, false, $ctx);
    if ($out === false) {
      logx("JWKS fetch failed: file_get_contents");
      throw new Exception('jwks_fetch_failed');
    }
  }
  file_put_contents($cache, $out);
  $arr = json_decode($out, true);
  if (!is_array($arr)) throw new Exception('jwks_parse_failed');
  return $arr;
}

// ======= مولّد userid فريد من 8 أرقام (اختياري داخل التطبيق) =======
function generate_userid(PDO $pdo): int {
  do {
    $n = random_int(10000000, 99999999);
    $stmt = $pdo->prepare("SELECT 1 FROM users WHERE userid=? LIMIT 1");
    $stmt->execute([$n]);
    $exists = (bool)$stmt->fetchColumn();
  } while ($exists);
  return $n;
}

// ======= إعدادات CORS =======
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Authorization, Content-Type');
header('Access-Control-Allow-Methods: POST, OPTIONS');
if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }

// ======= التنفيذ =======
try {
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  $idToken  = trim($in['idToken']  ?? '');
  $username = trim($in['username'] ?? '');
  $email    = trim($in['email']    ?? '');

  if ($idToken === '') { json_out(['error'=>'missing_idToken'], 422); }

  // فك وتحقق من توقيع idToken
  $jwks    = fetch_jwks_cached();
  $decoded = JWT::decode($idToken, JWK::parseKeySet($jwks));
  $claims  = (array)$decoded;

  // المطالبات الأساسية
  $aud = $claims['aud'] ?? '';
  $iss = $claims['iss'] ?? '';
  $sub = $claims['sub'] ?? '';
  if ($aud !== FB_PROJECT_ID) { logx("bad_aud: $aud"); json_out(['error'=>'bad_aud'], 401); }
  if ($iss !== 'https://securetoken.google.com/'.FB_PROJECT_ID) { logx("bad_iss: $iss"); json_out(['error'=>'bad_iss'], 401); }
  if ($sub === '') { logx("empty_sub"); json_out(['error'=>'empty_sub'], 401); }

  $firebaseUid = $sub; // هذا هو users.uid لدينا
  $emailFromToken = $claims['email'] ?? null;
  if (!$email && $emailFromToken) $email = $emailFromToken;

  // أنشئ/حدّث المستخدم في MySQL
  $pdo = db();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

  $stmt = $pdo->prepare("SELECT id, uid, username, userid FROM users WHERE uid=? LIMIT 1");
  $stmt->execute([$firebaseUid]);
  $row = $stmt->fetch(PDO::FETCH_ASSOC);

  $dbId = null;             // رقم المستخدم (users.id)
  $dbUidText = $firebaseUid; // النصّي (users.uid) — عندنا هو Firebase UID
  $finalUsername = '';
  $finalUserId   = null;     // userid ذو 8 أرقام (اختياري)

  if (!$row) {
    $uname  = $username !== '' ? $username : ($email ? explode('@',$email)[0] : 'User'.substr($firebaseUid,0,6));
    $userid = generate_userid($pdo);

    $ins = $pdo->prepare("INSERT INTO users (uid,email,username,password_hash,userid) VALUES (?,?,?,?,?)");
    $ins->execute([$firebaseUid, $email ?: null, $uname, 'firebase', $userid]);

    $dbId          = (int)$pdo->lastInsertId();
    $finalUsername = $uname;
    $finalUserId   = $userid;
  } else {
    $dbId          = (int)$row['id'];
    $dbUidText     = (string)$row['uid'];
    $finalUsername = (string)$row['username'];

    if ($username !== '' && $username !== $finalUsername) {
      $upd = $pdo->prepare("UPDATE users SET username=? WHERE uid=?");
      $upd->execute([$username, $firebaseUid]);
      $finalUsername = $username;
    }
    if (empty($row['userid'])) {
      $finalUserId = generate_userid($pdo);
      $upd2 = $pdo->prepare("UPDATE users SET userid=? WHERE uid=?");
      $upd2->execute([$finalUserId, $firebaseUid]);
    } else {
      $finalUserId = (int)$row['userid'];
    }
  }

  // ======= الأهم: اجعل الـ JWT يحمل uid النصّي (users.uid) وأضف id الرقمي =======
  $token = jwt_encode([
    'uid'          => $dbUidText,   // ✅ نصّي (users.uid) — متوافق مع بقية الـ API
    'id'           => $dbId,        // ✅ رقمي (users.id) — لمن يحتاجه
    'username'     => $finalUsername,
    'userid'       => $finalUserId,
    'firebase_uid' => $firebaseUid, // (اختياري) للمعلومة فقط
    'iat'          => time(),
    'exp'          => time() + 86400*30,
  ]);

  // استجابة موحّدة: uid نصّي + id رقمي
  json_out([
    'ok'           => true,
    'uid'          => $dbUidText,   // ← نصّي
    'id'           => $dbId,        // ← رقمي
    'firebase_uid' => $firebaseUid,
    'userid'       => $finalUserId,
    'username'     => $finalUsername,
    'token'        => $token,
  ]);

} catch (Throwable $e) {
  logx('Exception: '.$e->getMessage());
  json_out(['error'=>'invalid_firebase_token'], 401);
}
