<?php
// /public_html/voicechat/agents/convert_usd.php
declare(strict_types=1);
require __DIR__ . '/_common.php';

/**
 * نقرأ سعر التحويل من جدول app_settings
 * ندعم name/value أو key/value حسب مشروعك.
 */
function read_setting(PDO $pdo, string $key, $default = null) {
  try {
    // اكتشاف أسماء الأعمدة
    $cols = $pdo->query("SHOW COLUMNS FROM app_settings")->fetchAll(PDO::FETCH_COLUMN, 0);
    $colKey = in_array('name', $cols, true) ? 'name' : (in_array('key', $cols, true) ? '`key`' : null);
    $colVal = in_array('value', $cols, true) ? 'value' : null;
    if (!$colKey || !$colVal) return $default;

    $st = $pdo->prepare("SELECT $colVal FROM app_settings WHERE $colKey = ? LIMIT 1");
    $st->execute([$key]);
    $v = $st->fetchColumn();
    if ($v === false || $v === null || $v === '') return $default;
    return is_numeric($v) ? (float)$v : $v;
  } catch (Throwable $e) { return $default; }
}

/** يرجع السعر Coins per 1 USD */
function current_rate(PDO $pdo): ?float {
  // أولوية: agent_coins_per_usd (عدد الكوين مقابل 1$)
  $coinsPerUsd = read_setting($pdo, 'agent_coins_per_usd', null);
  if (is_numeric($coinsPerUsd) && $coinsPerUsd > 0) return (float)$coinsPerUsd;

  // بديل: agent_usd_per_coin (كم دولار لكل كوين) → نقلبه
  $usdPerCoin = read_setting($pdo, 'agent_usd_per_coin', null);
  if (is_numeric($usdPerCoin) && $usdPerCoin > 0) return (float)(1.0 / (float)$usdPerCoin);

  return null; // لم يضبط الأدمن السعر بعد
}

$DEBUG = isset($_GET['dbg']);

try {
  $agent = require_agent(); // تحقق التوكن والحالة
  $pdo   = db();

  // GET => إرجاع السعر والأرصدة (للاستعلام/العرض)
  if (($_SERVER['REQUEST_METHOD'] ?? 'GET') === 'GET') {
    $rate = current_rate($pdo);
    if ($rate === null) json_out(['ok'=>false,'error'=>'rate_not_set'], 404);

    // أرصدة حديثة
    $st = $pdo->prepare("SELECT main_balance, IFNULL(usd_balance,0) AS usd_balance FROM agents WHERE id=? LIMIT 1");
    $st->execute([(int)$agent['id']]);
    $row = $st->fetch() ?: ['main_balance'=>0,'usd_balance'=>0];

    json_out([
      'ok'=>true,
      'rate_coins_per_usd'=>$rate,
      'balances'=>[
        'coins'=>(int)$row['main_balance'],
        'usd'  =>(float)$row['usd_balance'],
      ],
    ]);
  }

  // POST => تنفيذ التحويل
  if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
    json_out(['ok'=>false,'error'=>'method_not_allowed'], 405);
  }

  $in  = function_exists('json_in') ? json_in() : [];
  $usd = (float)($in['usd'] ?? 0);
  if ($usd <= 0) json_out(['ok'=>false,'error'=>'bad_amount'], 400);

  $rate = current_rate($pdo);
  if ($rate === null) json_out(['ok'=>false,'error'=>'rate_not_set'], 400);

  // عدد الكوين الناتج (أعمدة coins أعداد صحيحة)
  $coins = (int)floor($usd * $rate);
  if ($coins <= 0) json_out(['ok'=>false,'error'=>'too_small'], 400);

  // كشف الأعمدة الاختيارية في دفتر الحركات
  $has_currency = false;
  try { $has_currency = (bool)$pdo->query("SHOW COLUMNS FROM agent_ledger LIKE 'currency'")->fetch(); } catch (Throwable $e) {}

  $pdo->beginTransaction();

  // اقفل حساب الوكيل
  $st = $pdo->prepare("SELECT main_balance, IFNULL(usd_balance,0) AS usd_balance FROM agents WHERE id=? FOR UPDATE");
  $st->execute([(int)$agent['id']]);
  $a  = $st->fetch();
  if (!$a) throw new Exception('agent_not_found');

  $usd_before   = (float)$a['usd_balance'];
  $coins_before = (int)$a['main_balance'];

  if ($usd_before + 1e-9 < $usd) { // هامش صغير للكسور
    $pdo->rollBack();
    json_out(['ok'=>false,'error'=>'insufficient_usd'], 402);
  }

  // نفّذ التحويل: خصم USD وإضافة Coins
  $st = $pdo->prepare("UPDATE agents SET usd_balance = usd_balance - ?, main_balance = main_balance + ? WHERE id=? LIMIT 1");
  $st->execute([$usd, $coins, (int)$agent['id']]);

  // أرصدة بعد التحويل
  $st = $pdo->prepare("SELECT main_balance, IFNULL(usd_balance,0) AS usd_balance FROM agents WHERE id=? LIMIT 1");
  $st->execute([(int)$agent['id']]);
  $a2 = $st->fetch() ?: ['main_balance'=>0,'usd_balance'=>0];

  // دفتر حركات: سجل دخول كوين (coins/in) + إن وُجد currency سجّل خروج USD (usd/out)
  // 1) coins/in
  $cols = ['agent_id','direction','reason','amount','balance_after'];
  $vals = [(int)$agent['id'], 'in', 'usd_to_coins', $coins, (int)$a2['main_balance']];
  if ($has_currency) { $cols[]='currency'; $vals[]='coins'; }
  $ph   = implode(',', array_fill(0, count($cols), '?'));
  $pdo->prepare("INSERT INTO agent_ledger (".implode(',', $cols).") VALUES ($ph)")->execute($vals);

  // 2) usd/out (إن توفّر عمود currency)
  if ($has_currency) {
    $cols = ['agent_id','direction','reason','amount','balance_after','currency'];
    $vals = [(int)$agent['id'], 'out', 'usd_to_coins', $usd, (float)$a2['usd_balance'], 'usd'];
    $ph   = implode(',', array_fill(0, count($cols), '?'));
    $pdo->prepare("INSERT INTO agent_ledger (".implode(',', $cols).") VALUES ($ph)")->execute($vals);
  }

  $pdo->commit();

  json_out([
    'ok'=>true,
    'rate_coins_per_usd'=>$rate,
    'converted'=>[
      'usd'   =>$usd,
      'coins' =>$coins,
    ],
    'balances_after'=>[
      'usd'  =>(float)$a2['usd_balance'],
      'coins'=>(int)$a2['main_balance'],
    ],
  ]);
} catch (Throwable $e) {
  if (isset($pdo) && $pdo instanceof PDO && $pdo->inTransaction()) {
    try { $pdo->rollBack(); } catch (Throwable $e2) {}
  }
  if (isset($DEBUG) && $DEBUG) {
    json_out(['ok'=>false,'error'=>'server_error','message'=>$e->getMessage()], 500);
  }
  json_out(['ok'=>false,'error'=>'server_error'], 500);
}
