<?php
// admin/pages_edit.php
declare(strict_types=1);

require_once __DIR__ . '/inc/auth.php';
require_login();

/* ---------------- CSRF ---------------- */
if (!isset($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
function csrf_token(): string { return $_SESSION['csrf']; }
function csrf_ok(?string $t): bool { return $t && hash_equals($_SESSION['csrf'], $t); }

/* -------------- Helpers --------------- */
if (!function_exists('e')) {
  function e(string $s): string { return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }
}
function fmt_dt(?string $s): string {
  if (!$s) return '—';
  $t = strtotime($s); if ($t<=0) return e($s);
  return date('Y-m-d H:i', $t);
}

/* -------------- Setup ------------------ */
$allowed = [
  'privacy-policy' => 'سياسة الخصوصية',
  'terms'          => 'شروط الاستخدام',
];
$slug = $_GET['slug'] ?? 'privacy-policy';
if (!array_key_exists($slug, $allowed)) $slug = 'privacy-policy';

$pdo = db();
$msg = null; $err = null;

/* -------------- Save ------------------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  if (!csrf_ok($_POST['csrf'] ?? '')) {
    $err = 'رمز الأمان غير صالح.';
  } else {
    $slug = $_POST['slug'] ?? $slug;
    if (!array_key_exists($slug, $allowed)) $slug = 'privacy-policy';

    $title   = trim((string)($_POST['title'] ?? ''));
    $content = (string)($_POST['content_html'] ?? '');
    if ($title === '') $title = $allowed[$slug];

    try {
      $st = $pdo->prepare("
        INSERT INTO static_pages (slug, title, content_html, updated_by)
        VALUES (?, ?, ?, ?)
        ON DUPLICATE KEY UPDATE
          title=VALUES(title),
          content_html=VALUES(content_html),
          updated_by=VALUES(updated_by)
      ");
      $st->execute([$slug, $title, $content, $_SESSION['admin_id'] ?? null]);
      $msg = '🎉 تم الحفظ بنجاح.';
    } catch (Throwable $e) {
      $err = 'حدث خطأ أثناء الحفظ.';
    }
  }
}

/* -------------- Fetch ------------------ */
$st = $pdo->prepare("SELECT title, content_html, updated_at FROM static_pages WHERE slug=? LIMIT 1");
$st->execute([$slug]);
$row = $st->fetch() ?: ['title' => $allowed[$slug], 'content_html' => '', 'updated_at' => null];

/* -------------- Page meta -------------- */
$PAGE_TITLE = "الصفحات القانونية";
require __DIR__ . '/inc/header.php';
require __DIR__ . '/inc/sidebar.php';

/* مفتاح TinyMCE */
$TINYMCE_KEY = 'k36668h9d9cmy0wqvzdbdolt687snrln8jm3wptj2ltcqorq';
?>

<!-- ====== CSS خاص بالصفحة ====== -->
<style>
.legal-page .tabs{display:flex;gap:8px;flex-wrap:wrap;margin-bottom:10px}
.legal-page .tab-link{
  display:inline-flex;align-items:center;gap:8px;
  padding:8px 12px;border-radius:10px;border:1px solid var(--border);
  background:var(--panel);text-decoration:none;color:var(--text);
}
.legal-page .tab-link.active{background:rgba(59,130,246,.12);border-color:#3b82f6}
.legal-page .meta{opacity:.8;font-size:12px}
.legal-page .editor-wrap label{display:block;margin:10px 0 6px}
.legal-page .editor-wrap input[type=text]{
  width:100%;padding:10px 12px;border-radius:10px;border:1px solid var(--border);
  background:#0b1220;color:var(--text);
}
.legal-page .btn.sm{padding:8px 12px;font-size:14px}

/* تكبير المحرّر */
.legal-page textarea#content_html{
  min-height: 680px; /* في حال فشل تحميل TinyMCE */
}
.legal-page .tox-tinymce{
  height: 720px !important;   /* ارتفاع إطار المحرّر */
}
.legal-page .tox .tox-editor-container{
  min-height: 680px;          /* ارتفاع مساحة التحرير الداخلية */
}
</style>

<!-- TinyMCE -->
<script src="https://cdn.tiny.cloud/1/<?= e($TINYMCE_KEY) ?>/tinymce/6/tinymce.min.js" referrerpolicy="origin"></script>
<script>
document.addEventListener('DOMContentLoaded', function(){
  tinymce.init({
    selector: 'textarea#content_html',
    directionality: 'rtl',
    language: 'ar',
    language_url: 'https://cdn.tiny.cloud/1/<?= e($TINYMCE_KEY) ?>/tinymce/6/langs/ar.js',

    /* حجم كبير ومريح */
    plugins: 'link lists table code autoresize',
    min_height: 680,                 // ارتفاع أدنى كبير
    autoresize_bottom_margin: 24,    // هامش أسفل عند التمدد
    menubar: false,

    toolbar: 'undo redo | bold italic underline | alignright aligncenter alignleft | bullist numlist | link table | code',
    content_style: 'body{font-family:Tahoma,Arial,sans-serif;font-size:14px}'
  });
});
</script>

<div class="legal-page">
  <!-- الشريط العلوي -->
  <div class="card" style="margin-bottom:12px;">
    <div style="display:flex;gap:12px;align-items:center;justify-content:space-between;flex-wrap:wrap;">
      <h1 style="margin:0"><?= e($PAGE_TITLE) ?></h1>
      <div class="meta">آخر تحديث: <b><?= e(fmt_dt($row['updated_at'] ?? null)) ?></b></div>
    </div>
  </div>

  <!-- تبويبات الصفحات -->
  <div class="card" style="margin-bottom:12px;">
    <div class="tabs">
      <a class="tab-link <?= $slug==='privacy-policy'?'active':'' ?>" href="pages_edit.php?slug=privacy-policy">📄 سياسة الخصوصية</a>
      <a class="tab-link <?= $slug==='terms'?'active':'' ?>"          href="pages_edit.php?slug=terms">📜 شروط الاستخدام</a>
    </div>
    <?php if ($msg): ?><div class="alert" style="border-color:#22c55e;margin-top:8px;"><?= e($msg) ?></div><?php endif; ?>
    <?php if ($err): ?><div class="alert" style="margin-top:8px;"><?= e($err) ?></div><?php endif; ?>
  </div>

  <!-- النموذج -->
  <div class="card">
    <form method="post" class="editor-wrap">
      <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
      <input type="hidden" name="slug" value="<?= e($slug) ?>">

      <label>العنوان</label>
      <input type="text" name="title" value="<?= e($row['title']) ?>" placeholder="العنوان الظاهر للمستخدمين">

      <label>المحتوى</label>
      <textarea id="content_html" name="content_html"><?= e($row['content_html']) ?></textarea>

      <div style="margin-top:12px;display:flex;gap:8px;align-items:center;flex-wrap:wrap;">
        <button class="btn sm" type="submit">حفظ</button>
        <a class="btn sm secondary" href="pages_edit.php?slug=<?= e($slug) ?>">إلغاء التغييرات</a>
        <span class="meta">رابط API: <code>/voicechat/api/page_get.php?slug=<?= e($slug) ?></code></span>
      </div>
    </form>
  </div>
</div>

<?php require __DIR__ . '/inc/footer.php'; ?>
