<?php
// admin/moments.php
declare(strict_types=1);
require_once __DIR__ . '/inc/auth.php';
require_login();

/* Debug اختياري: moments.php?dbg=1 */
if (isset($_GET['dbg'])) { ini_set('display_errors','1'); error_reporting(E_ALL); }

/* ---------------- CSRF ---------------- */
if (!isset($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
function csrf_token(): string { return $_SESSION['csrf']; }
function csrf_ok(?string $t): bool { return $t && hash_equals($_SESSION['csrf'], $t); }

/* -------------- Helpers --------------- */
if (!function_exists('e')) {
  function e(string $s): string { return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }
}
function fmt_dt(?string $s): string {
  if (!$s) return '';
  $t = strtotime($s); if ($t<=0) return e((string)$s);
  return date('Y-m-d H:i', $t);
}

/* -------------- Actions ---------------- */
$pdo = db();
$msg = null; $err = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $action = $_POST['action'] ?? '';
  $id     = (int)($_POST['id'] ?? 0);

  // تنفيذ دُفعة (Bulk)
  if (isset($_POST['bulk_action'])) {
    if (!csrf_ok($_POST['csrf'] ?? '')) {
      $err = 'رمز الأمان غير صالح.';
    } else {
      $bulkAction = $_POST['bulk_action'] ?? '';
      $idsCsv = trim((string)($_POST['ids'] ?? ''));
      $ids = [];
      if ($idsCsv !== '') {
        foreach (explode(',', $idsCsv) as $x) {
          $v = (int)trim($x);
          if ($v > 0) $ids[$v] = $v;
        }
      }
      $ids = array_values($ids);

      if (!$ids) {
        $err = 'لم يتم اختيار أي منشورات.';
      } elseif (!in_array($bulkAction, ['toggle_delete', 'delete_post', 'clear_image'], true)) {
        $err = 'إجراء غير مدعوم.';
      } else {
        try {
          $place = implode(',', array_fill(0, count($ids), '?'));
          $pdo->beginTransaction();
          if ($bulkAction === 'toggle_delete') {
            $st = $pdo->prepare("UPDATE `moment_posts` SET `is_deleted` = 1 - `is_deleted` WHERE `id` IN ($place)");
            $st->execute($ids);
          } elseif ($bulkAction === 'delete_post') {
            $st = $pdo->prepare("DELETE FROM `moment_posts` WHERE `id` IN ($place)");
            $st->execute($ids);
          } else { // clear_image
            $st = $pdo->prepare("UPDATE `moment_posts` SET `image_url`=NULL WHERE `id` IN ($place)");
            $st->execute($ids);
          }
          $pdo->commit();
          $msg = 'تم تنفيذ العملية على ' . count($ids) . ' منشور(ات).';
        } catch (Throwable $e) {
          if ($pdo->inTransaction()) $pdo->rollBack();
          $err = 'حدث خطأ أثناء تنفيذ العملية.' . (isset($_GET['dbg']) ? ' ('.$e->getMessage().')' : '');
        }
      }
    }
  }
  // إجراءات مفردة
  else {
    if (!csrf_ok($_POST['csrf'] ?? '')) {
      $err = 'رمز الأمان غير صالح.';
    } elseif ($id <= 0) {
      $err = 'مُعرّف غير صالح.';
    } else {
      try {
        if ($action === 'toggle_delete') {
          $pdo->prepare("UPDATE `moment_posts` SET `is_deleted`=1-`is_deleted` WHERE `id`=? LIMIT 1")->execute([$id]);
          $msg = 'تم تغيير حالة الحذف.';
        } elseif ($action === 'clear_image') {
          $pdo->prepare("UPDATE `moment_posts` SET `image_url`=NULL WHERE `id`=? LIMIT 1")->execute([$id]);
          $msg = 'تم تفريغ صورة المنشور.';
        } elseif ($action === 'delete_post') {
          $pdo->prepare("DELETE FROM `moment_posts` WHERE `id`=? LIMIT 1")->execute([$id]);
          $msg = 'تم حذف المنشور نهائيًا.';
        }
      } catch (Throwable $e) {
        $err = 'حدث خطأ أثناء تنفيذ العملية.' . (isset($_GET['dbg']) ? ' ('.$e->getMessage().')' : '');
      }
    }
  }
}

/* --------- Filters / Search / Paging -------- */
$q        = trim($_GET['q'] ?? '');
$statusF  = trim($_GET['status'] ?? ''); // '', active, deleted
$catF     = (int)($_GET['category_id'] ?? 0);
$page     = max(1, (int)($_GET['page'] ?? 1));
$perPage  = 50;
$offset   = ($page - 1) * $perPage;

/* جلب قائمة التصنيفات */
$catNames = [];
try {
  $res = $pdo->query("SELECT * FROM `moment_categories` ORDER BY `id` ASC");
  foreach (($res?->fetchAll(PDO::FETCH_ASSOC) ?? []) as $c) {
    $catNames[(int)$c['id']] = (string)($c['name'] ?? ($c['title'] ?? ('#'.$c['id'])));
  }
} catch (Throwable $e) {
  // تجاهل لو الجدول غير موجود
}

/* بناء شرط البحث */
$where = "1=1"; $params = [];
if ($q !== '') {
  $where .= " AND (mp.content_text LIKE ? OR u.username LIKE ? OR u.email LIKE ?)";
  $kw = "%$q%"; array_push($params,$kw,$kw,$kw);
}
if ($statusF === 'active') { $where .= " AND mp.is_deleted = 0"; }
elseif ($statusF === 'deleted') { $where .= " AND mp.is_deleted = 1"; }
if ($catF > 0) { $where .= " AND mp.category_id = ?"; $params[] = $catF; }

/* إجمالي النتائج */
$total = 0;
try {
  $stCnt = $pdo->prepare("SELECT COUNT(*) FROM `moment_posts` mp LEFT JOIN `users` u ON u.id=mp.user_id WHERE $where");
  $stCnt->execute($params);
  $total = (int)$stCnt->fetchColumn();
} catch (Throwable $e) {
  $err = 'تعذّر حساب الإجمالي.' . (isset($_GET['dbg']) ? ' ('.$e->getMessage().')' : '');
}
$pages = max(1, (int)ceil($total / $perPage));

/* جلب الصفحة */
$rows = [];
try {
  $sql = "SELECT mp.id, mp.user_id, mp.category_id, mp.content_text, mp.image_url,
                 mp.likes_count, mp.comments_count, mp.created_at, mp.is_deleted,
                 u.username AS user_name, u.email AS user_email
          FROM `moment_posts` mp
          LEFT JOIN `users` u ON u.id = mp.user_id
          WHERE $where
          ORDER BY mp.created_at ASC, mp.id ASC
          LIMIT $perPage OFFSET $offset";
  $st = $pdo->prepare($sql);
  $st->execute($params);
  $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
} catch (Throwable $e) {
  $rows = [];
  $err = 'تعذّر جلب البيانات.' . (isset($_GET['dbg']) ? ' ('.$e->getMessage().')' : '');
}

/* -------------- Render ------------------ */
$PAGE_TITLE = "اللحظات";
require __DIR__ . '/inc/header.php';
require __DIR__ . '/inc/sidebar.php';
?>

<!-- ====== CSS ====== -->
<style>
.moments-page .table{width:100%;border-collapse:separate;border-spacing:0;background:var(--panel);border:1px solid var(--border);border-radius:12px;overflow:hidden;table-layout:fixed}
.moments-page .table td,.moments-page .table th{padding:10px 12px;border-bottom:1px solid var(--border);vertical-align:middle;text-align:start}
.moments-page .table thead th{background:#0b1220;color:var(--text);font-weight:700;position:sticky;top:0;z-index:3;box-shadow:inset 0 -1px 0 var(--border)}
.moments-page .table tbody tr:nth-child(even) td{background:rgba(255,255,255,.015)}
.moments-page .table tbody tr:hover td{background:rgba(59,130,246,.08)}
.moments-page .table-wrap{max-height:70vh;overflow:auto;position:relative}
.moments-page .td-center{text-align:center}
.moments-page .ellipsis{max-width:260px;display:inline-block;overflow:hidden;white-space:nowrap;text-overflow:ellipsis}
.moments-page .ellipsis.small{max-width:180px}
.moments-page .post-img{width:52px;height:52px;border-radius:10px;object-fit:cover;border:1px solid var(--border);cursor:zoom-in}
.moments-page .badge{display:inline-block;padding:2px 8px;border-radius:999px;border:1px solid var(--border);font-size:12px}
.moments-page .badge.ok{border-color:#22c55e}
.moments-page .badge.muted{opacity:.7}
.moments-page .actions{display:flex;gap:8px;flex-wrap:nowrap;align-items:center}
.moments-page .actions form{display:inline-block;margin:0}
.moments-page .btn.sm{padding:6px 10px;font-size:14px}
.moments-page .col-sel{width:44px}.moments-page .col-id{width:60px}.moments-page .col-img{width:76px}.moments-page .col-text{width:360px}
.moments-page .col-user{width:220px}.moments-page .col-cat{width:160px}
.moments-page .col-likes{width:90px}.moments-page .col-comments{width:110px}.moments-page .col-created{width:170px}
.moments-page .col-status{width:100px}.moments-page .col-actions{width:360px}

/* شريط الأدوات فوق الجدول */
.moments-page .bulkbar{display:flex;gap:12px;align-items:center;flex-wrap:wrap;margin:12px 0}
.moments-page .bulkbar .cnt{opacity:.9}

/* ===== Lightbox Modal (عام) ===== */
#imgModal{
  position: fixed;
  inset: 0;
  display: none;                 /* يُظهر عند إضافة .open */
  align-items: center;
  justify-content: center;
  background: rgba(0,0,0,.85);
  z-index: 2147483647;           /* فوق أي عناصر ثابتة/لاصقة */
  padding: 24px;
  box-sizing: border-box;
}
#imgModal.open{ display: flex; }
#imgModal img{
  max-width: 92vw;
  max-height: 92vh;
  border-radius: 12px;
  box-shadow: 0 10px 40px rgba(0,0,0,.6);
  display: block;
}
#imgModal #imgClose{
  position: absolute;
  top: 16px;
  inset-inline-end: 16px;
  background: rgba(255,255,255,.12);
  color: #fff;
  border: 1px solid rgba(255,255,255,.25);
  padding: 8px 12px;
  border-radius: 10px;
  cursor: pointer;
  font-weight: 600;
}
body.modal-open{ overflow: hidden; } /* يمنع تمرير الصفحة أثناء فتح المودال */

/* حركة دخول خفيفة (اختياري) */
@keyframes imgPop { from{ transform: scale(.96); opacity: .5 } to{ transform: scale(1); opacity: 1 } }
#imgModal.open img{ animation: imgPop .12s ease-out; }
</style>

<div class="moments-page">
  <div class="card" style="margin-bottom:16px;">
    <form method="get" class="form" style="display:flex; gap:12px; align-items:flex-end; flex-wrap:wrap;">
      <div class="input" style="min-width:260px; flex:1;">
        <label>ابحث بالنص / اسم أو إيميل صاحب المنشور</label>
        <input type="text" name="q" value="<?= e($q) ?>" placeholder="مثال: حب / user@mail.com">
      </div>

      <div class="input">
        <label>الحالة</label>
        <select name="status" style="padding:10px 12px;border-radius:10px;border:1px solid var(--border);background:#0b1220;color:var(--text);">
          <option value="" <?= $statusF===''?'selected':'' ?>>الكل</option>
          <option value="active" <?= $statusF==='active'?'selected':'' ?>>نشط</option>
          <option value="deleted" <?= $statusF==='deleted'?'selected':'' ?>>محذوف</option>
        </select>
      </div>

      <div class="input">
        <label>التصنيف</label>
        <select name="category_id" style="padding:10px 12px;border-radius:10px;border:1px solid var(--border);background:#0b1220;color:var(--text);">
          <option value="0" <?= $catF===0?'selected':'' ?>>الكل</option>
          <?php foreach ($catNames as $cid=>$cname): ?>
            <option value="<?= (int)$cid ?>" <?= $catF===$cid?'selected':'' ?>><?= e($cname) ?></option>
          <?php endforeach; ?>
        </select>
      </div>

      <button class="btn sm" type="submit">بحث</button>
      <?php if ($q!=='' || $statusF!=='' || $catF>0): ?>
        <a class="btn sm secondary" href="moments.php">مسح الفلاتر</a>
      <?php endif; ?>
      <div style="margin-inline-start:auto;opacity:.9;">إجمالي: <b><?= $total ?></b> نتيجة</div>
    </form>
    <?php if ($msg): ?><div class="alert" style="margin-top:12px;border-color:#22c55e;"><?= e($msg) ?></div><?php endif; ?>
    <?php if ($err): ?><div class="alert" style="margin-top:12px;"><?= e($err) ?></div><?php endif; ?>
  </div>

  <!-- شريط أدوات التحديد الجماعي -->
  <div class="card">
    <div class="bulkbar">
      <label style="display:flex;align-items:center;gap:8px;">
        <input type="checkbox" id="selAllTop">
        <span>تحديد الكل (في الصفحة الحالية)</span>
      </label>

      <div class="cnt" id="selCount">0 محدد</div>

      <form method="post" id="bulkForm" style="margin-inline-start:auto;display:flex;gap:8px;align-items:center;">
        <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
        <input type="hidden" name="ids" id="bulkIds" value="">
        <select name="bulk_action" id="bulkAction" style="padding:10px 12px;border-radius:10px;border:1px solid var(--border);background:#0b1220;color:var(--text);">
          <option value="toggle_delete">تبديل حذف/استعادة</option>
          <option value="clear_image">تفريغ الصور</option>
          <option value="delete_post">حذف نهائي</option>
        </select>
        <button class="btn sm" type="submit">تنفيذ على المحدد</button>
      </form>
    </div>
  </div>

  <div class="card">
    <div class="table-wrap">
      <table class="table">
        <colgroup>
          <col class="col-sel"><col class="col-id"><col class="col-img"><col class="col-text"><col class="col-user">
          <col class="col-cat"><col class="col-likes"><col class="col-comments">
          <col class="col-created"><col class="col-status"><col class="col-actions">
        </colgroup>
        <thead>
          <tr>
            <th class="td-center"><input type="checkbox" id="selAllHead" title="تحديد/إلغاء تحديد الكل"></th>
            <th class="td-center">#</th>
            <th class="td-center">الصورة</th>
            <th>النص</th>
            <th>المالك</th>
            <th>التصنيف</th>
            <th>إعجابات</th>
            <th>تعليقات</th>
            <th>أُنشيء</th>
            <th>الحالة</th>
            <th>إجراءات</th>
          </tr>
        </thead>
        <tbody>
          <?php foreach ($rows as $r): $rid=(int)$r['id']; $img=(string)($r['image_url'] ?? ''); ?>
            <tr>
              <td class="td-center"><input type="checkbox" class="row-sel" value="<?= $rid ?>" aria-label="تحديد المنشور #<?= $rid ?>"></td>
              <td class="td-center"><?= $rid ?></td>
              <td class="td-center">
                <?php if ($img !== ''): ?>
                  <img src="<?= e($img) ?>" data-full="<?= e($img) ?>" class="post-img" alt="صورة المنشور #<?= $rid ?>">
                <?php else: ?>
                  <span class="badge muted">بدون</span>
                <?php endif; ?>
              </td>
              <td title="<?= e($r['content_text'] ?? '') ?>"><span class="ellipsis"><?= e($r['content_text'] ?? '') ?></span></td>
              <td>
                <?php if (!empty($r['user_name'])): ?>
                  <span title="<?= e($r['user_email'] ?? '') ?>"><?= e($r['user_name']) ?></span>
                <?php else: ?>
                  <span class="ellipsis small" title="user_id: <?= (int)($r['user_id'] ?? 0) ?>">ID: <?= (int)($r['user_id'] ?? 0) ?></span>
                <?php endif; ?>
              </td>
              <td>
                <?php $catLabel = $catNames[(int)($r['category_id'] ?? 0)] ?? ('#'.(int)($r['category_id'] ?? 0)); ?>
                <span class="ellipsis small" title="<?= e($catLabel) ?>"><?= e($catLabel) ?></span>
              </td>
              <td class="td-center"><span class="badge ok"><?= (int)($r['likes_count'] ?? 0) ?></span></td>
              <td class="td-center"><span class="badge"><?= (int)($r['comments_count'] ?? 0) ?></span></td>
              <td title="<?= e($r['created_at'] ?? '') ?>"><?= fmt_dt($r['created_at'] ?? '') ?></td>
              <td class="td-center">
                <?= ((int)($r['is_deleted'] ?? 0) === 1) ? '<span class="badge" style="border-color:#ef4444">محذوف</span>' : '<span class="badge ok">نشط</span>' ?>
              </td>
              <td class="actions">
                <a class="btn sm secondary" href="moments_edit.php?id=<?= $rid ?>">تعديل</a>

                <form method="post" onsubmit="return confirm('تبديل حالة الحذف (حذف/استعادة)؟');">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= $rid ?>">
                  <input type="hidden" name="action" value="toggle_delete">
                  <button class="btn sm" type="submit"><?= ((int)$r['is_deleted']===1)?'استعادة':'حذف' ?></button>
                </form>

                <form method="post" onsubmit="return confirm('تفريغ صورة المنشور؟');">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= $rid ?>">
                  <input type="hidden" name="action" value="clear_image">
                  <button class="btn sm" type="submit">تفريغ الصورة</button>
                </form>

                <form method="post" onsubmit="return confirm('حذف نهائي؟ لا يمكن التراجع.');">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= $rid ?>">
                  <input type="hidden" name="action" value="delete_post">
                  <button class="btn sm" type="submit" style="background:var(--danger);">حذف نهائي</button>
                </form>
              </td>
            </tr>
          <?php endforeach; ?>
          <?php if (!$rows): ?>
            <tr><td colspan="11" style="text-align:center;opacity:.8;">لا توجد بيانات</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>

    <?php if ($pages > 1): ?>
      <div style="display:flex;gap:8px;margin-top:12px;flex-wrap:wrap;">
        <?php for($p=1;$p<=$pages;$p++):
          $url = 'moments.php?page='.$p
               .($q!=='' ? '&q='.urlencode($q) : '')
               .($statusF!=='' ? '&status='.urlencode($statusF) : '')
               .($catF>0 ? '&category_id='.$catF : '');
          $is = $p===$page; ?>
          <a href="<?= e($url) ?>" class="btn sm <?= $is ? '' : 'secondary' ?>"<?= $is?' style="pointer-events:none; opacity:.8;"':''; ?>><?= $p ?></a>
        <?php endfor; ?>
      </div>
    <?php endif; ?>
  </div>
</div>

<!-- Lightbox Modal (سيُنقل تلقائيًا إلى <body> عبر الجافاسكربت) -->
<div id="imgModal" role="dialog" aria-modal="true" aria-hidden="true">
  <button id="imgClose" type="button" aria-label="إغلاق">✕</button>
  <img id="imgFull" src="" alt="معاينة الصورة">
</div>

<script>
(function(){
  // ==== أدوات التحديد ====
  const $ = (sel, root=document) => root.querySelector(sel);
  const $$ = (sel, root=document) => Array.from(root.querySelectorAll(sel));

  const selAllTop  = $('#selAllTop');
  const selAllHead = $('#selAllHead');
  const bulkIds    = $('#bulkIds');
  const selCount   = $('#selCount');
  const bulkForm   = $('#bulkForm');
  const bulkAction = $('#bulkAction');

  function updateSelectionState(){
    const boxes = $$('.row-sel');
    const checked = boxes.filter(b => b.checked);
    selCount.textContent = checked.length + ' محدد';
    bulkIds.value = checked.map(b => b.value).join(',');
    const allChecked = boxes.length > 0 && checked.length === boxes.length;
    if (selAllTop)  selAllTop.checked  = allChecked;
    if (selAllHead) selAllHead.checked = allChecked;
  }
  function setAll(v){
    $$('.row-sel').forEach(b => b.checked = v);
    updateSelectionState();
  }
  if (selAllTop)  selAllTop.addEventListener('change', e => setAll(e.target.checked));
  if (selAllHead) selAllHead.addEventListener('change', e => setAll(e.target.checked));
  document.addEventListener('change', (e)=>{
    if (e.target && e.target.classList && e.target.classList.contains('row-sel')) {
      updateSelectionState();
    }
  });
  updateSelectionState();

  if (bulkForm) {
    bulkForm.addEventListener('submit', (e)=>{
      const ids = (bulkIds.value||'').trim();
      if (!ids) { e.preventDefault(); alert('من فضلك اختر منشور واحد على الأقل.'); return; }
      const act = bulkAction ? bulkAction.value : '';
      let msg = 'تنفيذ "'+act+'" على العناصر المحددة؟';
      if (act === 'delete_post') msg = 'حذف نهائي للعناصر المحددة؟ لا يمكن التراجع.';
      if (!confirm(msg)) { e.preventDefault(); return; }
    });
  }

  // ==== Lightbox ====
  const modal   = document.getElementById('imgModal');
  const imgFull = document.getElementById('imgFull');
  const btnClose= document.getElementById('imgClose');

  // نقل المودال لجذر الـ body لضمان الظهور فوق كل شيء
  if (modal && modal.parentElement !== document.body) {
    document.body.appendChild(modal);
  }

  function openModal(src){
    if (!src) return;
    imgFull.src = src;
    modal.classList.add('open');
    modal.setAttribute('aria-hidden','false');
    document.body.classList.add('modal-open'); // قفل السكرول
    btnClose && btnClose.focus?.();
  }
  function closeModal(){
    modal.classList.remove('open');
    modal.setAttribute('aria-hidden','true');
    document.body.classList.remove('modal-open');
    imgFull.src = '';
  }

  // فتح عند الضغط على صورة منشور
  document.addEventListener('click', (e)=>{
    const t = e.target;
    if (t && t.matches('img.post-img')) {
      e.preventDefault();
      const src = t.getAttribute('data-full') || t.getAttribute('src');
      openModal(src);
    }
  });

  // إغلاق بالزر أو بالضغط على الخلفية أو بـ Esc
  btnClose && btnClose.addEventListener('click', closeModal);
  modal.addEventListener('click', (e)=>{ if (e.target === modal) closeModal(); });
  document.addEventListener('keydown', (e)=>{ if (e.key === 'Escape') closeModal(); });
})();
</script>

<?php require __DIR__ . '/inc/footer.php'; ?>
