<?php
// admin/emojis.php
declare(strict_types=1);
require_once __DIR__ . '/inc/auth.php';
require_login();

/* ---------------- CSRF ---------------- */
if (!isset($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
function csrf_token(): string { return $_SESSION['csrf']; }
function csrf_ok(?string $t): bool { return $t && hash_equals($_SESSION['csrf'], $t); }

/* -------------- Helpers --------------- */
if (!function_exists('e')) {
  function e(string $s): string { return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }
}

/* -------------- DB -------------------- */
$pdo = db();
$msg = null; $err = null;

/* -------------- Actions --------------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $action = $_POST['action'] ?? '';
  $id     = (int)($_POST['id'] ?? 0);
  if (!csrf_ok($_POST['csrf'] ?? '')) {
    $err = 'رمز الأمان غير صالح.';
  } elseif ($id <= 0) {
    $err = 'مُعرّف غير صالح.';
  } else {
    try {
      if ($action === 'toggle_animated') {
        $pdo->prepare("UPDATE emojis SET animated = IF(animated=1,0,1) WHERE id=? LIMIT 1")->execute([$id]);
        $msg = 'تم تغيير حالة التحريك.';
      } elseif ($action === 'delete_emoji') {
        $pdo->prepare("DELETE FROM emojis WHERE id=? LIMIT 1")->execute([$id]);
        $msg = 'تم حذف الايموجي نهائيًا.';
      }
    } catch (Throwable $e) { $err = 'حدث خطأ أثناء تنفيذ العملية.'; }
  }
}

/* -------------- Filters / Paging ------ */
$q        = trim($_GET['q'] ?? '');
$catF     = (int)($_GET['category_id'] ?? 0);
$page     = max(1, (int)($_GET['page'] ?? 1));
$perPage  = 50;
$offset   = ($page - 1) * $perPage;

/* جلب أسماء الفئات لاستخدامها بالفلتر/العرض */
$catMap = [];
foreach ($pdo->query("SELECT id,name FROM emoji_categories ORDER BY sort_order ASC, id ASC") as $c) {
  $catMap[(int)$c['id']] = (string)$c['name'];
}

/* where */
$where = "1=1"; $params = [];
if ($q !== '') {
  $where .= " AND (e.name LIKE ?)";
  $params[] = "%$q%";
}
if ($catF > 0) {
  $where .= " AND e.category_id = ?";
  $params[] = $catF;
}

/* count */
$st = $pdo->prepare("SELECT COUNT(*) FROM emojis e WHERE $where");
$st->execute($params);
$total = (int)$st->fetchColumn();
$pages = max(1, (int)ceil($total / $perPage));

/* list */
$sql = "SELECT e.id, e.name, e.image_url, e.animated, e.sort_order, e.category_id
        FROM emojis e
        WHERE $where
        ORDER BY e.id DESC
        LIMIT $perPage OFFSET $offset";
$st = $pdo->prepare($sql); $st->execute($params);
$rows = $st->fetchAll();

/* -------------- Render ---------------- */
$PAGE_TITLE = "الايموجي";
require __DIR__ . '/inc/header.php';
require __DIR__ . '/inc/sidebar.php';
?>
<style>
.emojis-page .table{width:100%;border-collapse:separate;border-spacing:0;background:var(--panel);border:1px solid var(--border);border-radius:12px;overflow:hidden;table-layout:fixed}
.emojis-page .table td,.emojis-page .table th{padding:10px 12px;border-bottom:1px solid var(--border);vertical-align:middle}
.emojis-page .table thead th{background:#0b1220;color:var(--text);font-weight:700;position:sticky;top:0;z-index:3;box-shadow:inset 0 -1px 0 var(--border)}
.emojis-page .table tbody tr:nth-child(even) td{background:rgba(255,255,255,.015)}
.emojis-page .table tbody tr:hover td{background:rgba(59,130,246,.08)}
.emojis-page .table-wrap{max-height:70vh;overflow:auto;position:relative}
.emojis-page .td-center{text-align:center}
.emojis-page .ellipsis{max-width:220px;display:inline-block;overflow:hidden;white-space:nowrap;text-overflow:ellipsis}
.emojis-page .emoji-img{width:44px;height:44px;border-radius:8px;object-fit:contain;background:#0d111a;border:1px solid var(--border)}
.emojis-page .badge{display:inline-block;padding:2px 8px;border-radius:999px;border:1px solid var(--border);font-size:12px}
.emojis-page .badge.ok{border-color:#22c55e}
.emojis-page .actions{display:flex;gap:8px;flex-wrap:nowrap;align-items:center}
.emojis-page .btn.sm{padding:6px 10px;font-size:14px}
.emojis-page .col-id{width:60px}.emojis-page .col-img{width:72px}.emojis-page .col-name{width:220px}
.emojis-page .col-cat{width:180px}.emojis-page .col-anim{width:110px}.emojis-page .col-sort{width:100px}.emojis-page .col-actions{width:360px}
</style>

<div class="emojis-page">
  <div class="card" style="margin-bottom:16px;">
    <form method="get" class="form" style="display:flex;gap:12px;align-items:flex-end;flex-wrap:wrap;">
      <div class="input" style="min-width:240px;flex:1">
        <label>ابحث بالاسم</label>
        <input type="text" name="q" value="<?= e($q) ?>" placeholder="مثال: ضحك">
      </div>
      <div class="input">
        <label>الفئة</label>
        <select name="category_id" style="padding:10px 12px;border-radius:10px;border:1px solid var(--border);background:#0b1220;color:var(--text);">
          <option value="0"<?= $catF===0?' selected':''; ?>>الكل</option>
          <?php foreach ($catMap as $cid=>$cname): ?>
            <option value="<?= (int)$cid ?>"<?= $catF===$cid?' selected':''; ?>><?= e($cname) ?></option>
          <?php endforeach; ?>
        </select>
      </div>
      <button class="btn sm" type="submit">بحث</button>
      <?php if ($q!=='' || $catF>0): ?><a class="btn sm secondary" href="emojis.php">مسح الفلاتر</a><?php endif; ?>
      <div style="margin-inline-start:auto">إجمالي: <b><?= $total ?></b></div>
      <a class="btn sm" href="emojis_add.php" style="margin-inline-start:auto">➕ إضافة ايموجي</a>
    </form>
    <?php if ($msg): ?><div class="alert" style="margin-top:10px;border-color:#22c55e;"><?= e($msg) ?></div><?php endif; ?>
    <?php if ($err): ?><div class="alert" style="margin-top:10px;"><?= e($err) ?></div><?php endif; ?>
  </div>

  <div class="card">
    <div class="table-wrap">
      <table class="table">
        <colgroup>
          <col class="col-id"><col class="col-img"><col class="col-name"><col class="col-cat"><col class="col-anim"><col class="col-sort"><col class="col-actions">
        </colgroup>
        <thead>
          <tr>
            <th class="td-center">#</th>
            <th class="td-center">صورة</th>
            <th>الاسم</th>
            <th>الفئة</th>
            <th>متحرّك</th>
            <th>الترتيب</th>
            <th>إجراءات</th>
          </tr>
        </thead>
        <tbody>
          <?php foreach ($rows as $r): ?>
            <tr>
              <td class="td-center"><?= (int)$r['id'] ?></td>
              <td class="td-center"><?php if ($r['image_url']): ?><img class="emoji-img" src="<?= e($r['image_url']) ?>" alt=""><?php else: ?><span class="badge">—</span><?php endif; ?></td>
              <td><span class="ellipsis" title="<?= e($r['name']) ?>"><?= e($r['name']) ?></span></td>
              <td><span class="ellipsis" title="<?= e($catMap[(int)$r['category_id']] ?? ('#'.(int)$r['category_id'])) ?>"><?= e($catMap[(int)$r['category_id']] ?? ('#'.(int)$r['category_id'])) ?></span></td>
              <td class="td-center"><?= ((int)$r['animated']===1) ? '<span class="badge ok">نعم</span>' : '<span class="badge">لا</span>' ?></td>
              <td class="td-center"><?= (int)$r['sort_order'] ?></td>
              <td class="actions">
                <a class="btn sm secondary" href="emojis_add.php?id=<?= (int)$r['id'] ?>">تعديل</a>

                <form method="post" onsubmit="return confirm('تبديل حالة التحريك؟');">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                  <input type="hidden" name="action" value="toggle_animated">
                  <button class="btn sm" type="submit"><?= ((int)$r['animated']===1) ? 'تعطيل' : 'تفعيل' ?></button>
                </form>

                <form method="post" onsubmit="return confirm('حذف نهائي؟ لا يمكن التراجع.');">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                  <input type="hidden" name="action" value="delete_emoji">
                  <button class="btn sm" type="submit" style="background:var(--danger);">حذف</button>
                </form>
              </td>
            </tr>
          <?php endforeach; ?>
          <?php if (!$rows): ?><tr><td colspan="7" style="text-align:center;opacity:.8;">لا توجد بيانات</td></tr><?php endif; ?>
        </tbody>
      </table>
    </div>

    <?php if ($pages > 1): ?>
      <div style="display:flex;gap:8px;margin-top:12px;flex-wrap:wrap;">
        <?php for($p=1;$p<=$pages;$p++):
          $url = 'emojis.php?page='.$p.($q!=='' ? '&q='.urlencode($q) : '').($catF>0 ? '&category_id='.$catF : '');
          $is = $p===$page; ?>
          <a href="<?= e($url) ?>" class="btn sm <?= $is?'':'secondary' ?>"<?= $is?' style="pointer-events:none;opacity:.8"':''; ?>><?= $p ?></a>
        <?php endfor; ?>
      </div>
    <?php endif; ?>
  </div>
</div>
<?php require __DIR__ . '/inc/footer.php'; ?>
