<?php
// admin/comments.php
declare(strict_types=1);
require_once __DIR__ . '/inc/auth.php';
require_login();

/* Debug اختياري: comments.php?dbg=1 */
if (isset($_GET['dbg'])) { ini_set('display_errors','1'); error_reporting(E_ALL); }

/* ---------------- CSRF ---------------- */
if (!isset($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
function csrf_token(): string { return $_SESSION['csrf']; }
function csrf_ok(?string $t): bool { return $t && hash_equals($_SESSION['csrf'], $t); }

/* -------------- Helpers --------------- */
if (!function_exists('e')) {
  function e(string $s): string { return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }
}
function fmt_dt(?string $s): string {
  if (!$s) return '';
  $t = strtotime($s); if ($t<=0) return e($s);
  return date('Y-m-d H:i', $t);
}

/* -------------- Actions ---------------- */
$pdo = db();
$msg = null; $err = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $action = $_POST['action'] ?? '';
  $id     = (int)($_POST['id'] ?? 0);
  if (!csrf_ok($_POST['csrf'] ?? '')) {
    $err = 'رمز الأمان غير صالح.';
  } elseif ($id <= 0) {
    $err = 'مُعرّف غير صالح.';
  } else {
    try {
      if ($action === 'delete_comment') {
        $pdo->prepare("DELETE FROM `moment_comments` WHERE `id`=? LIMIT 1")->execute([$id]);
        $msg = 'تم حذف التعليق نهائيًا.';
      }
    } catch (Throwable $e) {
      $err = 'حدث خطأ أثناء تنفيذ العملية.' . (isset($_GET['dbg']) ? ' ('.$e->getMessage().')' : '');
    }
  }
}

/* --------- Filters / Search / Paging -------- */
$q       = trim($_GET['q'] ?? '');
$postId  = (int)($_GET['post_id'] ?? 0);
$userId  = (int)($_GET['user_id'] ?? 0);
$page    = max(1, (int)($_GET['page'] ?? 1));
$perPage = 50;                     // 50 تعليق في الصفحة
$offset  = ($page - 1) * $perPage;

/* بناء الشرط */
$where  = "1=1"; $params = [];
if ($q !== '') {
  // لاحظ: عمود نص التعليق اسمه text في جدولك
  $where .= " AND (mc.`text` LIKE ? OR u.username LIKE ? OR u.email LIKE ? OR mp.content_text LIKE ?)";
  $kw = "%$q%"; array_push($params,$kw,$kw,$kw,$kw);
}
if ($postId > 0) { $where .= " AND mc.post_id = ?"; $params[] = $postId; }
if ($userId > 0) { $where .= " AND mc.user_id = ?"; $params[] = $userId; }

/* إجمالي النتائج */
$total = 0;
try {
  $st = $pdo->prepare("SELECT COUNT(*)
                       FROM `moment_comments` mc
                       LEFT JOIN `users` u ON u.id = mc.user_id
                       LEFT JOIN `moment_posts` mp ON mp.id = mc.post_id
                       WHERE $where");
  $st->execute($params);
  $total = (int)$st->fetchColumn();
} catch (Throwable $e) {
  $err = 'تعذّر حساب الإجمالي.' . (isset($_GET['dbg']) ? ' ('.$e->getMessage().')' : '');
}
$pages = max(1, (int)ceil($total / $perPage));

/* جلب الصفحة */
$rows = [];
try {
  $sql = "SELECT mc.id, mc.post_id, mc.user_id, mc.`text` AS comment_text, mc.created_at,
                 u.username AS user_name, u.email AS user_email,
                 mp.content_text AS post_text
          FROM `moment_comments` mc
          LEFT JOIN `users` u  ON u.id  = mc.user_id
          LEFT JOIN `moment_posts` mp ON mp.id = mc.post_id
          WHERE $where
          ORDER BY mc.created_at ASC, mc.id ASC
          LIMIT $perPage OFFSET $offset";
  $st = $pdo->prepare($sql);
  $st->execute($params);
  $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
} catch (Throwable $e) {
  $rows = [];
  $err = 'تعذّر جلب البيانات.' . (isset($_GET['dbg']) ? ' ('.$e->getMessage().')' : '');
}

/* -------------- Render ------------------ */
$PAGE_TITLE = "تعليقات اللحظات";
require __DIR__ . '/inc/header.php';
require __DIR__ . '/inc/sidebar.php';
?>

<style>
.comments-page .table{width:100%;border-collapse:separate;border-spacing:0;background:var(--panel);border:1px solid var(--border);border-radius:12px;overflow:hidden;table-layout:fixed}
.comments-page .table td,.comments-page .table th{padding:10px 12px;border-bottom:1px solid var(--border);vertical-align:middle;text-align:start}
.comments-page .table thead th{background:#0b1220;color:var(--text);font-weight:700;position:sticky;top:0;z-index:3;box-shadow:inset 0 -1px 0 var(--border)}
.comments-page .table tbody tr:nth-child(even) td{background:rgba(255,255,255,.015)}
.comments-page .table tbody tr:hover td{background:rgba(59,130,246,.08)}
.comments-page .table-wrap{max-height:70vh;overflow:auto;position:relative}
.comments-page .td-center{text-align:center}
.comments-page .ellipsis{max-width:300px;display:inline-block;overflow:hidden;white-space:nowrap;text-overflow:ellipsis}
.comments-page .ellipsis.small{max-width:220px}
.comments-page .badge{display:inline-block;padding:2px 8px;border-radius:999px;border:1px solid var(--border);font-size:12px}
.comments-page .actions{display:flex;gap:8px;flex-wrap:nowrap;align-items:center}
.comments-page .actions form{display:inline-block;margin:0}
.comments-page .btn.sm{padding:6px 10px;font-size:14px}
.comments-page .col-id{width:60px}.comments-page .col-text{width:360px}.comments-page .col-post{width:320px}.comments-page .col-user{width:240px}.comments-page .col-ids{width:160px}.comments-page .col-created{width:170px}.comments-page .col-actions{width:220px}
</style>

<div class="comments-page">
  <div class="card" style="margin-bottom:16px;">
    <form method="get" class="form" style="display:flex; gap:12px; align-items:flex-end; flex-wrap:wrap;">
      <div class="input" style="min-width:260px; flex:1;">
        <label>ابحث بالنص / اسم أو إيميل صاحب التعليق / نص المنشور</label>
        <input type="text" name="q" value="<?= e($q) ?>" placeholder="مثال: تعليق / user@mail.com / نص منشور">
      </div>
      <div class="input">
        <label>ID المنشور</label>
        <input type="number" name="post_id" min="1" value="<?= $postId>0?(int)$postId:'' ?>">
      </div>
      <div class="input">
        <label>ID المستخدم</label>
        <input type="number" name="user_id" min="1" value="<?= $userId>0?(int)$userId:'' ?>">
      </div>
      <button class="btn sm" type="submit">بحث</button>
      <?php if ($q!=='' || $postId>0 || $userId>0): ?>
        <a class="btn sm secondary" href="comments.php">مسح الفلاتر</a>
      <?php endif; ?>
      <div style="margin-inline-start:auto; opacity:.9;">إجمالي: <b><?= $total ?></b> نتيجة</div>
    </form>
    <?php if ($msg): ?><div class="alert" style="margin-top:12px; border-color:#22c55e;"><?= e($msg) ?></div><?php endif; ?>
    <?php if ($err): ?><div class="alert" style="margin-top:12px;"><?= e($err) ?></div><?php endif; ?>
  </div>

  <div class="card">
    <div class="table-wrap">
      <table class="table">
        <colgroup>
          <col class="col-id"><col class="col-text"><col class="col-post"><col class="col-user">
          <col class="col-ids"><col class="col-created"><col class="col-actions">
        </colgroup>
        <thead>
          <tr>
            <th class="td-center">#</th>
            <th>نص التعليق</th>
            <th>المنشور</th>
            <th>صاحب التعليق</th>
            <th>IDs</th>
            <th>أُنشيء</th>
            <th>إجراءات</th>
          </tr>
        </thead>
        <tbody>
          <?php foreach ($rows as $r): ?>
            <tr>
              <td class="td-center"><?= (int)$r['id'] ?></td>

              <td title="<?= e($r['comment_text'] ?? '') ?>"><span class="ellipsis"><?= e($r['comment_text'] ?? '') ?></span></td>

              <td>
                <?php $postLabel = trim((string)($r['post_text'] ?? '')); if ($postLabel==='') $postLabel = 'ID: '.(int)($r['post_id'] ?? 0); ?>
                <a class="ellipsis small" title="<?= e($r['post_text'] ?? '') ?>" href="moments_edit.php?id=<?= (int)($r['post_id'] ?? 0) ?>" style="text-decoration:underline;">
                  <?= e($postLabel) ?>
                </a>
              </td>

              <td>
                <?php if (!empty($r['user_name'])): ?>
                  <a class="ellipsis small" title="<?= e($r['user_email'] ?? '') ?>" href="users_edit.php?id=<?= (int)($r['user_id'] ?? 0) ?>" style="text-decoration:underline;">
                    <?= e($r['user_name']) ?>
                  </a>
                <?php else: ?>
                  <span class="ellipsis small">ID: <?= (int)($r['user_id'] ?? 0) ?></span>
                <?php endif; ?>
              </td>

              <td class="td-center">
                <span class="badge">P: <?= (int)($r['post_id'] ?? 0) ?></span>
                <span class="badge">U: <?= (int)($r['user_id'] ?? 0) ?></span>
              </td>

              <td title="<?= e($r['created_at'] ?? '') ?>"><?= fmt_dt($r['created_at'] ?? '') ?></td>

              <td class="actions">
                <a class="btn sm secondary" href="comments_edit.php?id=<?= (int)$r['id'] ?>">تعديل</a>
                <form method="post" onsubmit="return confirm('حذف نهائي للتعليق؟');">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                  <input type="hidden" name="action" value="delete_comment">
                  <button class="btn sm" type="submit" style="background:var(--danger);">حذف</button>
                </form>
              </td>
            </tr>
          <?php endforeach; ?>
          <?php if (!$rows): ?>
            <tr><td colspan="7" style="text-align:center; opacity:.8;">لا توجد بيانات</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>

    <?php if ($pages > 1): ?>
      <div style="display:flex; gap:8px; margin-top:12px; flex-wrap:wrap;">
        <?php for($p=1;$p<=$pages;$p++):
          $url = 'comments.php?page='.$p
               .($q!=='' ? '&q='.urlencode($q) : '')
               .($postId>0 ? '&post_id='.$postId : '')
               .($userId>0 ? '&user_id='.$userId : '');
          $is = $p===$page; ?>
          <a href="<?= e($url) ?>" class="btn sm <?= $is ? '' : 'secondary' ?>"<?= $is?' style="pointer-events:none; opacity:.8;"':''; ?>><?= $p ?></a>
        <?php endfor; ?>
      </div>
    <?php endif; ?>
  </div>
</div>

<?php require __DIR__ . '/inc/footer.php'; ?>
