<?php
// admin/banners.php
declare(strict_types=1);
require_once __DIR__ . '/inc/auth.php';
require_login();

/* ---------------- CSRF ---------------- */
if (!isset($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
function csrf_token(): string { return $_SESSION['csrf']; }
function csrf_ok(?string $t): bool { return $t && hash_equals($_SESSION['csrf'], $t); }

/* -------------- Helpers --------------- */
if (!function_exists('e')) {
  function e(string $s): string { return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }
}
function fmt_date(?string $s): string {
  if (!$s) return '—';
  $t = strtotime($s); if ($t<=0) return e($s);
  return date('Y-m-d H:i', $t);
}
function badge(string $text, string $class=''): string {
  $cls = 'badge'.($class ? ' '.$class : '');
  return '<span class="'.$cls.'">'.e($text).'</span>';
}
function active_badge(int $a): string {
  return $a ? badge('مفعّل','ok') : badge('غير مفعّل','muted');
}

/* -------------- Actions ---------------- */
$pdo = db();
$msg = null; $err = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $action = $_POST['action'] ?? '';
  $id     = (int)($_POST['id'] ?? 0);

  if (!csrf_ok($_POST['csrf'] ?? '')) {
    $err = 'رمز الأمان غير صالح. أعد المحاولة.';
  } elseif ($id <= 0) {
    $err = 'مُعرّف غير صالح.';
  } else {
    try {
      if ($action === 'toggle_active') {
        // قلب الحالة
        $cur = (int)$pdo->query("SELECT is_active FROM banners WHERE id=".((int)$id)." LIMIT 1")->fetchColumn();
        $new = $cur ? 0 : 1;
        $st  = $pdo->prepare("UPDATE banners SET is_active=?, updated_at=NOW() WHERE id=? LIMIT 1");
        $st->execute([$new,$id]);
        $msg = $new ? 'تم تفعيل البنر.' : 'تم تعطيل البنر.';
      } elseif ($action === 'delete_banner') {
        $pdo->prepare("DELETE FROM banners WHERE id=? LIMIT 1")->execute([$id]);
        $msg = 'تم حذف البنر.';
      } elseif ($action === 'update_sort') {
        $sort = (int)($_POST['sort_order'] ?? 0);
        $st   = $pdo->prepare("UPDATE banners SET sort_order=?, updated_at=NOW() WHERE id=? LIMIT 1");
        $st->execute([$sort, $id]);
        $msg = 'تم تحديث ترتيب العرض.';
      }
    } catch (Throwable $e) {
      $err = 'حدث خطأ أثناء تنفيذ العملية.';
    }
  }
}

/* -------------- Filters & Page ---------- */
$q        = trim($_GET['q'] ?? '');
$f_status = trim($_GET['status'] ?? ''); // '' | '1' | '0'
$page     = max(1, (int)($_GET['page'] ?? 1));
$perPage  = 50;
$offset   = ($page - 1) * $perPage;

$where  = "1=1"; $params = [];
if ($q !== '') {
  $where .= " AND (image_url LIKE ? OR page_url LIKE ? OR CAST(id AS CHAR) LIKE ?)";
  $kw = "%$q%"; array_push($params, $kw,$kw,$kw);
}
if ($f_status === '1' || $f_status === '0') {
  $where .= " AND is_active = ?";
  $params[] = (int)$f_status;
}

/* إجمالي وعدد الصفحات */
$st = $pdo->prepare("SELECT COUNT(*) FROM banners WHERE $where");
$st->execute($params);
$total = (int)$st->fetchColumn();
$pages = max(1, (int)ceil($total / $perPage));

/* جلب البيانات */
$sql = "SELECT id, image_url, page_url, is_active, sort_order, starts_at, ends_at, created_at
        FROM banners
        WHERE $where
        ORDER BY sort_order ASC, id DESC
        LIMIT $perPage OFFSET $offset";
$st = $pdo->prepare($sql); $st->execute($params);
$rows = $st->fetchAll();

/* ملخصات سريعة للحالة */
$counts = ['1'=>0,'0'=>0];
try {
  $counts = $pdo->query("SELECT is_active, COUNT(*) c FROM banners GROUP BY is_active")
                ->fetchAll(PDO::FETCH_KEY_PAIR) ?: $counts;
} catch(Throwable $e) { /* ignore */ }

/* -------------- Render ------------------ */
$PAGE_TITLE = "البنرات";
require __DIR__ . '/inc/header.php';
require __DIR__ . '/inc/sidebar.php';
?>
<!-- ====== تنسيق خاص بالصفحة ====== -->
<style>
.banners-page .table{
  width:100%;
  border-collapse:separate;
  border-spacing:0;
  background:var(--panel);
  border:1px solid var(--border);
  border-radius:12px;
  overflow:hidden;
  table-layout: fixed;
}
.banners-page .table td, .banners-page .table th{
  padding:10px 12px;
  border-bottom:1px solid var(--border);
  vertical-align:middle;
  text-align:start;
}
.banners-page .table thead th{
  background:#0b1220;color:var(--text);font-weight:700;
  position:sticky;top:0;z-index:3;
  box-shadow:inset 0 -1px 0 var(--border);
}
.banners-page .table tbody tr:nth-child(even) td{background:rgba(255,255,255,.015)}
.banners-page .table tbody tr:hover td{background:rgba(59,130,246,.08)}

.banners-page .td-center{text-align:center}
.banners-page .td-num{text-align:center;opacity:.9}
.banners-page .ellipsis{max-width:260px;display:inline-block;overflow:hidden;white-space:nowrap;text-overflow:ellipsis;vertical-align:bottom}
.banners-page .mono{font-family:monospace}

.banners-page .thumb{width:120px;height:60px;border-radius:8px;border:1px solid var(--border);object-fit:cover;background:#000}
.banners-page .actions{display:flex;gap:8px;flex-wrap:nowrap;align-items:center}
.banners-page .actions form{display:inline-block;margin:0}
.banners-page .btn.sm{padding:6px 10px;font-size:14px}

.banners-page .badge{display:inline-block;padding:2px 8px;border-radius:999px;border:1px solid var(--border);font-size:12px;opacity:.95}
.banners-page .badge.ok{border-color:#22c55e}
.banners-page .badge.muted{border-color:#64748b}

.banners-page .table-wrap{max-height:70vh;overflow:auto;position:relative}

/* أعمدة */
.banners-page .col-id{width:64px}
.banners-page .col-thumb{width:140px}
.banners-page .col-img{width:340px}
.banners-page .col-page{width:260px}
.banners-page .col-active{width:130px}
.banners-page .col-sort{width:120px}
.banners-page .col-dates{width:220px}
.banners-page .col-created{width:150px}
.banners-page .col-actions{width:320px}
</style>

<div class="banners-page">
  <!-- شريط علوي: فلاتر وإضافة -->
  <div class="card" style="margin-bottom:16px;">
    <form method="get" class="form" style="display:flex; gap:12px; align-items:flex-end; flex-wrap: wrap;">
      <div class="input" style="min-width:260px; flex:1;">
        <label>ابحث في رابط الصورة / الصفحة / رقم البنر</label>
        <input type="text" name="q" value="<?= e($q) ?>" placeholder="مثال: /voicechat/files/banners/ أو https://…">
      </div>

      <div class="input">
        <label>الحالة</label>
        <select name="status">
          <option value="">الكل</option>
          <option value="1" <?= $f_status==='1'?'selected':''; ?>>مفعّل</option>
          <option value="0" <?= $f_status==='0'?'selected':''; ?>>غير مفعّل</option>
        </select>
      </div>

      <button class="btn sm" type="submit">تطبيق</button>
      <?php if ($q!=='' || $f_status!==''): ?>
        <a class="btn sm secondary" href="banners.php">تفريغ الفلاتر</a>
      <?php endif; ?>

      <a class="btn sm" href="banners_add.php" style="margin-inline-start:auto;">➕ إضافة بانر</a>
    </form>

    <?php if ($msg): ?><div class="alert" style="margin-top:12px; border-color:#22c55e;"><?= e($msg) ?></div><?php endif; ?>
    <?php if ($err): ?><div class="alert" style="margin-top:12px;"><?= e($err) ?></div><?php endif; ?>

    <!-- ملخص سريع -->
    <div style="margin-top:10px; display:flex; gap:10px; flex-wrap:wrap; opacity:.95;">
      <div><?= active_badge(1) ?> <b><?= (int)($counts['1'] ?? 0) ?></b></div>
      <div><?= active_badge(0) ?> <b><?= (int)($counts['0'] ?? 0) ?></b></div>
      <div>الإجمالي: <b><?= $total ?></b></div>
    </div>
  </div>

  <!-- الجدول -->
  <div class="card">
    <div class="table-wrap">
      <table class="table">
        <colgroup>
          <col class="col-id"><col class="col-thumb"><col class="col-img"><col class="col-page">
          <col class="col-active"><col class="col-sort"><col class="col-dates"><col class="col-created"><col class="col-actions">
        </colgroup>
        <thead>
          <tr>
            <th class="td-center">#</th>
            <th>صورة</th>
            <th>رابط الصورة</th>
            <th>رابط الصفحة</th>
            <th>الحالة</th>
            <th>الترتيب</th>
            <th>المدة</th>
            <th>أُنشيء</th>
            <th class="col-actions">إجراءات</th>
          </tr>
        </thead>
        <tbody>
          <?php foreach ($rows as $r): ?>
            <tr>
              <td class="td-num"><?= (int)$r['id'] ?></td>

              <td class="td-center">
                <?php if (!empty($r['image_url'])): ?>
                  <a href="<?= e($r['image_url']) ?>" target="_blank" rel="noopener">
                    <img class="thumb" src="<?= e($r['image_url']) ?>" alt="banner #<?= (int)$r['id'] ?>">
                  </a>
                <?php else: ?>
                  <?= badge('لا توجد','muted') ?>
                <?php endif; ?>
              </td>

              <td title="<?= e((string)$r['image_url']) ?>">
                <span class="ellipsis mono"><?= e((string)$r['image_url']) ?></span>
              </td>

              <td title="<?= e((string)($r['page_url'] ?? '')) ?>">
                <?php if (!empty($r['page_url'])): ?>
                  <a class="ellipsis mono" href="<?= e((string)$r['page_url']) ?>" target="_blank" rel="noopener"><?= e((string)$r['page_url']) ?></a>
                <?php else: ?>
                  <span class="badge muted">—</span>
                <?php endif; ?>
              </td>

              <td><?= active_badge((int)$r['is_active']) ?></td>

              <td>
                <!-- تحديث سريع للترتيب -->
                <form method="post" class="actions" style="gap:6px;">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                  <input type="hidden" name="action" value="update_sort">
                  <input type="number" name="sort_order" value="<?= (int)$r['sort_order'] ?>" style="width:80px;padding:6px 8px;border:1px solid var(--border);border-radius:8px;background:transparent;color:var(--text)">
                  <button class="btn sm" type="submit">حفظ</button>
                </form>
              </td>

              <td>
                <div style="display:flex;flex-direction:column;gap:4px;">
                  <div><b>من:</b> <?= fmt_date($r['starts_at'] ?? null) ?></div>
                  <div><b>إلى:</b> <?= fmt_date($r['ends_at']   ?? null) ?></div>
                </div>
              </td>

              <td title="<?= e($r['created_at'] ?? '') ?>"><?= fmt_date($r['created_at'] ?? '') ?></td>

              <td class="actions">
                <!-- تحرير (يعاد استخدام صفحة الإضافة كمحرّر إن وُجد id) -->
                <a class="btn sm secondary" href="banners_add.php?id=<?= (int)$r['id'] ?>">تعديل</a>

                <!-- تفعيل/تعطيل -->
                <form method="post">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                  <input type="hidden" name="action" value="toggle_active">
                  <button class="btn sm" type="submit"><?= (int)$r['is_active'] ? 'تعطيل' : 'تفعيل' ?></button>
                </form>

                <!-- حذف -->
                <form method="post" onsubmit="return confirm('حذف البنر نهائيًا؟ لا يمكن التراجع.');">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                  <input type="hidden" name="action" value="delete_banner">
                  <button class="btn sm" type="submit" style="background:var(--danger);">حذف</button>
                </form>
              </td>
            </tr>
          <?php endforeach; ?>
          <?php if (!$rows): ?>
            <tr><td colspan="9" style="text-align:center; opacity:.8;">لا توجد بنرات</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>

    <!-- ترقيم الصفحات -->
    <?php if ($pages > 1): ?>
      <div style="display:flex; gap:8px; margin-top:12px; flex-wrap: wrap;">
        <?php
          $qs = [];
          if ($q!=='')        $qs[] = 'q='.urlencode($q);
          if ($f_status!=='') $qs[] = 'status='.urlencode($f_status);
          $base = 'banners.php'.($qs?('?'.implode('&',$qs)).'&':'?');
        ?>
        <?php for($p=1;$p<=$pages;$p++):
          $url = $base.'page='.$p;
          $is  = $p===$page; ?>
          <a href="<?= e($url) ?>" class="btn sm <?= $is ? '' : 'secondary' ?>"<?= $is?' style="pointer-events:none; opacity:.8;"':''; ?>><?= $p ?></a>
        <?php endfor; ?>
      </div>
    <?php endif; ?>
  </div>
</div>

<?php require __DIR__ . '/inc/footer.php'; ?>
