<?php
// backend/api/profile_get.php
declare(strict_types=1);
require __DIR__ . '/config.php';
require_once __DIR__ . '/vendor/autoload.php';

use Firebase\JWT\JWT;
use Firebase\JWT\Key;

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Authorization, Content-Type');
header('Access-Control-Allow-Methods: GET, OPTIONS');
if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }

function json_fail($e,$c=400){ http_response_code($c); echo json_encode(['ok'=>false,'error'=>$e], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES); exit; }
function json_ok($d=[]){ echo json_encode(['ok'=>true] + $d, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES); exit; }
function bearer(): ?string {
  $h = $_SERVER['HTTP_AUTHORIZATION'] ?? $_SERVER['Authorization'] ?? $_SERVER['REDIRECT_HTTP_AUTHORIZATION'] ?? '';
  if (!$h && function_exists('getallheaders')) {
    foreach (getallheaders() as $k => $v) {
      if (strcasecmp($k, 'Authorization') === 0) { $h = $v; break; }
    }
  }
  return stripos($h, 'Bearer ') === 0 ? trim(substr($h, 7)) : null;
}

/** فحص وجود عمود في جدول */
function table_has_col(PDO $pdo, string $table, string $col): bool {
  try {
    $st = $pdo->prepare("SHOW COLUMNS FROM `$table` LIKE ?");
    $st->execute([$col]);
    return (bool)$st->fetchColumn();
  } catch (Throwable $e) { return false; }
}

/**
 * تحويل رابط نسبي إلى مطلق + https
 * - إن كان الرابط يبدأ بـ http/https يُعاد كما هو مع إجبار https.
 * - إن كان نسبيًا، يُسبق بـ BASE_URL إن كانت معرفة في config.php وإلا يُستنتج من الطلب.
 */
function absolute_url(?string $raw): ?string {
  $s = trim((string)$raw);
  if ($s === '') return null;

  if (stripos($s, 'http://') === 0 || stripos($s, 'https://') === 0) {
    return preg_replace('#^http://#i', 'https://', $s);
  }

  $base = defined('BASE_URL') ? rtrim((string)BASE_URL, '/') : null;
  if (!$base) {
    $proto = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    if (!empty($_SERVER['HTTP_X_FORWARDED_PROTO'])) $proto = $_SERVER['HTTP_X_FORWARDED_PROTO'];
    $host = $_SERVER['HTTP_X_FORWARDED_HOST'] ?? ($_SERVER['HTTP_HOST'] ?? 'localhost');
    $base = $proto . '://' . $host;
  }
  if ($s[0] !== '/') $s = '/' . $s;
  return $base . $s;
}

$jwt = bearer(); if (!$jwt) json_fail('missing authorization bearer token', 401);

try {
  $payload = JWT::decode($jwt, new Key($JWT_SECRET, 'HS256'));
} catch (Throwable $e) {
  json_fail('invalid token: '.$e->getMessage(), 401);
}

// من التوكن (ادعم أكثر من مفتاح عددي إن أردت)
$dbId        = (int)($payload->uid ?? $payload->user_id ?? $payload->id ?? 0); // users.id العددي
$firebaseUid = (string)($payload->firebase_uid ?? '');                         // Firebase UID النصّي

$user = [
  'id'            => null,
  'uid'           => null,
  'userid'        => null,
  'username'      => null,
  'avatar_url'    => null,
  'cover_url'     => null,  // ✅ جديد
  'gender'        => null,
  'coins'         => 0,     // الرصيد
  'entry'         => null,  // رابط فيديو الدخوليّة (Full Screen) يُرفعه الأدمن
  'bio'           => null,  // نبذة المستخدم
  'frame_pag_url' => null,  // إطار المستخدم حول الصورة
];

try {
  $pdo = db();

  // نبني قائمة الأعمدة ديناميكيًا لتفادي أخطاء الأعمدة الناقصة
  $cols = "id, uid, userid, username, gender, COALESCE(coins, 0) AS coins, bio";

  // avatar_url (دعم بدائل شائعة)
  if (table_has_col($pdo, 'users', 'avatar_url')) {
    $cols .= ", avatar_url";
  } elseif (table_has_col($pdo, 'users', 'avatar')) {
    $cols .= ", avatar AS avatar_url";
  } elseif (table_has_col($pdo, 'users', 'image_url')) {
    $cols .= ", image_url AS avatar_url";
  }

  // cover_url (دعم بدائل)
  if (table_has_col($pdo, 'users', 'cover_url')) {
    $cols .= ", cover_url";
  } elseif (table_has_col($pdo, 'users', 'cover')) {
    $cols .= ", cover AS cover_url";
  }

  // entry (لو مخزّن كرابط)
  if (table_has_col($pdo, 'users', 'entry')) {
    $cols .= ", entry";
  } elseif (table_has_col($pdo, 'users', 'entry_url')) {
    $cols .= ", entry_url AS entry";
  }

  // frame_pag_url (بدائل متعددة)
  if (table_has_col($pdo, 'users', 'frame_pag_url')) {
    $cols .= ", frame_pag_url";
  } elseif (table_has_col($pdo, 'users', 'frame_url')) {
    $cols .= ", frame_url AS frame_pag_url";
  } elseif (table_has_col($pdo, 'users', 'frame')) {
    $cols .= ", frame AS frame_pag_url";
  } elseif (table_has_col($pdo, 'users', 'ring_url')) {
    $cols .= ", ring_url AS frame_pag_url";
  }

  if ($dbId > 0) {
    $stmt = $pdo->prepare("SELECT $cols FROM users WHERE id = ? LIMIT 1");
    $stmt->execute([$dbId]);
  } elseif ($firebaseUid !== '') {
    $stmt = $pdo->prepare("SELECT $cols FROM users WHERE uid = ? LIMIT 1");
    $stmt->execute([$firebaseUid]);
  } else {
    json_fail('no user identifier in token', 401);
  }

  if ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
    $user['id']         = isset($row['id']) ? (int)$row['id'] : null;
    $user['uid']        = isset($row['uid']) ? (string)$row['uid'] : null;       // Firebase UID
    $user['userid']     = isset($row['userid']) ? (string)$row['userid'] : null; // رقم المستخدم
    $user['username']   = $row['username']   ?? null;
    $user['gender']     = $row['gender']     ?? null;
    $user['coins']      = (int)($row['coins'] ?? 0);

    // روابط مطلقة/آمنة
    $user['avatar_url'] = absolute_url($row['avatar_url'] ?? null);
    $user['cover_url']  = absolute_url($row['cover_url']  ?? null);  // ✅ تغذية الكفر
    $user['entry']      = absolute_url($row['entry']      ?? null);

    // البايو
    $user['bio']        = isset($row['bio']) ? (string)$row['bio'] : null;

    // إطار الصورة
    $user['frame_pag_url'] = absolute_url($row['frame_pag_url'] ?? null);
  } else {
    json_fail('user_not_found', 404);
  }

} catch (Throwable $e) {
  json_fail('db_error: '.$e->getMessage(), 500);
}

json_ok(['user' => $user]);
