<?php
// voicechat/api/gifts_send.php
declare(strict_types=1);
require_once __DIR__ . '/config.php'; // db(), json_in(), json_out(), require_auth()

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Authorization, Content-Type');
header('Access-Control-Allow-Methods: POST, OPTIONS');
if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }

$pdo = db();

/** ردود مختصرة */
function fail($msg, int $code = 400){
  http_response_code($code);
  echo json_encode(['ok'=>false,'error'=>$msg], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
  exit;
}
function ok(array $data){
  echo json_encode(['ok'=>true] + $data, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
  exit;
}

/** يحوِّل أي قيمة (id/uid/userid) إلى users.id */
function resolve_user_id(PDO $pdo, $value): ?int {
  if ($value === null) return null;
  if (is_int($value)) return $value > 0 ? $value : null;

  $s = trim((string)$value);
  if ($s === '') return null;

  if (ctype_digit($s)) {
    $id = (int)$s;
    if ($id > 0) {
      $st = $pdo->prepare('SELECT id FROM users WHERE id=? LIMIT 1');
      $st->execute([$id]);
      if ($st->fetchColumn()) return $id;
    }
    $st = $pdo->prepare('SELECT id FROM users WHERE userid=? LIMIT 1');
    $st->execute([$s]);
    $id = (int)$st->fetchColumn();
    return $id > 0 ? $id : null;
  }

  $st = $pdo->prepare('SELECT id FROM users WHERE uid=? LIMIT 1');
  $st->execute([$s]);
  $id = (int)$st->fetchColumn();
  return $id > 0 ? $id : null;
}

try {
  // المصادقة (ينبغي أن تدعم require_auth() القراءة من Authorization أو من توكن بالكوكي/POST حسب إعدادك)
  $claims   = require_auth();
  $senderId = resolve_user_id($pdo, $claims['id'] ?? ($claims['uid'] ?? null));
  if (!$senderId) fail('sender_not_found', 404);

  // === (1) قراءة المُدخلات: ادعم JSON أو form-POST + مفاتيح احتياطية ===
  $in = json_in();
  if (!$in) {
    // x-www-form-urlencoded
    $in = $_POST ?: $_GET;
  }

  // أسماء مقبولة
  $roomUid     = trim((string)($in['room_uid'] ?? $in['roomId'] ?? ''));
  $giftId      = (int)($in['gift_id'] ?? $in['gid'] ?? 0);
  $qty         = (int)($in['quantity'] ?? $in['qty'] ?? 1);
  $receiverRaw = $in['receiver_id']
              ?? $in['receiver_uid']
              ?? $in['receiver_userid']
              ?? $in['to_uid']
              ?? null;

  if ($roomUid === '' || $giftId <= 0) fail('missing_parameters', 422);
  $qty = max(1, min(999, $qty)); // حماية

  $receiverId = resolve_user_id($pdo, $receiverRaw);
  if ($receiverRaw !== null && !$receiverId) fail('receiver_not_found', 404);

  // === (2) اجلب بيانات الهدية مع الحقول الجديدة + اشترط أن تكون مفعلة ===
  $st = $pdo->prepare('
    SELECT id,name,price,image_url,video_url,thumb_url,fullscreen,video_crop_right
    FROM gift_items
    WHERE id=? AND is_active=1
    LIMIT 1
  ');
  $st->execute([$giftId]);
  $gift = $st->fetch(PDO::FETCH_ASSOC);
  if (!$gift) fail('gift_not_found', 404);

  $unitPrice = (int)($gift['price'] ?? 0);
  $total     = $unitPrice * $qty;

  // معاملة ذرّية
  $pdo->beginTransaction();

  // اقفل رصيد المُرسل
  $st = $pdo->prepare('SELECT id, coins FROM users WHERE id=? FOR UPDATE');
  $st->execute([$senderId]);
  $row = $st->fetch(PDO::FETCH_ASSOC);
  if (!$row) { $pdo->rollBack(); fail('sender_not_found', 404); }

  $balance = (int)$row['coins'];
  if ($balance < $total) { $pdo->rollBack(); fail('insufficient_coins', 402); }

  // خصم
  $upd = $pdo->prepare('UPDATE users SET coins = coins - ? WHERE id = ?');
  $upd->execute([$total, $senderId]);

  // سجل العملية
  $ins = $pdo->prepare('
    INSERT INTO gift_transactions
      (gift_id, sender_user_id, receiver_user_id, room_uid, quantity, unit_price, total_price, meta)
    VALUES (?,?,?,?,?,?,?,?)
  ');
  $ins->execute([
    $giftId, $senderId, $receiverId, $roomUid, $qty, $unitPrice, $total,
    json_encode(['src'=>'api','v'=>2], JSON_UNESCAPED_UNICODE),
  ]);
  $txId = (int)$pdo->lastInsertId();

  $pdo->commit();

  // (اختياري) ارجع بيانات الهدية ليستطيع العميل إظهار الأوفرلاي فورًا
  ok([
    'tx_id'       => $txId,
    'room_uid'    => $roomUid,
    'quantity'    => $qty,
    'unit_price'  => $unitPrice,
    'total'       => $total,
    'balance'     => $balance - $total,
    'gift' => [
      'id'              => (int)$gift['id'],
      'name'            => (string)$gift['name'],
      'image_url'       => (string)($gift['image_url'] ?? ''),
      'thumb_url'       => (string)($gift['thumb_url'] ?? ''),
      'video_url'       => (string)($gift['video_url'] ?? ''),
      'fullscreen'      => (int)($gift['fullscreen'] ?? 0),
      'video_crop_right'=> isset($gift['video_crop_right']) ? (float)$gift['video_crop_right'] : null,
    ],
    'receiver_user_id' => $receiverId, // قد تكون null للبث العام
  ]);

} catch (Throwable $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  // error_log('gifts_send error: '.$e->getMessage());
  fail('server_error', 500);
}
