<?php
// voicechat/api/gifts.php
declare(strict_types=1);

require_once __DIR__ . '/config.php';   // يحتوي db() و json_out()

/**
 * فحص وجود عمود في جدول (للتوافق مع قواعد بيانات قد تكون قديمة)
 */
function col_exists(PDO $pdo, string $table, string $col): bool {
  try {
    $st = $pdo->prepare("SHOW COLUMNS FROM `$table` LIKE ?");
    $st->execute([$col]);
    return (bool)$st->fetchColumn();
  } catch (Throwable $e) {
    return false;
  }
}

try {
  $pdo = db();

  // ====== قراءة المعاملات ======
  $categoryId = (int)($_GET['category_id'] ?? 0);
  if ($categoryId <= 0) {
    json_out(['ok' => false, 'error' => 'missing_category_id'], 400);
  }

  $page     = max(1, (int)($_GET['page'] ?? 1));
  $pageSize = (int)($_GET['page_size'] ?? 20);
  if ($pageSize < 1)  $pageSize = 20;
  if ($pageSize > 50) $pageSize = 50;
  $offset = ($page - 1) * $pageSize;

  // أعمدة اختيارية (قد لا تكون موجودة في بعض الجداول القديمة)
  $hasVideo      = col_exists($pdo, 'gift_items', 'video_url');
  $hasThumb      = col_exists($pdo, 'gift_items', 'thumb_url');
  $hasFullscreen = col_exists($pdo, 'gift_items', 'fullscreen');
  $hasCropRight  = col_exists($pdo, 'gift_items', 'video_crop_right');

  // نبني SELECT ديناميكياً حسب الأعمدة المتاحة
  $select = "
    SELECT
      id, category_id, name,
      image_url, price, sort_order,
      " . ($hasVideo      ? "video_url,"       : "NULL AS video_url,") . "
      " . ($hasThumb      ? "thumb_url,"       : "NULL AS thumb_url,") . "
      " . ($hasFullscreen ? "fullscreen,"      : "0    AS fullscreen,") . "
      " . ($hasCropRight  ? "video_crop_right" : "NULL AS video_crop_right") . "
    FROM gift_items
    WHERE category_id = :cid AND (is_active = 1 OR is_active IS NULL)
    ORDER BY sort_order ASC, id ASC
    LIMIT :lim OFFSET :off
  ";

  $stmt = $pdo->prepare($select);
  $stmt->bindValue(':cid', $categoryId, PDO::PARAM_INT);
  $stmt->bindValue(':lim', $pageSize, PDO::PARAM_INT);
  $stmt->bindValue(':off', $offset, PDO::PARAM_INT);
  $stmt->execute();
  $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

  // إجمالي العناصر لحساب has_more
  $stmt2 = $pdo->prepare("
    SELECT COUNT(*) FROM gift_items
    WHERE category_id = :cid AND (is_active = 1 OR is_active IS NULL)
  ");
  $stmt2->execute([':cid' => $categoryId]);
  $total = (int)$stmt2->fetchColumn();

  // تجهيز العناصر للإخراج
  $items = array_map(function(array $r): array {
    // تطبيع القيم وأنواعها
    $id          = (int)($r['id'] ?? 0);
    $category_id = (int)($r['category_id'] ?? 0);
    $name        = (string)($r['name'] ?? '');
    $image_url   = (string)($r['image_url'] ?? '');
    $price       = (int)  ($r['price'] ?? 0);
    $sort_order  = (int)  ($r['sort_order'] ?? 0);
    $video_url   = isset($r['video_url']) ? (string)$r['video_url'] : '';
    $thumb_url   = isset($r['thumb_url']) ? (string)$r['thumb_url'] : '';
    $fullscreen  = (int)  ($r['fullscreen'] ?? 0);
    $crop_right  = isset($r['video_crop_right']) ? (float)$r['video_crop_right'] : null;

    // تحسّب: إن لم يوجد video_url وكان image_url نفسه mp4
    // سنعيده أيضًا في video_url حتى يفهمه العميل الحديث.
    if ($video_url === '' && preg_match('~\.mp4(\?.*)?$~i', $image_url)) {
      $video_url = $image_url;
    }

    return [
      'id'               => $id,
      'category_id'      => $category_id,
      'name'             => $name,
      'image_url'        => $image_url,
      'thumb_url'        => $thumb_url,       // قد يكون فارغًا إن لم يوجد عمود
      'video_url'        => $video_url,       // قد يكون فارغًا إن لم يوجد عمود أو لا يوجد فيديو
      'fullscreen'       => $fullscreen,      // 0/1
      'video_crop_right' => $crop_right,      // قد يكون null
      'price'            => $price,
      'sort_order'       => $sort_order,
    ];
  }, $rows);

  $hasMore = ($offset + count($items)) < $total;

  json_out([
    'ok'        => true,
    'data'      => $items,     // يقرأها التطبيق لديك بالفعل
    'page'      => $page,
    'page_size' => $pageSize,
    'has_more'  => $hasMore,
    'total'     => $total,
  ]);

} catch (Throwable $e) {
  json_out(['ok' => false, 'error' => $e->getMessage()], 500);
}
