<?php
declare(strict_types=1);
require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../inc/auth.php'; // مهم لمعرفة المستخدم
header('Content-Type: application/json; charset=utf-8');

// === مصادقة ===
// عدل حسب نظامك: هنا أفترض أن user_id متاح عبر جلسة/توكن
$uid = $_SESSION['uid'] ?? 0; // أو اجلبه من دالة require_login_api()
if ($uid <= 0) {
  http_response_code(401);
  echo json_encode(['ok'=>false, 'message'=>'غير مصرح.']);
  exit;
}

// === مُدخلات ===
$sku           = trim($_POST['sku'] ?? '');
$purchaseToken = trim($_POST['purchaseToken'] ?? '');
$coinsExpected = isset($_POST['coinsExpected']) ? (int)$_POST['coinsExpected'] : null;

if ($sku === '' || $purchaseToken === '' || strlen($purchaseToken) < 10) {
  http_response_code(400);
  echo json_encode(['ok'=>false, 'message'=>'بيانات غير صالحة.']);
  exit;
}

$pdo = db();

try {
  $pdo->beginTransaction();

  // 1) افحص التكرار/idempotency
  //    يفضل إضافة فهرس UNIQUE على purchase_token لمنع التكرار.
  $st = $pdo->prepare("SELECT id, user_id, status FROM billing_google_receipts WHERE purchase_token=? LIMIT 1 FOR UPDATE");
  $st->execute([$purchaseToken]);
  $existing = $st->fetch(PDO::FETCH_ASSOC);

  if ($existing) {
    if ($existing['status'] === 'credited') {
      // تم شحنها مسبقاً
      $stU = $pdo->prepare("SELECT coins FROM users WHERE id=?");
      $stU->execute([$existing['user_id']]);
      $bal = (int)$stU->fetchColumn();
      $pdo->commit();
      echo json_encode(['ok'=>true, 'newBalance'=>$bal, 'alreadyCredited'=>true]);
      exit;
    } else {
      // موجودة ولكن ليست credited — تعامل حسب حالتك
      $pdo->rollBack();
      echo json_encode(['ok'=>false, 'message'=>'إيصال قيد المعالجة/مرفوض.']);
      exit;
    }
  }

  // 2) إحضار الباقة والتحقق من تفعيلها
  $st = $pdo->prepare("SELECT sku, coins, enabled FROM billing_google_products WHERE sku=? LIMIT 1");
  $st->execute([$sku]);
  $product = $st->fetch(PDO::FETCH_ASSOC);
  if (!$product || !(int)$product['enabled']) {
    $pdo->rollBack();
    echo json_encode(['ok'=>false, 'message'=>'الباقة غير موجودة أو معطّلة.']);
    exit;
  }

  $coinsToAdd = (int)$product['coins'];
  if ($coinsExpected !== null && $coinsExpected !== $coinsToAdd) {
    // حماية بسيطة ضد العبث
    $pdo->rollBack();
    echo json_encode(['ok'=>false, 'message'=>'عدم تطابق قيمة الكوين.']);
    exit;
  }

  // 3) (اختياري مُستحسن) تحقق مع Google Play عبر API الرسمي:
  //    لو عندك Service Account، فعّل التحقق الحقيقي هنا ثم اضبط $verified و $orderId.
  $verified = true;     // ← غيّرها عند ربط Google API
  $orderId  = null;     // عيّن orderId الحقيقي إذا توفر من Google

  if (!$verified) {
    $pdo->rollBack();
    echo json_encode(['ok'=>false, 'message'=>'تعذّر التحقق من Google Play.']);
    exit;
  }

  // 4) حدّث رصيد المستخدم + سجّل الإيصال (ذو حالة credited)
  //    اقفل صف المستخدم لضمان الذرّة
  $st = $pdo->prepare("SELECT coins FROM users WHERE id=? FOR UPDATE");
  $st->execute([$uid]);
  $current = (int)($st->fetchColumn() ?? 0);
  $newBal  = $current + $coinsToAdd;

  $st = $pdo->prepare("UPDATE users SET coins=? WHERE id=?");
  $st->execute([$newBal, $uid]);

  $st = $pdo->prepare("
    INSERT INTO billing_google_receipts (user_id, sku, purchase_token, order_id, coins, status, created_at)
    VALUES (?, ?, ?, ?, ?, 'credited', NOW())
  ");
  $st->execute([$uid, $sku, $purchaseToken, $orderId, $coinsToAdd]);
  $receiptId = (int)$pdo->lastInsertId();

  // (اختياري) سجل حركة بمحفظة/معاملاتك لو لديك جدول معاملات عام
  // $pdo->prepare("INSERT INTO coin_transactions (user_id, amount, type, ref) VALUES (?,?, 'google_play', ?)")->execute([$uid, $coinsToAdd, $receiptId]);

  $pdo->commit();

  echo json_encode(['ok'=>true, 'newBalance'=>$newBal, 'receiptId'=>$receiptId]);
} catch (Throwable $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  http_response_code(500);
  echo json_encode(['ok'=>false, 'message'=>'خطأ داخلي أثناء الشحن.']);
}
